/**
  **************************************************************************************
  * @file    main.c
  * @brief   main C File
  *
  * @version V1.00
  * @date    13/11/2018
  * @author  Eastsoft AE Team
  * @note
  *
  * Copyright (C) 2018 Shanghai Eastsoft Microelectronics Co., Ltd. ALL rights reserved.
  *
  **************************************************************************************
  */

/* Define to prevent recursive inclusion -------------------------------------*/
#define MAIN_GLOBALS

/* Includes -------------------------------------------------------------------*/
#include "es32f0283.h"
#include "md_csu.h"
#include "md_rcu.h"
#include <stdio.h>

/** @addtogroup Projects_Examples_MD
  * @{
  */

/* Private constants ----------------------------------------------------------*/
/* Private types --------------------------------------------------------------*/
/* Private variables ----------------------------------------------------------*/
/* Private macros -------------------------------------------------------------*/
#define FREQ48M         (480000000UL)               // 48MHz
#define RELOAD(x)       ((x / FSYNC) - 1)           // (Ftarget/Fsync)-1
#define CNTTH(x)        (x / FSYNC / 5 / 100 / 2)   // (Ftarget/Fsync)*SETP[%]/100%/2
#define COUNT_THRESHOLD CNTTH(FREQ48M)
#define SYNCCAP_OFFSET  (0x80000000UL)
#define SYNC_COUNT      (128)
#define KEEP_STABLE     (3)

static  uint8_t     SyncLoop = 0;
static  uint8_t     KeepCount = 0;
static  uint32_t    SyncCap = SYNCCAP_OFFSET;
uint8_t     SyncCount = SYNC_COUNT;
/* Private function prototypes ------------------------------------------------*/

/* Public functions -----------------------------------------------------------*/
/** @addtogroup Public_Function
  * @{
  */

/**
  * @brief  RCU_CSU_IRQHandler.
  * @param  None
  * @retval None
  */
void    RCU_CSU_IRQHandler(void)
{
    // Clear Interrupt
    md_csu_set_icr(CSU, md_csu_get_ifm(CSU));

    SyncCount--;

    if (md_csu_get_trim_counter_direction(CSU))
        // Frequency is slower than 48MHz
        SyncCap += md_csu_get_trim_counter_capture(CSU);
    else
        // Frequency is faster than 48MHz
        SyncCap -= md_csu_get_trim_counter_capture(CSU);
}

void    sw_csu_init()
{
    // Enable CSU Interrupt
    md_csu_set_ier(CSU, (CSU_IER_FWARN | CSU_IER_FMATCH));  //Enable FWARN, FMATCH

    // CSU Setting
    md_csu_set_trim_source(CSU, MD_CSU_CFG_TRIMSEL_OPTIONBYTE); // TRIM is Option Byte
    md_csu_set_sync_polarity(CSU, MD_CSU_CFG_POLSEL_RISING);    //Posedge = Rising
    md_csu_set_sync_source(CSU, MD_CSU_CFG_SYNCSRC_USB);    // SOF Package
    md_csu_set_sync_source_div(CSU, MD_CSU_CFG_SYNCDIV_DIV1);   // No Divider
    md_csu_set_trim_counter_tolerance(CSU, CNTTH(48000000));
    md_csu_set_trim_counter_reload(CSU, RELOAD(48000000));
    md_csu_enable_trim_counter(CSU);
    md_csu_disable_trim_auto(CSU);
}

void    sw_csu_sync()
{
    printf("\r\nLoop = %d\r\n", SyncLoop);
    printf("ThresHold = 0x%x\r\n", COUNT_THRESHOLD);
    SyncLoop++;
    md_csu_disable_trim_counter(CSU);
    SyncCount = SYNC_COUNT;

    if (SyncCap >= SYNCCAP_OFFSET)
    {
        SyncCap -= SYNCCAP_OFFSET;
        SyncCap = (SyncCap / SYNC_COUNT);
        printf("SyncCap = Over 0x%x - ", SyncCap);

        if (SyncCap > COUNT_THRESHOLD)
        {
            md_rcu_set_hrc48trim_source(RCU, MD_RCU_HRC48SEL_HRC48TRIM);
            md_rcu_set_hrc48trim(RCU, (md_rcu_get_hrc48trim(RCU) + 1));
            md_rcu_trigger_hrc48trim_update(RCU);
            KeepCount = 0;
            printf("Plus\r\n");
        }
        else
        {
            KeepCount++;
            printf("Keep\r\n");
        }
    }
    else
    {
        SyncCap = (SYNCCAP_OFFSET - SyncCap);
        SyncCap = (SyncCap / SYNC_COUNT);
        printf("SyncCap = Under 0x%x - ", SyncCap);

        if (SyncCap > COUNT_THRESHOLD)
        {
            md_rcu_set_hrc48trim_source(RCU, MD_RCU_HRC48SEL_HRC48TRIM);
            md_rcu_set_hrc48trim(RCU, (md_rcu_get_hrc48trim(RCU) - 1));
            md_rcu_trigger_hrc48trim_update(RCU);
            KeepCount = 0;
            printf("Minus\r\n");
        }
        else
        {
            KeepCount++;
            printf("Keep\r\n");
        }
    }

    printf("Trim=0x%x\r\n", md_rcu_get_hrc48trim(RCU));
    printf("Keep=%d\r\n", KeepCount);

    if (KeepCount < KEEP_STABLE)
    {
        SyncCap = SYNCCAP_OFFSET;
        md_csu_enable_trim_counter(CSU);
    }
}

/**
  * @} Public_Function
  */

/**
  * @} Projects_Examples_MD
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/


