/**
  **************************************************************************************
  * @file    main.c
  * @brief   main C File
  *
  * @version V1.00
  * @date    13/11/2018
  * @author  Eastsoft AE Team
  * @note
  *
  * Copyright (C) 2018 Shanghai Eastsoft Microelectronics Co., Ltd. ALL rights reserved.
  *
  **************************************************************************************
  */

/* Define to prevent recursive inclusion -------------------------------------*/
#define MAIN_GLOBALS

/* Includes -------------------------------------------------------------------*/
#include "main.h"
#include "usb_core.h"
#include "usb_enum.h"
#include "usb_dev_enum.h"
#include "app_dev_hidenum.h"
#include "sw_csu_sync.h"
#include <stdio.h>

/** @addtogroup Projects_Examples_MD
  * @{
  */

/* Private constants ----------------------------------------------------------*/
/* Private types --------------------------------------------------------------*/
/** @defgroup RCU_Init_Struct RCU Init Structure
  * @{
  */
md_rcu_init_typedef rcu_initStruct =    /**< RCU init structure */
{
    MD_RCU_MPRE_MCO_DIV1,
    MD_RCU_MSW_MCO_DISABLE,
    MD_RCU_PLLSRC_HRC48,
    MD_RCU_PLLCLK_PASS,
    MD_RCU_PPRE_HCLK_DIV_1,
    MD_RCU_HPRE_SYSCLK_DIV_1,
    MD_RCU_SW_SYSCLK_HRC48,
    (RCU_CON_HRCON_MSK | RCU_CON_PLL0ON_MSK | RCU_CON_HRC48ON_MSK),
};

/** @defgroup UART_Init_Struct UART Init Structure
  * @{
  */
md_uart_init_typedef uart_initStruct =    /**< UART init structure */
{
    MD_UART_BAUDRATE_115200,
    MD_UART_LCON_LSB_FIRST,
    MD_UART_LCON_PS_NONE,
    MD_UART_LCON_STOP_1,
    MD_UART_LCON_DLS_8,
};

_USB_FS_DEVICE  HID_FS_Dev;

/* Private variables ----------------------------------------------------------*/
/* Private macros -------------------------------------------------------------*/
/* Private function prototypes ------------------------------------------------*/

/* Public functions -----------------------------------------------------------*/
/** @addtogroup Public_Function
  * @{
  */

/**
  * @brief  IrqInit
  * @param  None
  * @retval None
  */
void IrqInit(void)
{
    NVIC->ICER[0] = 0xFFFFFFFFUL;
    NVIC_EnableIRQ((IRQn_Type)USB_IRQn);        //USB IRQ
    NVIC_EnableIRQ((IRQn_Type)RCU_CSU_IRQn);    // RCU_CSU IRQHandler
}

/**
  * @brief  set irq priority
  * @param  None
  * @retval None
  */
void IrqPriority(void)
{
}

/**
  * @brief  configure i/o multiplexer
  * @param  None
  * @retval None
  */
void Iomux(void)
{
    /*  UART IOMux  */
    md_gpio_set_pin_mode(GPIOB, MD_GPIO_PIN_6,   MD_GPIO_MODE_FUNCTION);
    md_gpio_set_pin_mode(GPIOB, MD_GPIO_PIN_7,   MD_GPIO_MODE_FUNCTION);
    md_gpio_set_pin_pull(GPIOB, MD_GPIO_PIN_7,   MD_GPIO_PULL_UP);
    md_gpio_set_pin_function0_7(GPIOB, MD_GPIO_PIN_6,   MD_GPIO_AF2);
    md_gpio_set_pin_function0_7(GPIOB, MD_GPIO_PIN_7,   MD_GPIO_AF2);
}

/**
  * @brief  System Peri init
  * @param  None
  * @retval None
  */
void SysPeriInit(void)
{
    /*  AHB Clock Enable  */
    md_rcu_enable_gpiod(RCU);
    md_rcu_enable_gpioc(RCU);
    md_rcu_enable_gpiob(RCU);
    md_rcu_enable_gpioa(RCU);
    md_rcu_enable_usb(RCU);
    md_rcu_enable_csu(RCU);

    /*  APB Clock Enable  */
    md_rcu_enable_uart1(RCU);
}

/**
  * @brief  CoreInit.
  * @param  None
  * @retval None
  */
void  CoreInit()
{
    md_rcu_pll0_init(RCU, &rcu_initStruct);
    md_rcu_sys_init(RCU, &rcu_initStruct);

    SysPeriInit();
    IrqInit();
    Iomux();

    md_uart_init(UART1, &uart_initStruct);
    md_tick_init(SystemFrequency_AHBClk);

}

/**
  * @brief  Uart sendchar.
  * @param  arg: char to be sent.
  * @retval data to be sent.
  */
uint8_t  sendchar(uint8_t ch)
{
    while (!md_uart_is_active_flag_tfempty(UART1));  /* Tx FIFO full */

    md_uart_set_send_data8(UART1, ch);          // Send byte
    return (ch);
}

/**
  * @brief  main.
  * @param  None
  * @retval None
  */
int main(void)
{
    eUSBDState  u8USBState;

    CoreInit();

    printf("USBD HID Enumeration\r\n");
    printf("SysFosc=%dHz\r\n", SystemCoreClock);

    sw_csu_init();
    hidenum_dev_init(&HID_FS_Dev);
    usb_core_mode_set(eUSBModeForceDevice);

    u8USBState = USBD_STATE_IDLE;

    while (1)
    {
        u8USBState = hidenum_dev_enum(&HID_FS_Dev, u8USBState);

        if (u8USBState == USBD_STATE_EP1OUT)        //hid_dev_enumeration Int Out
            u8USBState = hidenum_dev_transaction(&HID_FS_Dev, u8USBState);

        if (SyncCount == 0)     //Sync Done
            sw_csu_sync();

    }

    while (1);
}

/**
  * @} Public_Function
  */

/**
  * @} Projects_Examples_MD
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/


