/**********************************************************************************
 *
 * @file    app_dev_hidenum.c
 * @brief   c File
 *
 * @date    5 May 2022
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          5 May 2022      AE Team         the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */
#include "es32f0283.h"
#include "md_usb.h"
#include "md_rcu.h"
#include "usb_core.h"
#include "usb_enum.h"
#include "usb_dev_enum.h"
#include "usbhid.h"
#include "usb_dev_hid.h"
#include "app_hidenum_desc.h"
#include "app_dev_hidenum.h"
#include "cb_dev_hidenum.h"
#include <stdio.h>

#pragma pack(1)
#define APP_HIDENUM_MSG 1

extern md_rcu_init_typedef rcu_initStruct;

static  _pUSB_SETUP_REQUEST pSetupReq = (_pUSB_SETUP_REQUEST) &SetupReq;
static  uint8_t     ENUMDataBuf[HID_INT_PKTSZ];

/****
    * @brief    USB Application HID Device Endpoint 0 Configuration
    * @param    pDev    : Device Instance
    * @retval   None
****/
void    hidenum_dev_ep0_config(_pUSB_FS_DEVICE pDev)
{
//Configure EP0
    md_usb_set_epx_index(USB, USBEP0);          //Endpoint 0
    md_usb_set_tx_max_packet_size(USB, (USBD_CTRL_PKTSZ / 8));  //64 bytes
    md_usb_set_tx_fifo_start_addr(USB, 0x0);        //Start Address=0x0
    md_usb_set_tx_max_fifo_packet_size(USB, MD_USB_TXFIFO2_MAXPKTSIZE_64);  //Size=64
    md_usb_set_rx_max_packet_size(USB, (USBD_CTRL_PKTSZ / 8));  //64 bytes
    md_usb_set_rx_fifo_start_addr(USB, 0x0);        //Start Address=0x0
    md_usb_set_rx_max_fifo_packet_size(USB, MD_USB_RXFIFO2_MAXPKTSIZE_64);  //Size=64

    md_usb_enable_it_ep0(USB);      //Interrupt Enable for EP0
    md_usb_enable_it_suspend(USB);
}
/****
    * @brief    USB Application HID Device Endpoint Configuration
    * @param    pDev    : Device Instance
    * @retval   None
****/
void    hidenum_dev_ep_config(_pUSB_FS_DEVICE pDev)
{
    uint8_t     EPFIFOIdx;

    EPFIFOIdx = 0x8;
//Configure EP1 In
    md_usb_set_epx_index(USB, USBEP1);          //Endpoint 1
    md_usb_disable_epx_tx_isochronous(USB);     //Non ISO
    md_usb_set_epx_mode(USB, MD_USB_TXCSRH_MODE_TX);        //TX
    md_usb_set_tx_max_packet_size(USB, (HID_INT_PKTSZ / 8));    //16 bytes
    md_usb_set_tx_fifo_start_addr(USB, EPFIFOIdx);      //Start Address=0x8
    md_usb_set_tx_max_fifo_packet_size(USB, MD_USB_TXFIFO2_MAXPKTSIZE_16);  //Size=16, No Double-Packet Buffering
    md_usb_trigger_epx_tx_flush_fifo(USB);          //Flush FIFO

    EPFIFOIdx += ((HID_INT_PKTSZ + 7) / 8);

    if (md_usb_is_enabled_tx_double_packet(USB))    //Double-Packet Buffering
        EPFIFOIdx += ((HID_INT_PKTSZ + 7) / 8);

//Configure EP1 Out
    md_usb_set_epx_index(USB, USBEP1);          //Endpoint 1
    md_usb_disable_epx_rx_isochronous(USB);         //Non ISO
    md_usb_set_epx_mode(USB, MD_USB_TXCSRH_MODE_RX);        //RX
    md_usb_set_rx_max_packet_size(USB, (HID_INT_PKTSZ / 8));    //16 Bytes
    md_usb_set_rx_fifo_start_addr(USB, EPFIFOIdx);  //Start Address=0xa
    md_usb_set_rx_max_fifo_packet_size(USB, MD_USB_RXFIFO2_MAXPKTSIZE_16);  //Size=16, No Double-Packet Buffering
    md_usb_trigger_epx_rx_flush_fifo(USB);          //Flush FIFO

    EPFIFOIdx += ((HID_INT_PKTSZ + 7) / 8);

    if (md_usb_is_enabled_rx_double_packet(USB))    //Double-Packet Buffering
        EPFIFOIdx += ((HID_INT_PKTSZ + 7) / 8);

//Initial MUSB
    md_usb_enable_it_ep1_tx(USB);       //Tx Interrupt Enable for EP1
    md_usb_enable_it_ep1_rx(USB);       //Rx Interrupt Enable for EP1

    md_usb_set_device_address(USB, 0);          //Default Address

#if APP_HIDENUM_MSG
    printf("HID EP Config Done\r\n");
#endif
}
/****
    * @brief    USB Application HID Device Initialization
    * @param    pDev    : Device Instance
    * @retval   None
****/
void    hidenum_dev_init(_pUSB_FS_DEVICE pDev)
{
    hidenum_dev_ep0_config(pDev);
//Device Callback
    pDev->cb_app_dev_request = (void *)&cb_app_dev_hidenum;
}

void usb_lpm(void)
{
    rcu_initStruct.Sw = MD_RCU_SW_SYSCLK_HRC;
    rcu_initStruct.HS_Clock = (RCU_CON_HRCON_MSK | RCU_CON_HRC48ON_MSK);
    md_rcu_sys_init(RCU, &rcu_initStruct);

    md_rcu_enable_usb_in_sleep_mode(RCU);

    SCB->SCR &= ~(1 << 2);
    __WFI();

    rcu_initStruct.Sw = MD_RCU_SW_SYSCLK_PLL0;
    rcu_initStruct.HS_Clock = (RCU_CON_HRCON_MSK | RCU_CON_PLL0ON_MSK | RCU_CON_HRC48ON_MSK);
    md_rcu_sys_init(RCU, &rcu_initStruct);
}

eUSBDState  hidenum_dev_enum(_pUSB_FS_DEVICE pDev, eUSBDState u8USBState)
{
    u8USBState = usbd_dev_enum(pDev, pSetupReq, u8USBState);

    switch (u8USBState)
    {
        case USBD_STATE_IDLE:
            return (USBD_STATE_IDLE);

        case USBD_STATE_RESET:          //Reset
#if APP_HIDENUM_MSG
            printf("Reset\r\n");
#endif
            hidenum_dev_ep_config(pDev);
            return (USBD_STATE_IDLE);

        case USBD_STATE_RESUME:         //Resume
#if APP_HIDENUM_MSG
            printf("Resume\r\n");
#endif
            return (USBD_STATE_IDLE);

        case USBD_STATE_SUSPEND:        //Suspend
#if APP_HIDENUM_MSG
            printf("Suspend\r\n");
#endif
            usb_lpm();
            return (USBD_STATE_IDLE);

        case USBD_STATE_SETUP:          //Setup
            if ((pSetupReq->bmRequestType & USB_REQ_TYPE_MASK) == USB_REQ_TYPE_CLASS)       //bmRequestType=Class
                u8USBState = (*pHIDClassRequest[pSetupReq->bRequest])(pDev, pSetupReq);

            return (u8USBState);

        case USBD_STATE_CTRLOUT:        //Ctrl Out
            return (USBD_STATE_CTRLOUT);

        case USBD_STATE_CTRLIN:         //Ctrl In
//          u8Event=usbd_handle_ctrl_in();
//          return(u8Event);
            return (USBD_STATE_CTRLIN);

        case USBD_STATE_EP1IN:          //EP1, Bulk In
            return (USBD_STATE_IDLE);

        case USBD_STATE_EP1OUT:         //EP1, Bulk Out
            return (USBD_STATE_EP1OUT);

        default:
            return (USBD_STATE_IDLE);
    }

    return (USBD_STATE_IDLE);
}
/****
    * @brief    USB Application HID Device Enumeration
    * @param    pDev        : Device Instance
    * @param    u8USBState  : Current Enumeration State
    * @retval   Updated Enumeration State
****/
eUSBDState  hidenum_dev_transaction(_pUSB_FS_DEVICE pDev, eUSBDState u8USBState)
{
    uint16_t    u16DLen, ii;

#if APP_HIDENUM_MSG
//  printf("Transaction\r\n");
#endif

    if (u8USBState != USBD_STATE_EP1OUT)
        return (USBD_STATE_IDLE);

#if APP_HIDENUM_MSG
//  printf("Command\r\n");
#endif
    u16DLen = usbd_epx_out_available(pDev, USBEP1);
    usbd_handle_epx_out(pDev, USBEP1, u16DLen, ENUMDataBuf);
    return (USBD_STATE_IDLE);
}

/******************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
