/**********************************************************************************
 *
 * @file    iap_update.c
 * @brief   c File
 *
 * @date    5 May 2022
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          5 May 2022      AE Team         the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */
/* Includes-------------------------------------------------------------------*/

#include "es32f0283.h"
#include "md_syscfg.h"
#include "md_fc.h"
#include "iap_update.h"

#pragma pack(1)

/*  Private Constants----------------------------------------------------------*/
/*  Public Structure ----------------------------------------------------------*/
/*  Public Macros -------------------------------------------------------------*/
/*  Public Constants ----------------------------------------------------------*/
#define FC_INFO         (1 << 24) //Information Enable

/*  Private variables ---------------------------------------------------------*/
static  md_fc_ControlTypeDef  FCControlPara;
/*  Private Macros ------------------------------------------------------------*/
/*  Private function prototypes -----------------------------------------------*/
/*  Private Functions ---------------------------------------------------------*/

/****
    * @addtogroup Private_Function
    * @{
****/

/****
    * @brief  Update remap location, either IAP or Application, based on situration
    * @param  EFAddr: flash address
    * @retval 0 : No error, 1 : Error
****/
uint8_t iap_update_remap(uint32_t EFAddr)
{
    uint32_t EFIPage2, RemapData;

    EFIPage2 = (MD_FC_PC_EF_IPAGESZ * 2);

    md_fc_unlock();
    FCControlPara.SAddr = EFIPage2;
    FCControlPara.SAddrC = ~EFIPage2;
    // Enable Info erase/program
    md_fc_enable_info_region(FC);
    //Erase Info Page2
    md_fc_page_erase(&FCControlPara);

    RemapData = 0xff000000UL | (EFAddr / 0x1000);
    EFIPage2 += 8;
    FCControlPara.SAddr = EFIPage2;
    FCControlPara.SAddrC = ~EFIPage2;
    FCControlPara.BCnt = 4;
    FCControlPara.pU32Buf = &RemapData;
    // Enable Info erase/program
    md_fc_enable_info_region(FC);
    // Update remap
    md_fc_program(&FCControlPara);
    md_fc_lock();

    return (0);
}

/****
    * @brief  Flash start address fixed on IAP boot loader
    * @param  EFAddr: flash address
    * @retval 0 : No error, 1 : Error
****/
uint8_t iap_bldr_fixed(uint32_t EFAddr)
{
//  iap_update_remap(EFAddr);

    return (0);
}

/****
    * @brief  Check if the APP been programmed and jump to the APP
    * @param  EFAddr: flash address
    * @retval 0 : No error, 1 : Error
****/
uint8_t iap_check_and_run(uint32_t EFAddr)
{
    /* Test if user code is programmed starting from address "APPLICATION_ADDRESS" */
    if (((*(__IO uint32_t *)(EBDDATA_BASE + EFAddr)) & 0x2FFC0000) == 0x20000000)   //The first 32bit data = stack top
    {
        __disable_irq();

//      iap_update_remap(EFAddr);
        md_syscfg_set_flash_remap_base(SYSCFG, (EFAddr / 0x1000));      //Remap Base
        md_syscfg_set_memory_mapping(SYSCFG, MD_SYSCFG_MEMMOD_MAIN);    //Remap to EFlash
        md_syscfg_enable_memory_remap(SYSCFG);          //Start Remap

        NVIC_SystemReset();

        while (1);
    }

    return (0);
}

/****
    * @brief  Erase IAP Flash
    * @param  EFAddr: flash address
    * @param  EFSize: flash size
    * @retval 0 : No error, 1 : Error
****/
uint8_t iap_flash_erase(uint32_t EFAddr, uint32_t EFSize)
{
    uint8_t Page ;

    if ((EFAddr >= (IAP_FLASH_START + IAP_FLASH_SIZE)) || (EFAddr < IAP_FLASH_START))
        return (1);

    if (EFSize > IAP_FLASH_SIZE)
        return (1);

    Page = ((EFSize + IAP_FLASH_PAGE_SIZE - 1) / IAP_FLASH_PAGE_SIZE);

    while (Page--)
    {
        FCControlPara.SAddr = EFAddr;
        FCControlPara.SAddrC = ~EFAddr;

        if (md_fc_unlock() == 0)
            return (1);

        if (md_fc_page_erase(&FCControlPara) == 0)
        {
            md_fc_lock();
            return (1);
        }

        if (md_fc_lock() == 0)
            return (1);

        EFAddr += IAP_FLASH_PAGE_SIZE;
    }

    return (0);
}
/****
    * @brief  Program IAP Flash
    * @param  EFAddr: flash address
    * @param  Length: byte length to write
    * @param  pData: data buffer to write
    * @retval 0 : No error, 1 : Error
****/
uint8_t iap_flash_program(uint32_t EFAddr, uint32_t Length, uint32_t *pData)
{
    if ((EFAddr >= (IAP_FLASH_START + IAP_FLASH_SIZE)) || (EFAddr < IAP_FLASH_START))
        return (1);

    Length = ((Length + 3) / 4) * 4;
    FCControlPara.SAddr = EFAddr;
    FCControlPara.SAddrC = ~(EFAddr);
    FCControlPara.BCnt = Length;
    FCControlPara.pU32Buf = pData;

    if (md_fc_unlock() == 0)
        return (1);

    if (md_fc_program(&FCControlPara) == 0)
    {
        md_fc_lock();
        return (1);
    }

    if (md_fc_lock() == 0)
        return (1);

    return (0);
}

/****
    * @brief  Read IAP Flash
    * @param  EFAddr: flash address
    * @param  Length: byte length to read
    * @param  pData: data buffer to read
    * @retval 0 : No error, 1 : Error
****/
uint8_t iap_flash_read(uint32_t EFAddr, uint32_t Length, uint32_t *pData)
{
    uint32_t    *pEFAddr;

    if ((EFAddr >= (IAP_FLASH_START + IAP_FLASH_SIZE)) || (EFAddr < IAP_FLASH_START))
        return (1);

    pEFAddr = (uint32_t *)(EBDDATA_BASE + EFAddr);
    Length = ((Length + 3) / 4) * 4;

    while (Length--)
    {
        *pData++ = *pEFAddr++;
    }

    return (0);
}

/****
    * @brief  Get IAP Flash Info
    * @param  pEFAddr: flash start address
    * @param  pEFSize: flash size
    * @retval None
****/
void iap_flash_info(uint32_t *pEFAddr, uint32_t *pEFSize)
{
    if (pEFAddr)
        *pEFAddr = IAP_FLASH_START;

    if (pEFSize)
        *pEFSize =  IAP_FLASH_SIZE;
}
/******************* (C) COPYRIGHT Eastsoft Microelectronics END OF FILE****/

