/**********************************************************************************
 *
 * @file    eslab_parser.c
 * @brief   c File
 *
 * @date    5 May 2022
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          5 May 2022      AE Team         the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */
#include "es32f0283.h"
#include "iap_update.h"
#include "eslab_parser.h"
#include <stdio.h>

#pragma pack(1)

#define PARSER_MSG  1

static  const   char    *BLDRResponseStr[] =
{
    "OK\r\n",
    "CMD_INVALID\r\n",
    "PARAM_ERROR\r\n",
    "COUNT_ERROR\r\n",
    "ADDR_ERROR\r\n",
    "BUSY\r\n",
    "*\r\n",
    "CONFIG_FAIL\r\n",
    "CHECK_ERROR\r\n",
    "SLAVE_NACK\r\n",
    "SLAVE_BUSY\r\n",
};

static  uint32_t    HIDIAPAddr, HIDIAPSize;
uint32_t    HIDIAPStart;
uint8_t     HIDIAPDone = 0;

/**
  * @brief Calculate the checksum of the app flash
  * @param Addr is the flash address
  * @param Size is the flash size
  * @retval checksum
  */
static uint32_t eslab_flash_checksum(uint32_t EFAddr, uint32_t EFSize)
{
    uint32_t    cksum, *pEFAddr;

    printf("Flash CkSum-0x%x-0x%x", EFAddr, EFSize);
    cksum = 0;
    pEFAddr = (uint32_t *)(EBDDATA_BASE + EFAddr);
    EFSize = (EFSize + sizeof(uint32_t) - 1) / sizeof(uint32_t);

    while (EFSize--)
        cksum += *pEFAddr++;

    return (cksum);
}

/**
  * @brief The calculate the check sum of request
  * @param pRequest is pointer to the request data .
  * @retval 0:match, 1:not match
  */
static  uint8_t eslab_request_cksum(uint8_t *pRequest)
{
    uint8_t     cksum;

    cksum = 0;

    while (*pRequest != 0)
    {
        cksum += *pRequest;
        pRequest++;

        if (*pRequest == ESLAB_CMD_LF)
            break;
    }

    cksum += *pRequest++;

    if (cksum != *pRequest)
    {
#if PARSER_MSG
        printf("CkSum-0x%x-0x%x\r\n", cksum, *pRequest);
#endif
        return (1);
    }

    return (0);
}

/**
  * @brief The end position of request string. (index of "\r\n" + 2)
  * @param pRequest is pointer to the request data .
  * @retval index if "\r\n" found, 0 if "\r\n" not found
  */
static  uint8_t eslab_request_end(uint8_t *pRequest)
{
    uint8_t index;

    index = 0;

    while ((*pRequest != ESLAB_CMD_LF) && (*pRequest != 0))
    {
        pRequest++;
        index ++;
    }

    if (*pRequest == ESLAB_CMD_LF)
        index += 2;
    else
        index = 0;

    return (index);
}
/**
  * @brief The string is converted to the hex data
  * @param pStr is pointer to the string .
  * @param pData is pointer to the data.
  * @retval 0:done, 1:error
  */
static  uint8_t str2hex(uint8_t *pStr, uint32_t *pData)
{
    *pData = 0;

    while (*pStr != ESLAB_CMD_SPACE)
    {
        if ((*pStr == 'x') || (*pStr == 'X'))
        {
            *pData = 0;
            pStr++;
            continue;
        }

        if ((*pStr >= '0') && (*pStr <= '9'))       //Digial 0~9
        {
            *pData = ((*pData << 4) | (*pStr - '0'));
            pStr++;
            continue;
        }

        if ((*pStr >= 'A') && (*pStr <= 'F'))       //Digial A~F
        {
            *pData = ((*pData << 4) | (*pStr - 'A') + 10);
            pStr++;
            continue;
        }

        if ((*pStr >= 'a') && (*pStr <= 'f'))       //Digial a~f
        {
            *pData = ((*pData << 4) | (*pStr - 'a') + 10);
            pStr++;
            continue;
        }

        return (1);         //Invalid
    }

    return (0);
}

/**
  * @brief Response the request command based on response code
  * @param Command is command code .
  * @param RspnCode is response code .
  * @param pResponse is pointer to the response data.
  * @retval 0
  */
uint8_t eslab_command_response(int8_t Command, uint8_t RspnCode, uint8_t *pResponse)
{
    uint8_t     cksum, ii;


    for (ii = 0; ii < 64; ii++)
        *(pResponse + ii) = 0;

    cksum = 0;
    ii = 0;
    *pResponse = ESLAB_CMD_ACK;
    cksum += *pResponse++;
    *pResponse = Command;
    cksum += *pResponse++;

    while (BLDRResponseStr[RspnCode][ii] != ESLAB_CMD_END)
    {
#if PARSER_MSG
        printf("%c", BLDRResponseStr[RspnCode][ii]);
#endif
        *pResponse = BLDRResponseStr[RspnCode][ii++];
        cksum += *pResponse++;
    }

    *pResponse++ = cksum;
    *pResponse = ESLAB_CMD_END;
    return (0);
}
/**
  * @brief Parse the command parameter
  * @param pRequest is pointer to the request data.
  * @param pParameter is pointer to the  parameter data.
  * @param nParam is the number of parameter.
  * @retval 0:Done, 1:Error
  */
static  uint8_t eslab_parameter_parser(uint8_t *pRequest, uint32_t *pParameter, uint8_t nParam)
{
    uint8_t     ii;

    for (ii = 0; ii < nParam; ii++)
    {
        while (*pRequest == ESLAB_CMD_SPACE)
        {
            pRequest++;
        }

        if (*pRequest == 0)
            return (1);

        if ((*pRequest == ESLAB_CMD_CR) || ((*pRequest == ESLAB_CMD_LF)))
            return (1);

        if (str2hex(pRequest, pParameter))
            return (1);

        while (*pRequest != ESLAB_CMD_SPACE)
        {
            pRequest++;
        }

        pParameter++;
    }

    while (*pRequest == ESLAB_CMD_SPACE)
    {
        pRequest++;
    }

    if (*pRequest++ != ESLAB_CMD_CR)
        return (1);

    if (*pRequest != ESLAB_CMD_LF)
        return (1);

    return (0);
}
/**
  * @brief Parse the request command and execute function
  * @param pRequest is pointer to the request data .
  * @param pResponse is pointer to the response data.
  * @retval Response code
  */
uint8_t eslab_command_parser(uint8_t *pRequest, uint8_t *pResponse)
{
    eESLAB_ResponseCode ResponseCode;
    uint8_t     Command;
    uint32_t    Param[2], IAPCKSum, FlashCKSum;
    uint32_t    Data[64 / sizeof(uint32_t)];
    uint16_t    ii;

    Command = 0;

    if (eslab_request_cksum(pRequest))
    {
        ResponseCode = ESLAB_ACK_PARAM_ERROR;
        eslab_command_response(Command, ResponseCode, pResponse);
        return (ResponseCode);
    }

    if (*pRequest++ != ESLAB_CMD_REQUEST)
    {
        ResponseCode = ESLAB_ACK_CMD_INVALID;
        eslab_command_response(Command, ResponseCode, pResponse);
        return (ResponseCode);
    }

    Command = *pRequest++;

    switch (Command)
    {
        case ESLAB_CMD_WRITE:
#if PARSER_MSG
            printf("W-");
#endif

            if (eslab_parameter_parser(pRequest, Param, 2))
            {
#if PARSER_MSG
                printf("0x%x-0x%x-", Param[0], Param[1]);
#endif
                ResponseCode = ESLAB_ACK_PARAM_ERROR;
                break;
            }

#if PARSER_MSG
            printf("0x%x-0x%x-", Param[0], Param[1]);
#endif
            iap_flash_info(&HIDIAPStart, &HIDIAPSize);

//          if ((Param[0] < HIDIAPStart) || (Param[1] > HIDIAPSize))
            if ((Param[0] >= (HIDIAPStart + HIDIAPSize)) || (Param[0] < HIDIAPStart))
            {
                ResponseCode = ESLAB_ACK_PARAM_ERROR;
                break;
            }

            if (Param[1] > (HIDIAPSize - Param[0]))
            {
                ResponseCode = ESLAB_ACK_PARAM_ERROR;
                break;
            }

            if (iap_bldr_fixed(IAP_BLDR_START))
            {
                ResponseCode = ESLAB_ACK_CONFIG_FAIL;
                break;
            }

            if (iap_flash_erase(Param[0], Param[1]))
            {
                ResponseCode = ESLAB_ACK_SLAVE_NACK;
                break;
            }

            HIDIAPStart = Param[0];
            HIDIAPAddr = Param[0];
            HIDIAPSize = Param[1];
            HIDIAPDone = 0;
            ResponseCode = ESLAB_ACK_OK;
            break;

        case ESLAB_CMD_SEND:
#if PARSER_MSG
            printf("D-");
#endif

            if (eslab_parameter_parser(pRequest, Param, 2))
            {
                ResponseCode = ESLAB_ACK_PARAM_ERROR;
#if PARSER_MSG
                printf("Error\r\n");
#endif
                break;
            }

//          HIDIAPAddr = (HIDIAPStart + Param[0]);
            pRequest += eslab_request_end(pRequest);

            if (Param[1] == 0xff)       //Checksum
            {
                IAPCKSum = ((pRequest[3] << 24) | (pRequest[2] << 16) | (pRequest[1] << 8) | pRequest[0]);
#if PARSER_MSG
                printf("C-0x%x-0x%x-0x%x\t", HIDIAPStart, Param[0], IAPCKSum);
#endif
                FlashCKSum = eslab_flash_checksum(HIDIAPStart, HIDIAPSize);

                if (IAPCKSum != FlashCKSum)
                {
#if PARSER_MSG
                    printf("\r\nIAP=0x%x\tFlash=0x%x\t", IAPCKSum, FlashCKSum);
#endif
                    iap_flash_erase(HIDIAPStart, HIDIAPSize);
                    ResponseCode = ESLAB_ACK_CHECK_ERROR;
                    break;
                }

                HIDIAPDone = 1;
            }
            else
            {
#if PARSER_MSG
                printf("P-0x%x\t", HIDIAPAddr);
#endif

                for (ii = 0; ii < Param[1]; ii++)
                {
                    *((uint8_t *)Data + ii) = *(pRequest + ii);
                }

                if (iap_flash_program(HIDIAPAddr, Param[1], (uint32_t *)Data))
                {
                    ResponseCode = ESLAB_ACK_SLAVE_BUSY;
                    break;
                }

                HIDIAPAddr += Param[1];
            }

            ResponseCode = ESLAB_ACK_OK;
            break;

        default:
#if PARSER_MSG
            printf("Invalid\r\n");
#endif
            ResponseCode = ESLAB_ACK_CMD_INVALID;
            break;
    }

    eslab_command_response(Command, ResponseCode, pResponse);
    return (ResponseCode);
}
/******************* (C) COPYRIGHT Eastsoft Microelectronics END OF FILE****/
