/**
  **************************************************************************************
  * @file    main.c
  * @brief   main C File
  *
  * @version V1.00
  * @date    13/11/2018
  * @author  Eastsoft AE Team
  * @note
  *
  * Copyright (C) 2018 Shanghai Eastsoft Microelectronics Co., Ltd. ALL rights reserved.
  *
  **************************************************************************************
  */

/* Define to prevent recursive inclusion -------------------------------------*/
#define MAIN_GLOBALS

/* Includes -------------------------------------------------------------------*/
#include "main.h"
#include "md_dma.h"
#include "usb_core.h"
#include "usb_enum.h"
#include "usb_dev_enum.h"
#include "app_dev_dual_cdc.h"
#include "cdc_ring_ctrl.h"
#include <stdio.h>

/** @addtogroup Projects_Examples_MD
  * @{
  */

/* Private constants ----------------------------------------------------------*/
/* Private types --------------------------------------------------------------*/
/** @defgroup RCU_Init_Struct RCU Init Structure
  * @{
  */
md_rcu_init_typedef rcu_initStruct =    /**< RCU init structure */
{
    MD_RCU_MPRE_MCO_DIV1,
    MD_RCU_MSW_MCO_DISABLE,
    MD_RCU_PLLSRC_HRC48,
    MD_RCU_PLLCLK_PASS,
    MD_RCU_PPRE_HCLK_DIV_1,
    MD_RCU_HPRE_SYSCLK_DIV_2,
    MD_RCU_SW_SYSCLK_HRC48,
    (RCU_CON_HRCON | RCU_CON_PLL0ON | RCU_CON_HRC48ON),
};

/** @defgroup UART_Init_Struct UART Init Structure
  * @{
  */
md_uart_init_typedef uart_initStruct =    /**< UART init structure */
{
    MD_UART_BAUDRATE_115200,
    MD_UART_LCON_LSB_FIRST,
    MD_UART_LCON_PS_NONE,
    MD_UART_LCON_STOP_1,
    MD_UART_LCON_DLS_8,
};

/* Private constants ----------------------------------------------------------*/
extern  _CDC_RING_CONTROL   CDC2RingCtrl;
extern  uint8_t CDC2DataBuf[CDC_RING_SIZE];
/* Private types --------------------------------------------------------------*/
_USB_FS_DEVICE  CDC_FS_Dev;

/* Private variables ----------------------------------------------------------*/
/* Private macros -------------------------------------------------------------*/
/* Private function prototypes ------------------------------------------------*/

/* Public functions -----------------------------------------------------------*/
/** @addtogroup Public_Function
  * @{
  */

void  DMAConfig()
{
//Master Enable
    md_dma_enable_master(DMA1);     //Master Enable

//DMA Channel3
    md_dma_set_primary_channel_channel_configuration(DMA1, MD_DMA_CHANNEL3, DMA3CFG);       //CDC1 Rx to CDC1 Tx
    md_dma_set_request_peripherals(DMA1, MD_DMA_CHANNEL3, MD_DMA_PRS_MEMORY);       //Memory, Memory to Memory
    md_dma_set_primary_channel_source_data_end_address(DMA1, MD_DMA_CHANNEL3, (uint32_t)&USBD->EP2FIFO);    //Source Address
    md_dma_set_primary_channel_destination_data_end_address(DMA1, MD_DMA_CHANNEL3, (uint32_t)&USBD->EP2FIFO);   //Target Address

//DMA Channel4
    md_dma_set_primary_channel_channel_configuration(DMA1, MD_DMA_CHANNEL4, DMA4CFG);       //CDC2 Rx to UART2 Tx
    md_dma_set_request_peripherals(DMA1, MD_DMA_CHANNEL4, MD_DMA_PRS_UART2_TX); //UART2 Tx, Memory to Peripherial
    md_dma_set_primary_channel_source_data_end_address(DMA1, MD_DMA_CHANNEL4, (uint32_t)&USBD->EP4FIFO);    //Source Address
    md_dma_set_primary_channel_destination_data_end_address(DMA1, MD_DMA_CHANNEL4, (uint32_t)&UART2->TXDATA);   //Target Address

//DMA Channe5
    md_dma_set_primary_channel_channel_configuration(DMA1, MD_DMA_CHANNEL5, DMA5CFG);       //UART2 Ring to CDC2 Tx
    md_dma_set_request_peripherals(DMA1, MD_DMA_CHANNEL5, MD_DMA_PRS_MEMORY);       //Memory, Memory to Memory
    md_dma_set_primary_channel_destination_data_end_address(DMA1, MD_DMA_CHANNEL5, (uint32_t)&USBD->EP4FIFO);   //Target Address

//DMA Interrupt
    md_dma_enable_it_channel_done(DMA1, MD_DMA_CHANNEL3);                   //DMA Ch3 Block Transmit Done
    md_dma_enable_it_channel_done(DMA1, MD_DMA_CHANNEL4);                   //DMA Ch4 Block Transmit Done
    md_dma_enable_it_channel_done(DMA1, MD_DMA_CHANNEL5);                   //DMA Ch5 Block Transmit Done

}

/**
  * @brief  IrqInit
  * @param  None
  * @retval None
  */
void IrqInit(void)
{
    NVIC->ICER[0] = 0xFFFFFFFFUL;
    NVIC_EnableIRQ((IRQn_Type)USB_IRQn);        //USB IRQ
    NVIC_EnableIRQ((IRQn_Type)UART2_IRQn);      //UART2 IRQ
    NVIC_EnableIRQ((IRQn_Type)DMA1_CH345_IRQn); //DMA Channel 345 IRQ
}

/**
  * @brief  set irq priority
  * @param  None
  * @retval None
  */
void IrqPriority(void)
{
    NVIC_SetPriority((IRQn_Type)USB_IRQn, 0);
    NVIC_SetPriority((IRQn_Type)DMA1_CH345_IRQn, 1);    //DMA Channel 345 IRQ
    NVIC_SetPriority((IRQn_Type)UART2_IRQn, 2);
}

/**
  * @brief  configure i/o multiplexer
  * @param  None
  * @retval None
  */
void Iomux(void)
{
    /*  UART1 IOMux  */
    md_gpio_set_pin_mode(GPIOB, MD_GPIO_PIN_6,   MD_GPIO_MODE_FUNCTION);
    md_gpio_set_pin_mode(GPIOB, MD_GPIO_PIN_7,   MD_GPIO_MODE_FUNCTION);
    md_gpio_set_pin_pull(GPIOB, MD_GPIO_PIN_7,   MD_GPIO_PULL_UP);
    md_gpio_set_pin_function0_7(GPIOB, MD_GPIO_PIN_6,   MD_GPIO_AF2);
    md_gpio_set_pin_function0_7(GPIOB, MD_GPIO_PIN_7,   MD_GPIO_AF2);

    /*  UART2 IOMux  */
    md_gpio_set_pin_mode(GPIOA, MD_GPIO_PIN_2,   MD_GPIO_MODE_FUNCTION);
    md_gpio_set_pin_mode(GPIOA, MD_GPIO_PIN_3,   MD_GPIO_MODE_FUNCTION);
    md_gpio_set_pin_pull(GPIOA, MD_GPIO_PIN_3,   MD_GPIO_PULL_UP);
    md_gpio_set_pin_function0_7(GPIOA, MD_GPIO_PIN_2,   MD_GPIO_AF2);
    md_gpio_set_pin_function0_7(GPIOA, MD_GPIO_PIN_3,   MD_GPIO_AF2);
}

/**
  * @brief  System Peri init
  * @param  None
  * @retval None
  */
void SysPeriInit(void)
{
    /*  AHB Clock Enable  */
    md_rcu_enable_gpiod(RCU);
    md_rcu_enable_gpioc(RCU);
    md_rcu_enable_gpiob(RCU);
    md_rcu_enable_gpioa(RCU);

    md_rcu_enable_usb(RCU);
    md_rcu_enable_dma1(RCU);

    /*  APB Clock Enable  */
    md_rcu_enable_uart1(RCU);
    md_rcu_enable_uart2(RCU);
}

/**
  * @brief  CoreInit.
  * @param  None
  * @retval None
  */
void  CoreInit()
{
    md_rcu_pll0_init(RCU, &rcu_initStruct);
    md_rcu_sys_init(RCU, &rcu_initStruct);

    SysPeriInit();
    IrqInit();
    Iomux();

    md_uart_init(UART1, &uart_initStruct);
    md_uart_init(UART2, &uart_initStruct);
    md_tick_init(SystemFrequency_AHBClk);

    md_uart_disable_rx(UART2);
    md_uart_enable_txdma(UART2);
//  md_uart_enable_it_rfnempty(UART2);
}

/**
  * @brief  Uart sendchar.
  * @param  arg: char to be sent.
  * @retval data to be sent.
  */
uint8_t  sendchar(uint8_t ch)
{
    while (!md_uart_is_active_flag_tfempty(UART1));  /* Tx FIFO full */

    md_uart_set_send_data8(UART1, ch);          // Send byte
    return (ch);
}

/**
  * @brief  main.
  * @param  None
  * @retval None
  */
int main(void)
{
    eUSBDState  u8USBState;
    uint32_t    DMACfgWord;
    uint32_t    DMADataEnd;
    uint16_t    u16CDCDLen;

    CoreInit();

    printf("USBD CDC Enumeration\r\n");
    printf("SysFosc=%dHz\r\n", SystemCoreClock);

    DMAConfig();
    cdc_ring_init(&CDC2RingCtrl, CDC_RING_SIZE, CDC2DataBuf);
    dual_cdc_dev_init(&CDC_FS_Dev);
    usb_core_mode_set(eUSBModeForceDevice);

    u8USBState = USBD_STATE_IDLE;

    while (1)
    {
        u8USBState = dual_cdc_dev_enum(&CDC_FS_Dev, u8USBState);

        if (u8USBState == USBD_STATE_EP2OUT)        //CDC1 Data Out
        {
            u8USBState = dual_cdc_dev_transaction(&CDC_FS_Dev, u8USBState);
        }

        if (u8USBState == USBD_STATE_EP4OUT)        //CDC2 Data Out
        {
            u8USBState = dual_cdc_dev_transaction(&CDC_FS_Dev, u8USBState);
        }

        if (u8USBState == USBD_STATE_EP2IN)         //CDC1 Data In
        {
            u8USBState = dual_cdc_dev_transaction(&CDC_FS_Dev, u8USBState);
        }

        if (u8USBState == USBD_STATE_EP4IN)         //CDC2 Data In
        {
            u8USBState = dual_cdc_dev_transaction(&CDC_FS_Dev, u8USBState);
        }

        if (cdc_ring_used(&CDC2RingCtrl))
        {
            u8USBState = USBD_STATE_EP4IN;
            u8USBState = dual_cdc_dev_transaction(&CDC_FS_Dev, u8USBState);
        }
    }

    while (1);
}

/**
  * @} Public_Function
  */

/**
  * @} Projects_Examples_MD
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/


