/**********************************************************************************
 *
 * @file    cdc_ring_ctrl.c
 * @brief   c File
 *
 * @date    5 May 2022
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          5 May 2022      AE Team         the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */
#include "es32f0283.h"
#include "cdc_ring_ctrl.h"
#include <stdio.h>

#pragma pack(1)

void cdc_ring_init(_pCDC_RING_CONTROL pRingCtrl, uint16_t RingSize, uint8_t *pRingBuf)
{
    pRingCtrl->RingSize = RingSize;
    pRingCtrl->pRingBuf = pRingBuf;
    pRingCtrl->RingHead = 0;
    pRingCtrl->RingTail = 0;
}

void cdc_ring_push(_pCDC_RING_CONTROL pRingCtrl, uint8_t data)
{
    pRingCtrl->pRingBuf[pRingCtrl->RingHead] = data;

    __set_PRIMASK(1);
    pRingCtrl->RingHead ++;

    if (pRingCtrl->RingHead >= pRingCtrl->RingSize)
        pRingCtrl->RingHead = 0;

    __set_PRIMASK(0);
}

uint8_t cdc_ring_pop(_pCDC_RING_CONTROL pRingCtrl)
{
    uint8_t data;

    data = pRingCtrl->pRingBuf[pRingCtrl->RingTail];

    __set_PRIMASK(1);
    pRingCtrl->RingTail ++;

    if (pRingCtrl->RingTail >= pRingCtrl->RingSize)
        pRingCtrl->RingTail = 0;

    __set_PRIMASK(0);

    return data;
}

uint16_t cdc_ring_used(_pCDC_RING_CONTROL pRingCtrl)
{
    uint32_t cnt;

    __set_PRIMASK(1);

    if (pRingCtrl->RingHead >= pRingCtrl->RingTail)
        cnt = pRingCtrl->RingHead - pRingCtrl->RingTail;
    else
        cnt = pRingCtrl->RingHead + pRingCtrl->RingSize - pRingCtrl->RingTail;

    __set_PRIMASK(0);

    return cnt;
}

uint16_t cdc_ring_wrap(_pCDC_RING_CONTROL pRingCtrl)
{
    uint32_t cnt;

    __set_PRIMASK(1);

    if (pRingCtrl->RingHead >= pRingCtrl->RingTail)
        cnt = pRingCtrl->RingHead - pRingCtrl->RingTail;
    else
        cnt = pRingCtrl->RingSize - pRingCtrl->RingTail;

    __set_PRIMASK(0);

    return cnt;
}

uint16_t cdc_ring_free(_pCDC_RING_CONTROL pRingCtrl)
{
    uint32_t free;

    __set_PRIMASK(1);

    if (pRingCtrl->RingHead >= pRingCtrl->RingTail)
        free = pRingCtrl->RingSize - (pRingCtrl->RingHead - pRingCtrl->RingTail);
    else
        free = pRingCtrl->RingTail - pRingCtrl->RingHead;

    __set_PRIMASK(0);

    return free;
}

uint16_t cdc_ring_read(_pCDC_RING_CONTROL pRingCtrl, uint16_t size, uint8_t *data)
{
    uint32_t cnt;
    uint32_t ii;

    cnt = cdc_ring_used(pRingCtrl);

    if (size > cnt)
        size = cnt;

    for (ii = 0; ii < size; ii++)
        *data++ = cdc_ring_pop(pRingCtrl);

    return size;
}

uint16_t cdc_ring_write(_pCDC_RING_CONTROL pRingCtrl, uint16_t size, uint8_t *data)
{
    uint32_t free;
    uint32_t ii;

    free = cdc_ring_free(pRingCtrl);

    if (size > free)
        size = free;

    for (ii = 0; ii < size; ii++)
        cdc_ring_push(pRingCtrl, *data++);

    return size;
}

void cdc_ring_flush(_pCDC_RING_CONTROL pRingCtrl)
{
    __set_PRIMASK(1);
    pRingCtrl->RingTail = 0;
    pRingCtrl->RingHead = 0;
    __set_PRIMASK(0);
}

/******************* (C) COPYRIGHT Eastsoft Microelectronics END OF FILE****/
