/**********************************************************************************
 *
 * @file    cb_dev_dual_cdc.c
 * @brief   c File
 *
 * @date    5 May 2022
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          5 May 2022      AE Team         the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */
#include "es32f0283.h"
#include "md_usb.h"
#include "usb_core.h"
#include "usb_enum.h"
#include "usb_dev_enum.h"
#include "usbhid.h"
#include "usbcdc.h"
#include "usbmsc.h"
#include "usb_dev_cdc.h"
#include "app_dual_cdc_desc.h"
#include "cb_dev_dual_cdc.h"
#include "cdc_ring_ctrl.h"
#include "md_uart.h"
#include <stdio.h>

#pragma pack(1)
#define CB_DEV_APP_MSG  1

extern  _CDC_RING_CONTROL   CDC2RingCtrl;

//_CB_ENUM_REQUEST
static  eUSBDState  cb_dev_request_enum(_pUSB_FS_DEVICE, _pUSB_SETUP_REQUEST);

//_CB_CLASS_REQUEST
static  eUSBDState  cb_dev_request_cdc(_pUSB_FS_DEVICE, _pUSB_SETUP_REQUEST);
static  eUSBDState  cb_dev_request_hid(_pUSB_FS_DEVICE, _pUSB_SETUP_REQUEST);
static  eUSBDState  cb_dev_request_msc(_pUSB_FS_DEVICE, _pUSB_SETUP_REQUEST);

_CB_APP_DEV_REQUEST cb_app_dev_dual_cdc =
{
    cb_dev_request_enum,    //cb_dev_request_enum
    cb_dev_request_cdc,     //cb_dev_request_cdc
};

static  _CDC_LINE_CODING_DATA   CDC1LineCoding =
{
    CDC_LINE_CODING_DTR_115200,
    CDC_LINE_CODING_STOP_1,
    CDC_LINE_CODING_PARITY_NONE,
    CDC_LINE_CODING_BITS_8,
};
static  _CDC_LINE_CODING_DATA   CDC2LineCoding =
{
    CDC_LINE_CODING_DTR_115200,
    CDC_LINE_CODING_STOP_1,
    CDC_LINE_CODING_PARITY_NONE,
    CDC_LINE_CODING_BITS_8,
};

/****
    * @brief    USB Device Enumeration Callback
    * @param    pDev        : Device Instance
    * @param    pSetupReq   : Pointer to Standard Request Data
    * @retval   Updated eUSBDState
****/
static  eUSBDState  cb_dev_request_enum(_pUSB_FS_DEVICE pDev, _pUSB_SETUP_REQUEST pSetupReq)
{
    uint8_t bValueH, bValueL;

#if CB_DEV_APP_MSG
    printf("CB Dev Enum - ");
#endif

    if ((pSetupReq->bmRequestType & USB_REQ_TYPE_MASK) != USB_REQ_TYPE_STANDARD)    //bmRequestType != Standard
        return (USBD_STATE_IDLE);

    if (pSetupReq->bRequest == USB_REQ_GET_DESCRIPTOR)
    {
#if CB_DEV_APP_MSG
        printf("Get Descriptor\r\n");
#endif
        bValueL = (pSetupReq->wValue & 0xff);
        bValueH = (pSetupReq->wValue >> 8);

        switch (bValueH)    //wValueH
        {
            case USB_DTYPE_DEVICE:  //Device
                pDev->pCtrlBurstData = (uint8_t *)&CDCDeviceDesc;       //Device Descriptor
                pDev->CtrlBurstRemain = sizeof(_USB_DEVICE_DESC);       //bLength
                break;

            case USB_DTYPE_CONFIGURATION:   //Configuration
                pDev->pCtrlBurstData = (uint8_t *)&CDCConfigurationDesc;        //Configuration Descriptor
                pDev->CtrlBurstRemain = sizeof(_CDC_CONFIGURATION_DESC);        //wTotalLength
                break;

            case USB_DTYPE_STRING:  //String
                switch (bValueL)
                {
                    case 0:
                        pDev->pCtrlBurstData = (uint8_t *)&CDCString0Desc;      //String 0 Descriptor
                        pDev->CtrlBurstRemain = sizeof(_USB_STRING_0_DESC);     //bLength
                        break;

                    case 1:
                        pDev->pCtrlBurstData = (uint8_t *)&CDCString1Desc;      //String 1 Descriptor
                        pDev->CtrlBurstRemain = sizeof(_USB_STRING_1_DESC);     //bLength
                        break;

                    case 2:
                        pDev->pCtrlBurstData = (uint8_t *)&CDCString2Desc;      //String 2 Descriptor
                        pDev->CtrlBurstRemain = sizeof(CDCString2Desc);         //bLength
                        break;

                    default:        //Not Supported
                        return (USBD_STATE_STALL);
                }

                break;

            case USB_DTYPE_QUALIFIER:       //Qualifier
                pDev->pCtrlBurstData = NULL;
                pDev->CtrlBurstRemain = 0;
                break;

            default:        //Not Supported
                return (USBD_STATE_STALL);
        }

        return (USBD_STATE_IDLE);
    }

    if (pSetupReq->bRequest == USB_REQ_SET_ADDRESS)
    {
#if CB_DEV_APP_MSG
        printf("Set Address\r\n");
#endif
        return (USBD_STATE_IDLE);
    }

    if (pSetupReq->bRequest == USB_REQ_SET_CONFIGURATION)
    {
#if CB_DEV_APP_MSG
        printf("Set Configuration\r\n");
#endif
        return (USBD_STATE_IDLE);
    }

    if (pSetupReq->bRequest == USB_REQ_SET_INTERFACE)
    {
#if CB_DEV_APP_MSG
        printf("Set Interface\r\n");
#endif
        return (USBD_STATE_IDLE);
    }

    return (USBD_STATE_IDLE);
}
/****
    * @brief    USB Device HID Class Callback
    * @param    pDev        : Device Instance
    * @param    pSetupReq   : Pointer to Standard Request Data
    * @retval   Updated eUSBDState
****/
static  eUSBDState  cb_dev_request_hid(_pUSB_FS_DEVICE pDev, _pUSB_SETUP_REQUEST pSetupReq)
{
#if CB_DEV_APP_MSG
    printf("CB Dev HID - ");
#endif

    if ((pSetupReq->bmRequestType & USB_REQ_TYPE_MASK) != USB_REQ_TYPE_CLASS)       //bmRequestType != Class
        return (USBD_STATE_IDLE);

    if (pSetupReq->bRequest == HID_REQ_GET_REPORT)
    {
#if CB_DEV_APP_MSG
        printf("Get Report\r\n");
#endif
        return (USBD_STATE_IDLE);
    }

    if (pSetupReq->bRequest == HID_REQ_GET_IDLE)
    {
#if CB_DEV_APP_MSG
        printf("Get Idle\r\n");
#endif
        return (USBD_STATE_IDLE);
    }

    if (pSetupReq->bRequest == HID_REQ_GET_PROTOCOL)
    {
#if CB_DEV_APP_MSG
        printf("Get Protocol\r\n");
#endif
        return (USBD_STATE_IDLE);
    }

    if (pSetupReq->bRequest == HID_REQ_SET_REPORT)
    {
#if CB_DEV_APP_MSG
        printf("Set Report\r\n");
#endif
        return (USBD_STATE_CTRLOUT);
    }

    if (pSetupReq->bRequest == HID_REQ_SET_IDLE)
    {
#if CB_DEV_APP_MSG
        printf("Set Idle\r\n");
#endif
        return (USBD_STATE_IDLE);
    }

    if (pSetupReq->bRequest == HID_REQ_SET_PROTOCOL)
    {
#if CB_DEV_APP_MSG
        printf("Set Protocol\r\n");
#endif
        return (USBD_STATE_IDLE);
    }

    return (USBD_STATE_IDLE);
}
/****
    * @brief    USB Device CDC Class Callback
    * @param    pDev        : Device Instance
    * @param    pSetupReq   : Pointer to Standard Request Data
    * @retval   Updated eUSBDState
****/
static  eUSBDState  cb_dev_request_cdc(_pUSB_FS_DEVICE pDev, _pUSB_SETUP_REQUEST pSetupReq)
{
    eUSBDState  USBState;
    uint8_t bIndexH, bIndexL;
    uint8_t bData;

#if CB_DEV_APP_MSG
    printf("CB Dev CDC - ");
#endif

    if ((pSetupReq->bmRequestType & USB_REQ_TYPE_MASK) != USB_REQ_TYPE_CLASS)       //bmRequestType != Class
        return (USBD_STATE_IDLE);

    bIndexL = (pSetupReq->wIndex & 0xff);
    bIndexH = (pSetupReq->wIndex >> 8);

    if (pSetupReq->bRequest == CDC_SET_LINE_CODING)
    {
#if CB_DEV_APP_MSG
        printf("Set Line Coding - ");
#endif
        USBState = USBD_STATE_SETUP;

        while (USBState != USBD_STATE_CTRLOUT)
            USBState = usbd_dev_enum(pDev, pSetupReq, USBState);

        if (bIndexL == CDC1_INTERFACE_COMM)     //CDC1
        {
#if CB_DEV_APP_MSG
            printf("CDC1\r\n");
#endif
            pDev->pCtrlBurstData = (uint8_t *)&CDC1LineCoding;      //CDC1 LineCoding Line Coding Data
            pDev->CtrlBurstRemain = pSetupReq->wLength;             //wLength
        }

        if (bIndexL == CDC2_INTERFACE_COMM)     //CDC2
        {
#if CB_DEV_APP_MSG
            printf("CDC2\r\n");
#endif
            pDev->pCtrlBurstData = (uint8_t *)&CDC2LineCoding;      //CDC2 LineCoding Line Coding Data
            pDev->CtrlBurstRemain = pSetupReq->wLength;             //wLength
        }

        usbd_handle_ctrl_out(pDev);
        return (USBD_STATE_IDLE);
    }

    if (pSetupReq->bRequest == CDC_GET_LINE_CODING)
    {
#if CB_DEV_APP_MSG
        printf("Get Line Coding - ");
#endif

        if (bIndexL == CDC1_INTERFACE_COMM)     //CDC1
        {
#if CB_DEV_APP_MSG
            printf("CDC1\r\n");
#endif
            pDev->pCtrlBurstData = (uint8_t *)&CDC1LineCoding;      //CDC1 LineCoding Line Coding Data
            pDev->CtrlBurstRemain = pSetupReq->wLength;             //wLength
        }

        if (bIndexL == CDC2_INTERFACE_COMM)     //CDC2
        {
#if CB_DEV_APP_MSG
            printf("CDC2\r\n");
#endif
            pDev->pCtrlBurstData = (uint8_t *)&CDC2LineCoding;      //CDC2 LineCoding Line Coding Data
            pDev->CtrlBurstRemain = pSetupReq->wLength;             //wLength
        }

        usbd_handle_ctrl_in(pDev);
        return (USBD_STATE_IDLE);
    }

    if (pSetupReq->bRequest == CDC_SET_CONTROL_LINE_STATE)
    {
#if CB_DEV_APP_MSG
        printf("Set Control Line State - ");
#endif

        if (bIndexL == CDC1_INTERFACE_COMM)     //CDC1
        {
#if CB_DEV_APP_MSG
            printf("CDC1 - 0x%x\r\n", pSetupReq->wValue);
#endif
        }

        if (bIndexL == CDC2_INTERFACE_COMM)     //CDC2
        {
#if CB_DEV_APP_MSG
            printf("CDC2 - 0x%x\r\n", pSetupReq->wValue);
#endif
#if 1

            if (pSetupReq->wValue & CDC_ACTIVATE_CARRIER)       //Activate Carrier
            {
                if (md_uart_get_rif(UART2))
                    md_uart_set_icr(UART2, md_uart_get_rif(UART2));

                if (md_uart_is_active_flag_rfnempty(UART2))
                    bData = md_uart_get_recv_data8(UART2);

                if (pSetupReq->wValue & CDC_DTE_PRESENT)        //DTE Present
                {
                    cdc_ring_flush(&CDC2RingCtrl);
                    md_uart_enable_rx(UART2);
                    md_uart_enable_it_rfnempty(UART2);
                }
                else
                {
                    md_uart_disable_rx(UART2);
                    md_uart_disable_it_rfnempty(UART2);
                }
            }

#endif
        }

        return (USBD_STATE_IDLE);
    }

    return (USBD_STATE_IDLE);
}
/****
    * @brief    USB Device MSC Class Callback
    * @param    pDev        : Device Instance
    * @param    pSetupReq   : Pointer to Standard Request Data
    * @retval   Updated eUSBDState
****/
static  eUSBDState  cb_dev_request_msc(_pUSB_FS_DEVICE pDev, _pUSB_SETUP_REQUEST pSetupReq)
{
#if CB_DEV_APP_MSG
    printf("CB Dev MSC - ");
#endif

    if ((pSetupReq->bmRequestType & USB_REQ_TYPE_MASK) != USB_REQ_TYPE_CLASS)       //bmRequestType != Class
        return (USBD_STATE_IDLE);

    if (pSetupReq->bRequest == MSC_REQ_GET_MAX_LUN)
    {
#if CB_DEV_APP_MSG
        printf("Get Max Lun\r\n");
#endif
        return (USBD_STATE_IDLE);
    }

    if (pSetupReq->bRequest == MSC_REQ_STORAGE_RESET)
    {
#if CB_DEV_APP_MSG
        printf("Storage Reset\r\n");
#endif
        return (USBD_STATE_IDLE);
    }

    return (USBD_STATE_IDLE);
}

/******************* (C) COPYRIGHT Eastsoft Microelectronics END OF FILE****/
