/**********************************************************************************
 *
 * @file    app_dev_dual_cdc.c
 * @brief   c File
 *
 * @date    5 May 2022
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          5 May 2022      AE Team         the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */
#include "es32f0283.h"
#include "md_uart.h"
#include "md_dma.h"
#include "md_usb.h"
#include "usb_core.h"
#include "usb_enum.h"
#include "usb_dev_enum.h"
#include "usbcdc.h"
#include "usb_dev_cdc.h"
#include "app_dual_cdc_desc.h"
#include "app_dev_dual_cdc.h"
#include "cb_dev_dual_cdc.h"
#include "cdc_ring_ctrl.h"
#include <stdio.h>

#pragma pack(1)
#define APP_DUAL_CDC_MSG        1

static  _pUSB_SETUP_REQUEST pSetupReq = (_pUSB_SETUP_REQUEST) &SetupReq;
uint8_t CDC1DataBuf[CDC_BULK_PKTSZ], CDC2DataBuf[CDC_RING_SIZE];
_CDC_RING_CONTROL   CDC2RingCtrl;
uint8_t     cdc1zdp, cdc2zdp;

/****
    * @brief    UART2 IRQ Handler
    * @param    None
    * @retval   None
****/
void UART2_IRQHandler(void)
{
    uint8_t u8Data;

    if (md_uart_is_active_flag_rfnempty(UART2))
    {
        md_uart_clear_it_rfnempty(UART2);
        u8Data = md_uart_get_recv_data8(UART2);
        cdc_ring_push(&CDC2RingCtrl, u8Data);
#if APP_DUAL_CDC_MSG
//      md_uart_set_send_data8(UART1, 'R');     // Send byte
#endif
    }
    else
    {
#if APP_DUAL_CDC_MSG
//      printf("0x%x\r\n", md_uart_get_rif(UART2));     // Send Data
#endif
        md_uart_set_icr(UART2, md_uart_get_rif(UART2));
    }
}
/****
    * @brief    DMA Channe_345 IRQ Handler
    * @param    None
    * @retval   None
****/
void DMA1_CH345_IRQHandler(void)
{
    uint32_t    DMACfgWord;
    uint32_t    DMADataEnd;

    uint8_t EPBackup;
//uint8_t   ii;

    EPBackup = md_usb_get_epx_index(USB);       //Backup EPx

    if (md_dma_is_active_flag_channel_done(DMA1, MD_DMA_CHANNEL3))  //DMA CH3, CDC1 Rx to CDC1 Tx
    {
        md_dma_clear_it_channel_done(DMA1, MD_DMA_CHANNEL3);

        md_usb_set_epx_index(USB, CDC1_ENDPOINT_DATA);
//      md_usb_clear_flag_device_epx_rx_ready(USB); //Clear RxRdy
        md_usb_enable_epx_tx_ready(USB);        //TxRdy

#if APP_DUAL_CDC_MSG
//      md_uart_set_send_data8(UART1, '3');         // Send byte
#endif
    }

    if (md_dma_is_active_flag_channel_done(DMA1, MD_DMA_CHANNEL4))  //DMA CH4, CDC2 Rx to UART2 Tx
    {
        md_dma_clear_it_channel_done(DMA1, MD_DMA_CHANNEL4);

        md_usb_set_epx_index(USB, CDC2_ENDPOINT_DATA);
        md_usb_clear_flag_epx_rx_ready(USB);    //Clear RxRdy

#if APP_DUAL_CDC_MSG
//      md_uart_set_send_data8(UART1, '4');         // Send byte
#endif
    }

    if (md_dma_is_active_flag_channel_done(DMA1, MD_DMA_CHANNEL5))  //DMA CH5, UART2 Rx to CDC2 Tx
    {
        md_dma_clear_it_channel_done(DMA1, MD_DMA_CHANNEL5);

        md_usb_set_epx_index(USB, CDC2_ENDPOINT_DATA);
        md_usb_enable_epx_tx_ready(USB);        //TxRdy

#if APP_DUAL_CDC_MSG
//      md_uart_set_send_data8(UART1, '5');         // Send byte
#endif
    }

    md_usb_set_epx_index(USB, EPBackup);        //Restore EPx
}

/****
    * @brief    USB Application CDC Device Endpoint 0 Configuration
    * @param    pDev    : Device Instance
    * @retval   None
****/
void    dual_cdc_dev_ep0_config(_pUSB_FS_DEVICE pDev)
{
//Configure EP0
    md_usb_set_epx_index(USB, USBEP0);          //Endpoint 0
    md_usb_set_tx_max_packet_size(USB, (USBD_CTRL_PKTSZ / 8));  //64 bytes
    md_usb_set_tx_fifo_start_addr(USB, 0x0);        //Start Address=0x0
    md_usb_set_tx_max_fifo_packet_size(USB, MD_USB_TXFIFO2_MAXPKTSIZE_64);  //Size=64
    md_usb_set_rx_max_packet_size(USB, (USBD_CTRL_PKTSZ / 8));  //64 bytes
    md_usb_set_rx_fifo_start_addr(USB, 0x0);        //Start Address=0x0
    md_usb_set_rx_max_fifo_packet_size(USB, MD_USB_RXFIFO2_MAXPKTSIZE_64);  //Size=64

    md_usb_enable_it_ep0(USB);      //Interrupt Enable for EP0
}
/****
    * @brief    USB Application CDC Device Endpoint Configuration
    * @param    pDev    : Device Instance
    * @retval   None
****/
void    dual_cdc_dev_ep_config(_pUSB_FS_DEVICE pDev)
{
    uint8_t     EPFIFOIdx;

    EPFIFOIdx = 0x8;
//Configure EP1 In
    md_usb_set_epx_index(USB, USBEP1);          //Endpoint 1
    md_usb_disable_epx_tx_isochronous(USB);         //Non ISO
    md_usb_set_epx_mode(USB, MD_USB_TXCSRH_MODE_TX);        //TX
    md_usb_set_tx_max_packet_size(USB, (CDC_INT_PKTSZ / 8));    //16 bytes
    md_usb_set_tx_fifo_start_addr(USB, EPFIFOIdx);      //Start Address=0x8
    md_usb_set_tx_max_fifo_packet_size(USB, MD_USB_TXFIFO2_MAXPKTSIZE_16);  //Size=16, No Double-Packet Buffering
    md_usb_trigger_epx_tx_flush_fifo(USB);          //Flush FIFO

    EPFIFOIdx += ((CDC_INT_PKTSZ + 7) / 8);

    if (md_usb_is_enabled_tx_double_packet(USB))    //Double-Packet Buffering
        EPFIFOIdx += ((CDC_INT_PKTSZ + 7) / 8);

//Configure EP2 In
    md_usb_set_epx_index(USB, USBEP2);          //Endpoint 2
    md_usb_disable_epx_tx_isochronous(USB);         //Non ISO
    md_usb_set_epx_mode(USB, MD_USB_TXCSRH_MODE_TX);        //TX
    md_usb_set_tx_max_packet_size(USB, CDC_BULK_PKTSZ / 8);     //64 bytes
    md_usb_set_tx_fifo_start_addr(USB, EPFIFOIdx);  //Start Address=0xa
    md_usb_set_tx_max_fifo_packet_size(USB, MD_USB_TXFIFO2_MAXPKTSIZE_64);  //Size=64, No Double-Packet Buffering
    md_usb_trigger_epx_tx_flush_fifo(USB);          //Flush FIFO

    EPFIFOIdx += ((CDC_BULK_PKTSZ + 7) / 8);

    if (md_usb_is_enabled_tx_double_packet(USB))    //Double-Packet Buffering
        EPFIFOIdx += ((CDC_BULK_PKTSZ + 7) / 8);

//Configure EP2 Out
    md_usb_set_epx_index(USB, USBEP2);          //Endpoint 2
    md_usb_disable_epx_rx_isochronous(USB);         //Non ISO
    md_usb_set_epx_mode(USB, MD_USB_TXCSRH_MODE_RX);        //RX
    md_usb_set_rx_max_packet_size(USB, CDC_BULK_PKTSZ / 8);     //64 Bytes
    md_usb_set_rx_fifo_start_addr(USB, EPFIFOIdx);  //Start Address=0x10
    md_usb_set_rx_max_fifo_packet_size(USB, MD_USB_RXFIFO2_MAXPKTSIZE_64);  //Size=64, No Double-Packet Buffering
    md_usb_trigger_epx_rx_flush_fifo(USB);          //Flush FIFO

    EPFIFOIdx += ((CDC_BULK_PKTSZ + 7) / 8);

    if (md_usb_is_enabled_rx_double_packet(USB))    //Double-Packet Buffering
        EPFIFOIdx += ((CDC_BULK_PKTSZ + 7) / 8);

//Configure EP3 In
    md_usb_set_epx_index(USB, USBEP3);          //Endpoint 3
    md_usb_disable_epx_tx_isochronous(USB);         //Non ISO
    md_usb_set_epx_mode(USB, MD_USB_TXCSRH_MODE_TX);        //TX
    md_usb_set_tx_max_packet_size(USB, (CDC_INT_PKTSZ / 8));    //16 bytes
    md_usb_set_tx_fifo_start_addr(USB, EPFIFOIdx);      //Start Address=0x8
    md_usb_set_tx_max_fifo_packet_size(USB, MD_USB_TXFIFO2_MAXPKTSIZE_16);  //Size=16, No Double-Packet Buffering
    md_usb_trigger_epx_tx_flush_fifo(USB);          //Flush FIFO

    EPFIFOIdx += ((CDC_INT_PKTSZ + 7) / 8);

    if (md_usb_is_enabled_tx_double_packet(USB))    //Double-Packet Buffering
        EPFIFOIdx += ((CDC_INT_PKTSZ + 7) / 8);

//Configure EP4 In
    md_usb_set_epx_index(USB, USBEP4);          //Endpoint 4
    md_usb_disable_epx_tx_isochronous(USB);         //Non ISO
    md_usb_set_epx_mode(USB, MD_USB_TXCSRH_MODE_TX);        //TX
    md_usb_set_tx_max_packet_size(USB, CDC_BULK_PKTSZ / 8);     //64 bytes
    md_usb_set_tx_fifo_start_addr(USB, EPFIFOIdx);  //Start Address=0xa
    md_usb_set_tx_max_fifo_packet_size(USB, MD_USB_TXFIFO2_MAXPKTSIZE_64);  //Size=64, No Double-Packet Buffering
    md_usb_trigger_epx_tx_flush_fifo(USB);          //Flush FIFO

    EPFIFOIdx += ((CDC_BULK_PKTSZ + 7) / 8);

    if (md_usb_is_enabled_tx_double_packet(USB))    //Double-Packet Buffering
        EPFIFOIdx += ((CDC_BULK_PKTSZ + 7) / 8);

//Configure EP4 Out
    md_usb_set_epx_index(USB, USBEP4);          //Endpoint 4
    md_usb_disable_epx_rx_isochronous(USB);         //Non ISO
    md_usb_set_epx_mode(USB, MD_USB_TXCSRH_MODE_RX);        //RX
    md_usb_set_rx_max_packet_size(USB, CDC_BULK_PKTSZ / 8);     //64 Bytes
    md_usb_set_rx_fifo_start_addr(USB, EPFIFOIdx);  //Start Address=0x10
    md_usb_set_rx_max_fifo_packet_size(USB, MD_USB_RXFIFO2_MAXPKTSIZE_64);  //Size=64, No Double-Packet Buffering
    md_usb_trigger_epx_rx_flush_fifo(USB);          //Flush FIFO

    EPFIFOIdx += ((CDC_BULK_PKTSZ + 7) / 8);

    if (md_usb_is_enabled_rx_double_packet(USB))    //Double-Packet Buffering
        EPFIFOIdx += ((CDC_BULK_PKTSZ + 7) / 8);

//Initial MUSB
    md_usb_enable_it_ep1_tx(USB);       //Tx Interrupt Enable for EP1
    md_usb_enable_it_ep2_tx(USB);       //Tx Interrupt Enable for EP2
    md_usb_enable_it_ep2_rx(USB);       //Rx Interrupt Enable for EP2
    md_usb_enable_it_ep3_tx(USB);       //Tx Interrupt Enable for EP3
    md_usb_enable_it_ep4_tx(USB);       //Tx Interrupt Enable for EP4
    md_usb_enable_it_ep4_rx(USB);       //Rx Interrupt Enable for EP4

    md_usb_set_device_address(USB, 0);          //Default Address

#if APP_DUAL_CDC_MSG
    printf("CDC EP Config Done\r\n");
#endif
}
/****
    * @brief    USB Application CDC Device Initialization
    * @param    pDev    : Device Instance
    * @retval   None
****/
void    dual_cdc_dev_init(_pUSB_FS_DEVICE pDev)
{
    dual_cdc_dev_ep0_config(pDev);
//Device Callback
    pDev->cb_app_dev_request = (void *)&cb_app_dev_dual_cdc;
}
/****
    * @brief    USB Application CDC Device Enumeration
    * @param    pDev        : Device Instance
    * @param    u8USBState  : Current Enumeration State
    * @retval   Updated Enumeration State
****/
eUSBDState  dual_cdc_dev_enum(_pUSB_FS_DEVICE pDev, eUSBDState u8USBState)
{
    u8USBState = usbd_dev_enum(pDev, pSetupReq, u8USBState);

    switch (u8USBState)
    {
        case USBD_STATE_IDLE:
            return (USBD_STATE_IDLE);

        case USBD_STATE_RESET:          //Reset
#if APP_DUAL_CDC_MSG
            printf("Reset\r\n");
#endif
            dual_cdc_dev_ep_config(pDev);
            return (USBD_STATE_IDLE);

        case USBD_STATE_RESUME:         //Resume
#if APP_DUAL_CDC_MSG
            printf("Resume\r\n");
#endif
            return (USBD_STATE_IDLE);

        case USBD_STATE_SUSPEND:        //Suspend
#if APP_DUAL_CDC_MSG
            printf("Suspend\r\n");
#endif
            return (USBD_STATE_IDLE);

        case USBD_STATE_SETUP:          //Setup
            if ((pSetupReq->bmRequestType & USB_REQ_TYPE_MASK) == USB_REQ_TYPE_CLASS)       //bmRequestType=Class
                u8USBState = (*pCDCClassRequest[pSetupReq->bRequest - CDC_SET_LINE_CODING])(pDev, pSetupReq);

            return (u8USBState);

        case USBD_STATE_CTRLOUT:        //Ctrl Out
            return (USBD_STATE_CTRLOUT);

        case USBD_STATE_CTRLIN:         //Ctrl In
            return (USBD_STATE_CTRLIN);

        case USBD_STATE_EP1IN:          //EP1, Int In
            return (USBD_STATE_EP1IN);

        case USBD_STATE_EP2IN:          //EP2, Bulk In
            return (USBD_STATE_EP2IN);

        case USBD_STATE_EP2OUT:         //EP2, Bulk Out
            return (USBD_STATE_EP2OUT);

        case USBD_STATE_EP3IN:          //EP3, Int In
            return (USBD_STATE_EP3IN);

        case USBD_STATE_EP4IN:          //EP4, Bulk In
            return (USBD_STATE_EP4IN);

        case USBD_STATE_EP4OUT:         //EP4, Bulk Out
            return (USBD_STATE_EP4OUT);

        default:
            return (USBD_STATE_IDLE);
    }

    return (USBD_STATE_IDLE);
}
/****
    * @brief    USB Application CDC Device Transaction
    * @param    pDev        : Device Instance
    * @param    u8USBState  : Current Transaction State
    * @retval   Updated Transaction State
****/
eUSBDState  dual_cdc_dev_transaction(_pUSB_FS_DEVICE pDev, eUSBDState u8USBState)
{
    uint32_t    DMACfgWord;
    uint32_t    DMADataEnd;
    uint16_t    u16CDCDLen;

#if APP_DUAL_CDC_MSG
//  printf("Transaction\r\n");
#endif

    if (u8USBState == USBD_STATE_EP2IN)         //CDC1 Data In
    {
        if (cdc1zdp)
        {
            cdc1zdp = 0;
            md_usb_set_epx_index(USB, CDC1_ENDPOINT_DATA);
            md_usb_enable_epx_tx_ready(USB);        //TxRdy
        }
        else
        {
            md_usb_set_epx_index(USB, CDC1_ENDPOINT_DATA);
            md_usb_clear_flag_epx_rx_ready(USB);    //Clear RxRdy
        }
    }

    if (u8USBState == USBD_STATE_EP4IN)         //CDC2 Data In
    {
        if (cdc_ring_used(&CDC2RingCtrl) == 0)
        {
            if (cdc2zdp)
            {
                cdc2zdp = 0;
                md_usb_set_epx_index(USB, CDC2_ENDPOINT_DATA);
                md_usb_enable_epx_tx_ready(USB);        //TxRdy
            }

            return (USBD_STATE_IDLE);
        }

        md_usb_set_epx_index(USB, CDC2_ENDPOINT_DATA);

//      printf("\tI - ");
        if (md_usb_is_enabled_epx_tx_ready(USB))
        {
//          printf("T\r\n");
            return (u8USBState);
        }

        u16CDCDLen = cdc_ring_wrap(&CDC2RingCtrl);

//      printf("%d\r\n",u16CDCDLen);
        if (u16CDCDLen >= CDC_BULK_PKTSZ)
        {
            u16CDCDLen = CDC_BULK_PKTSZ;
            cdc2zdp = 1;
        }
        else
            cdc2zdp = 0;

        DMACfgWord = DMA5CFG;
        DMACfgWord |= ((u16CDCDLen - 1) << CHANNEL_CFG_N_MINUS_1_POSS);
        md_dma_set_primary_channel_channel_configuration(DMA1, MD_DMA_CHANNEL5, DMACfgWord);

        DMADataEnd = ((uint32_t)CDC2RingCtrl.pRingBuf + CDC2RingCtrl.RingTail + md_dma_get_primary_channel_transfer_number(DMA1, MD_DMA_CHANNEL5));
        md_dma_set_primary_channel_source_data_end_address(DMA1, MD_DMA_CHANNEL5, DMADataEnd);

        CDC2RingCtrl.RingTail += u16CDCDLen;

        if (CDC2RingCtrl.RingTail >= CDC2RingCtrl.RingSize)
            CDC2RingCtrl.RingTail -= CDC2RingCtrl.RingSize;

        md_dma_enable_channel(DMA1, MD_DMA_CHANNEL5);   //DMA CH5 Enable
        md_dma_enable_soft_request(DMA1, MD_DMA_CHANNEL5);  //DMA CH5 SW Req
    }

    if (u8USBState == USBD_STATE_EP2OUT)        //CDC1 Data Out
    {
        md_usb_set_epx_index(USB, CDC1_ENDPOINT_DATA);

        if (md_usb_is_enabled_epx_tx_ready(USB))
            return (u8USBState);

//      printf("\t1 - ");
        u16CDCDLen = usbd_epx_out_available(pDev, CDC1_ENDPOINT_DATA);
//      printf("%d\r\n",u16CDCDLen);

        if (u16CDCDLen == 0)
        {
            md_usb_set_epx_index(USB, CDC1_ENDPOINT_DATA);
            md_usb_clear_flag_epx_rx_ready(USB);    //Clear RxRdy
            return (USBD_STATE_IDLE);
        }

        DMACfgWord = DMA3CFG;
        DMACfgWord |= ((u16CDCDLen - 1) << CHANNEL_CFG_N_MINUS_1_POSS);
        md_dma_set_primary_channel_channel_configuration(DMA1, MD_DMA_CHANNEL3, DMACfgWord);

        md_dma_enable_channel(DMA1, MD_DMA_CHANNEL3);   //DMA CH3 Enable
        md_dma_enable_soft_request(DMA1, MD_DMA_CHANNEL3);  //DMA CH3 SW Req

        if (u16CDCDLen == CDC_BULK_PKTSZ)
            cdc1zdp = 1;
    }

    if (u8USBState == USBD_STATE_EP4OUT)        //CDC2 Data Out
    {
//      printf("\tO - ");
        u16CDCDLen = usbd_epx_out_available(pDev, CDC2_ENDPOINT_DATA);
//      printf("%d\r\n", u16CDCDLen);

        if (u16CDCDLen == 0)
        {
            md_usb_set_epx_index(USB, CDC2_ENDPOINT_DATA);
            md_usb_clear_flag_epx_rx_ready(USB);    //Clear RxRdy
            return (USBD_STATE_IDLE);
        }

        DMACfgWord = DMA4CFG;
        DMACfgWord |= ((u16CDCDLen - 1) << 4);
        md_dma_set_primary_channel_channel_configuration(DMA1, MD_DMA_CHANNEL4, DMACfgWord);

        md_dma_enable_channel(DMA1, MD_DMA_CHANNEL4);   //DMA CH4 Enable
    }

    return (USBD_STATE_IDLE);
}

/******************* (C) COPYRIGHT Eastsoft Microelectronics END OF FILE****/
