/**********************************************************************************
 *
 * @file    app_dev_composite.c
 * @brief   c File
 *
 * @date    5 May 2022
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          5 May 2022      AE Team         the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */
#include "es32f0283.h"
#include "md_usb.h"
#include "usb_core.h"
#include "usb_enum.h"
#include "usb_dev_enum.h"
#include "usbcdc.h"
#include "usbhid.h"
#include "usb_dev_cdc.h"
#include "usb_dev_hid.h"
#include "app_composite_desc.h"
#include "app_dev_composite.h"
#include "cb_dev_composite.h"
#include <stdio.h>

#pragma pack(1)
#define APP_COMPOSITE_MSG       1

static  _pUSB_SETUP_REQUEST pSetupReq = (_pUSB_SETUP_REQUEST) &SetupReq;
static  uint8_t     CDCDataBuf[CDC_BULK_PKTSZ];

/****
    * @brief    USB Application Composite Device Endpoint 0 Configuration
    * @param    pDev    : Device Instance
    * @retval   None
****/
void    composite_dev_ep0_config(_pUSB_FS_DEVICE pDev)
{
//Configure EP0
    md_usb_set_epx_index(USB, USBEP0);          //Endpoint 0
    md_usb_set_tx_max_packet_size(USB, (USBD_CTRL_PKTSZ / 8));  //64 bytes
    md_usb_set_tx_fifo_start_addr(USB, 0x0);        //Start Address=0x0
    md_usb_set_tx_max_fifo_packet_size(USB, MD_USB_TXFIFO2_MAXPKTSIZE_64);  //Size=64
    md_usb_set_rx_max_packet_size(USB, (USBD_CTRL_PKTSZ / 8));  //64 bytes
    md_usb_set_rx_fifo_start_addr(USB, 0x0);        //Start Address=0x0
    md_usb_set_rx_max_fifo_packet_size(USB, MD_USB_RXFIFO2_MAXPKTSIZE_64);  //Size=64

    md_usb_enable_it_ep0(USB);      //Interrupt Enable for EP0
#if APP_COMPOSITE_MSG
    printf("Composite EP0 Config Done\r\n");
#endif
}
/****
    * @brief    USB Application Composite Device Endpoint Configuration
    * @param    pDev    : Device Instance
    * @retval   None
****/
void    composite_dev_ep_config(_pUSB_FS_DEVICE pDev)
{
    uint8_t     EPFIFOIdx;

    EPFIFOIdx = 0x8;
//Configure EP1 In
    md_usb_set_epx_index(USB, USBEP1);          //Endpoint 1
    md_usb_disable_epx_tx_isochronous(USB);         //Non ISO
    md_usb_set_epx_mode(USB, MD_USB_TXCSRH_MODE_TX);        //TX
    md_usb_set_tx_max_packet_size(USB, (CDC_INT_PKTSZ / 8));    //16 bytes
    md_usb_set_tx_fifo_start_addr(USB, EPFIFOIdx);      //Start Address=0x8
    md_usb_set_tx_max_fifo_packet_size(USB, MD_USB_TXFIFO2_MAXPKTSIZE_16);  //Size=16, No Double-Packet Buffering
    md_usb_trigger_epx_tx_flush_fifo(USB);          //Flush FIFO

    EPFIFOIdx += ((CDC_INT_PKTSZ + 7) / 8);

    if (md_usb_is_enabled_tx_double_packet(USB))    //Double-Packet Buffering
        EPFIFOIdx += ((CDC_INT_PKTSZ + 7) / 8);

//Configure EP2 In
    md_usb_set_epx_index(USB, USBEP2);          //Endpoint 2
    md_usb_disable_epx_tx_isochronous(USB);         //Non ISO
    md_usb_set_epx_mode(USB, MD_USB_TXCSRH_MODE_TX);        //TX
    md_usb_set_tx_max_packet_size(USB, (CDC_BULK_PKTSZ / 8));   //64 bytes
    md_usb_set_tx_fifo_start_addr(USB, EPFIFOIdx);  //Start Address=0xa
    md_usb_set_tx_max_fifo_packet_size(USB, MD_USB_TXFIFO2_MAXPKTSIZE_64);  //Size=64, No Double-Packet Buffering
    md_usb_trigger_epx_tx_flush_fifo(USB);          //Flush FIFO

    EPFIFOIdx += ((CDC_BULK_PKTSZ + 7) / 8);

    if (md_usb_is_enabled_tx_double_packet(USB))    //Double-Packet Buffering
        EPFIFOIdx += ((CDC_BULK_PKTSZ + 7) / 8);

//Configure EP2 Out
    md_usb_set_epx_index(USB, USBEP2);          //Endpoint 2
    md_usb_disable_epx_rx_isochronous(USB);         //Non ISO
    md_usb_set_epx_mode(USB, MD_USB_TXCSRH_MODE_RX);        //RX
    md_usb_set_rx_max_packet_size(USB, (CDC_BULK_PKTSZ / 8));   //64 Bytes
    md_usb_set_rx_fifo_start_addr(USB, EPFIFOIdx);  //Start Address=0x10
    md_usb_set_rx_max_fifo_packet_size(USB, MD_USB_RXFIFO2_MAXPKTSIZE_64);  //Size=64, No Double-Packet Buffering
    md_usb_trigger_epx_rx_flush_fifo(USB);          //Flush FIFO

    EPFIFOIdx += ((CDC_BULK_PKTSZ + 7) / 8);

    if (md_usb_is_enabled_rx_double_packet(USB))    //Double-Packet Buffering
        EPFIFOIdx += ((CDC_BULK_PKTSZ + 7) / 8);

//Configure EP3 In
    md_usb_set_epx_index(USB, USBEP3);          //Endpoint 3
    md_usb_disable_epx_tx_isochronous(USB);         //Non ISO
    md_usb_set_epx_mode(USB, MD_USB_TXCSRH_MODE_TX);        //TX
    md_usb_set_tx_max_packet_size(USB, (HID_INT_PKTSZ / 8));    //64 bytes
    md_usb_set_tx_fifo_start_addr(USB, EPFIFOIdx);  //Start Address=0xa
    md_usb_set_tx_max_fifo_packet_size(USB, MD_USB_TXFIFO2_MAXPKTSIZE_64);  //Size=64, No Double-Packet Buffering
    md_usb_trigger_epx_tx_flush_fifo(USB);          //Flush FIFO

    EPFIFOIdx += ((HID_INT_PKTSZ + 7) / 8);

    if (md_usb_is_enabled_tx_double_packet(USB))    //Double-Packet Buffering
        EPFIFOIdx += ((HID_INT_PKTSZ + 7) / 8);

//Configure EP3 Out
    md_usb_set_epx_index(USB, USBEP3);          //Endpoint 3
    md_usb_disable_epx_rx_isochronous(USB);         //Non ISO
    md_usb_set_epx_mode(USB, MD_USB_TXCSRH_MODE_RX);        //RX
    md_usb_set_rx_max_packet_size(USB, (HID_INT_PKTSZ / 8));    //64 Bytes
    md_usb_set_rx_fifo_start_addr(USB, EPFIFOIdx);  //Start Address=0x10
    md_usb_set_rx_max_fifo_packet_size(USB, MD_USB_RXFIFO2_MAXPKTSIZE_64);  //Size=64, No Double-Packet Buffering
    md_usb_trigger_epx_rx_flush_fifo(USB);          //Flush FIFO

    EPFIFOIdx += ((HID_INT_PKTSZ + 7) / 8);

    if (md_usb_is_enabled_rx_double_packet(USB))    //Double-Packet Buffering
        EPFIFOIdx += ((HID_INT_PKTSZ + 7) / 8);

//Initial MUSB
    md_usb_enable_it_ep1_tx(USB);       //Tx Interrupt Enable for EP1
    md_usb_enable_it_ep2_tx(USB);       //Tx Interrupt Enable for EP2
    md_usb_enable_it_ep3_tx(USB);       //Tx Interrupt Enable for EP3
    md_usb_enable_it_ep2_rx(USB);       //Rx Interrupt Enable for EP2
    md_usb_enable_it_ep3_rx(USB);       //Rx Interrupt Enable for EP3

    md_usb_set_device_address(USB, 0);          //Default Address

#if APP_COMPOSITE_MSG
    printf("Composite EP Config Done\r\n");
#endif
}
/****
    * @brief    USB Application Composite Device Initialization
    * @param    pDev    : Device Instance
    * @retval   None
****/
void    composite_dev_init(_pUSB_FS_DEVICE pDev)
{
    composite_dev_ep0_config(pDev);
//Device Callback
    pDev->cb_app_dev_request = (void *)&cb_app_dev_composite;
}
/****
    * @brief    USB Application Composite Device Enumeration
    * @param    pDev        : Device Instance
    * @param    USBState    : Current Enumeration State
    * @retval   Updated Enumeration State
****/
uint8_t composite_dev_enum(_pUSB_FS_DEVICE pDev, eUSBDState USBState)
{
    USBState = usbd_dev_enum(pDev, pSetupReq, USBState);

    switch (USBState)
    {
        case USBD_STATE_IDLE:
            return (USBD_STATE_IDLE);

        case USBD_STATE_RESET:          //Reset
#if APP_COMPOSITE_MSG
            printf("Reset\r\n");
#endif
            composite_dev_ep_config(pDev);
            return (USBD_STATE_IDLE);

        case USBD_STATE_RESUME:         //Resume
#if APP_COMPOSITE_MSG
            printf("Resume\r\n");
#endif
            return (USBD_STATE_IDLE);

        case USBD_STATE_SUSPEND:        //Suspend
#if APP_COMPOSITE_MSG
            printf("Suspend\r\n");
#endif
            return (USBD_STATE_IDLE);

        case USBD_STATE_SETUP:          //Setup
            if ((pSetupReq->bmRequestType & USB_REQ_TYPE_MASK) == USB_REQ_TYPE_CLASS)       //bmRequestType=Class
            {
                if ((pSetupReq->wIndex & 0xff) == CDC_INTERFACE_CONTROL)    //interface 0, CDC
                    USBState = (*pCDCClassRequest[pSetupReq->bRequest - CDC_SET_LINE_CODING])(pDev, pSetupReq);

                if ((pSetupReq->wIndex & 0xff) == HID_INTERFACE)            //interface 2, HID
                    USBState = (*pHIDClassRequest[pSetupReq->bRequest])(pDev, pSetupReq);
            }

            return (USBState);

        case USBD_STATE_CTRLOUT:        //Ctrl Out
            return (USBD_STATE_CTRLOUT);

        case USBD_STATE_CTRLIN:         //Ctrl In
            return (USBD_STATE_CTRLIN);

        case USBD_STATE_EP1IN:          //EP1, Int In
            return (USBD_STATE_EP1IN);

        case USBD_STATE_EP2IN:          //EP2, Bulk In
            return (USBD_STATE_EP2IN);

        case USBD_STATE_EP2OUT:         //EP2, Bulk Out
            return (USBD_STATE_EP2OUT);

        default:
            return (USBD_STATE_IDLE);
    }

    return (USBD_STATE_IDLE);
}
/****
    * @brief    USB Application Composite Device Transaction
    * @param    pDev        : Device Instance
    * @param    USBState    : Current Transaction State
    * @retval   Updated Transaction State
****/
uint8_t composite_dev_transaction(_pUSB_FS_DEVICE pDev, eUSBDState USBState)
{
    uint16_t    DLen, ii;

#if APP_COMPOSITE_MSG
//  printf("Transaction\r\n");
#endif

    if (USBState != USBD_STATE_EP2OUT)
        return (USBD_STATE_IDLE);

    DLen = usbd_epx_out_available(pDev, USBEP2);
    usbd_handle_epx_out(pDev, USBEP2, DLen, CDCDataBuf);

    for (ii = 0; ii < DLen; ii++)
    {
        printf("0x%x ", CDCDataBuf[ii]);

        if ((ii & 0xf) == 0xf)
            printf("\r\n");
    }

    printf("\r\n");
    usbd_handle_epx_in(pDev, USBEP2, DLen, CDCDataBuf);
    return (USBD_STATE_IDLE);
}

/******************* (C) COPYRIGHT Eastsoft Microelectronics END OF FILE****/

