/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    23 Nov 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          23 Nov 2021     Ginger          the first version
 *          23 Dec 2021     Ginger          Modify the example for PDS board
 *          21 Mar 2022     AE Team         Modify MD Driver
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Define to prevent recursive inclusion -------------------------------------*/
#define MAIN_GLOBALS

/* Includes -------------------------------------------------------------------*/
#include <stdio.h>
#include "main.h"


/** @addtogroup Projects_Examples_MD
  * @{
  */

/* Private types --------------------------------------------------------------*/
md_rcu_init_typedef rcu_initStruct =    /**< RCU init structure */
{
    MD_RCU_MPRE_MCO_DIV1,
    MD_RCU_MSW_MCO_DISABLE,
    MD_RCU_PLLSRC_HRC,
    MD_RCU_PLLCLK_PASS,
    MD_RCU_PPRE_HCLK_DIV_1,
    MD_RCU_HPRE_SYSCLK_DIV_1,
    MD_RCU_SW_SYSCLK_HRC,
    (RCU_CON_HRCON),
};

md_gpio_inittypedef GPIOB_PIN6_Init =    /**< Uart Tx init structure */
{
    MD_GPIO_PIN_6,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF2
};

md_gpio_inittypedef GPIOB_PIN7_Init =    /**< Uart Rx init structure */
{
    MD_GPIO_PIN_7,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF2
};

md_uart_init_typedef uart_initStruct =    /**< UART init structure */
{
    MD_UART_BAUDRATE_115200,
    MD_UART_LCON_LSB_FIRST,
    MD_UART_LCON_PS_NONE,
    MD_UART_LCON_STOP_1,
    MD_UART_LCON_DLS_8,
};

md_adc_initial adc_initStruct =    /**< ADC init structure */
{
    MD_ADC_CFG_ALIGN_RIGHT,     //Data alignment
    MD_ADC_CFG_RSEL_12BIT,      //Data resolution
    MD_ADC_MODE_NCHS,           //Regular or Injected
    MD_ADC_CFG_CM_SINGLE,       //Single mode
    MD_ADC_NCHS1_NSL_1CON,      //sample count
    MD_ADC_NCHS1_NSL_1CON,      //ADC prescale
};

/* Private define -------------------------------------------------------------*/
/* Private macro --------------------------------------------------------------*/
/* Private variable -----------------------------------------------------------*/
/* Private function prototypes ------------------------------------------------*/
void Iomux(void);
void SysPeriInit(void);
void bsp_adc_init(void);
void reset_tick(void);
uint32_t get_tick(void);
/* Private functions ----------------------------------------------------------*/
/**
  * @brief  main.
  * @param  None
  * @retval None
  */
int main(void)
{

    bool key_flag = 0;
    int ptt_init_val = 0;
    int ptt_current_val = 0;
    __disable_irq();
    md_rcu_pll0_init(RCU, &rcu_initStruct);
    md_rcu_sys_init(RCU, &rcu_initStruct);
    SysPeriInit();
    Iomux();
    md_uart_init(UART1, &uart_initStruct);
    md_tick_init(MD_SYSTICK_CLKSRC_HCLK);
    __enable_irq();

    /* Initialize peripheral */
    bsp_key_init();
    bsp_led_init();
    bsp_adc_init();

    md_adc_set_start_normal(ADC, MD_ADC_CON_NSTART_START_REGULAR);

    while (md_adc_get_start_normal(ADC) == MD_ADC_CON_NSTART_START_REGULAR)
    {
        while (md_adc_is_active_flag_normal_status(ADC));

        ptt_init_val = md_adc_get_data(ADC);
    }

    /* Test UART*/
    printf("Rotate the potentiometer to change the LED\r\n");



    while (1)
    {
        md_adc_set_start_normal(ADC, MD_ADC_CON_NSTART_START_REGULAR);

        while (md_adc_get_start_normal(ADC) == MD_ADC_CON_NSTART_START_REGULAR)
        {
            while (md_adc_is_active_flag_normal_status(ADC));

            ptt_current_val = md_adc_get_data(ADC);
        }

        if (ptt_current_val >= 0 && ptt_current_val <= 767)
        {
            bsp_Led1_Off();
            bsp_Led2_On();
            bsp_Led3_Off();
            bsp_Led4_Off();
        }
        else if (ptt_current_val >= 768 && ptt_current_val <= 1535)
        {
            bsp_Led1_On();
            bsp_Led2_Off();
            bsp_Led3_Off();
            bsp_Led4_Off();
        }
        else if (ptt_current_val >= 1536 && ptt_current_val <= 2303)
        {
            bsp_Led1_Off();
            bsp_Led2_Off();
            bsp_Led3_On();
            bsp_Led4_Off();
        }
        else if (ptt_current_val >= 2304 && ptt_current_val <= 3071)
        {
            bsp_Led1_Off();
            bsp_Led2_Off();
            bsp_Led3_Off();
            bsp_Led4_On();
        }

        if (ptt_current_val - ptt_init_val >= 50 || ptt_init_val - ptt_current_val >= 50)
        {
            ptt_init_val = ptt_current_val;
            printf("Potentiometer is rotated\r\n");
        }

        if (bsp_key_mid_gpio_get() && key_flag == 0)
        {
            key_flag = 1;
            printf("middle key is pressed\r\n");
        }

        if (bsp_key_left_gpio_get() && key_flag == 0)
        {
            key_flag = 1;
            printf("left key is pressed\r\n");
        }

        if (bsp_key_up_gpio_get() && key_flag == 0)
        {
            key_flag = 1;
            printf("up key is pressed\r\n");
        }

        if (bsp_key_right_gpio_get() && key_flag == 0)
        {
            key_flag = 1;
            printf("right key is pressed\r\n");
        }

        if (bsp_key_down_gpio_get() && key_flag == 0)
        {
            key_flag = 1;
            printf("down key is pressed\r\n");
        }

        if (!bsp_key_down_gpio_get() && !bsp_key_right_gpio_get() && !bsp_key_up_gpio_get() && !bsp_key_left_gpio_get() && !bsp_key_mid_gpio_get())
        {
            key_flag = 0;
        }


    }
}

/**
  * @brief  Peripheral Init
  * @note   Enable peripheral clock
  * @param  None
  * @retval None
  */
void SysPeriInit(void)
{
    md_rcu_enable_gpiod(RCU);
    md_rcu_enable_gpioc(RCU);
    md_rcu_enable_gpiob(RCU);
    md_rcu_enable_gpioa(RCU);

    md_rcu_enable_gp32c4t1(RCU);
    md_rcu_enable_adc(RCU);
    md_rcu_enable_uart1(RCU);
}

/**
  * @brief  Configure I/O Multiplexer
  * @note   PB6: UART1_TX.
  *         PB7: UART1_RX (Internal weak pull-up).
  * @param  None
  * @retval None
  */
void Iomux(void)
{
    md_gpio_init(GPIOB, &GPIOB_PIN6_Init);
    md_gpio_init(GPIOB, &GPIOB_PIN7_Init);
}

void bsp_adc_init(void)
{
    uint32_t sample_seq_channel[] =
    {
        MD_ADC_NCHS1_NS1_CH15
    };
    md_adc_calibration(ADC, &adc_initStruct);
    md_adc_set_sampletime_channel_15(ADC, 0xFF);

    md_adc_init(ADC, &adc_initStruct);

    while ((ADC->RIF & 0x1) == 0);

    md_adc_sequence_conversion(ADC, &adc_initStruct, sample_seq_channel);
}

void reset_tick()
{
    md_tick_set_cvr_current(TICK, 0xffffff);
}

uint32_t get_tick()
{
    //uint:s
    return (0xffffff - md_tick_get_cvr_current(TICK)) / 4000000;
}

uint32_t get_tick(void);

/**
  * @brief  Uart sendchar.
  * @param  arg: char to be sent.
  * @retval data to be sent.
  */
uint8_t  sendchar(uint8_t ch)
{
    while (!(UART1->STAT & (UART_STAT_TFEMPTY)));  // Tx FIFO empty

    UART1->TXDATA = ch;            // Sent byte
    return (ch);
}

/**
  * @} Projects_Examples_MD
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/


