/**********************************************************************************
 *
 * @file    spi_flash.c
 * @brief   SPI Flash file for DEMO
 *
 * @date    9 Jan. 2023
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          9 Jan. 2023     Lisq            the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */

#include "spi_flash.h"

/* Private Macros ------------------------------------------------------------ */
#define FLASH_CS_SET()  md_gpio_set_pin_high(GPIOA, MD_GPIO_PIN_15);
#define FLASH_CS_CLR()  md_gpio_set_pin_low(GPIOA, MD_GPIO_PIN_15);

/* Private Variables --------------------------------------------------------- */

static md_spi_handle_t s_spi;

/* Public Variables ---------------------------------------------------------- */

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Function ---------------------------------------------------------- */
/**
  * @brief  wait until flash unbusy.
  * @retval Status, see @ref md_status_t.
  */
md_status_t flash_wait_unbusy(void)
{
    uint16_t cnt = 5000, temp;
    uint8_t status;

    FLASH_CS_CLR(); /*Choose lower, the selected Flash*/

    md_spi_set_data_reg(s_spi.SPI, (uint8_t)FLASH_STATUS);

    while ((md_spi_is_active_flag_txe(s_spi.SPI) == 0) && (--cnt));

    cnt = 5000;

    while ((!((md_spi_is_active_flag_rxne(s_spi.SPI)))) && (--cnt));

    temp = md_spi_get_data_reg(s_spi.SPI);
    UNUSED(temp);

    do
    {
        cnt = 5000;
        md_spi_set_data_reg(s_spi.SPI, 0xFF);

        while ((md_spi_is_active_flag_txe(s_spi.SPI) == 0) && (--cnt));

        cnt = 5000;

        while ((!((md_spi_is_active_flag_rxne(s_spi.SPI)))) && (--cnt));

        status = md_spi_get_data_reg(s_spi.SPI);
    }
    while (status & 0x01);

    FLASH_CS_SET();

    return MD_OK;
}

/**
  * @brief  Read flash id in blocking mode.
  * @retval flash id.
  */
uint32_t flash_read_id(void)
{
    uint16_t cnt = 5000, temp;

    uint8_t i;

    uint8_t flash_id[4] = {0U};

    flash_id[0] = FLASH_ID;

    FLASH_CS_CLR(); /*Choose lower, the selected Flash*/

    for (i = 0; i < sizeof(flash_id); i++)
    {
        md_spi_set_data_reg(s_spi.SPI, flash_id[i]);

        while ((md_spi_is_active_flag_txe(s_spi.SPI) == 0) && (--cnt));

        cnt = 5000;

        while ((!((md_spi_is_active_flag_rxne(s_spi.SPI)))) && (--cnt));

        temp = md_spi_get_data_reg(s_spi.SPI);
        UNUSED(temp);

    }

    for (i = 0; i < 3; i++)
    {
        cnt = 5000;
        md_spi_set_data_reg(s_spi.SPI, 0xFF);

        while ((md_spi_is_active_flag_txe(s_spi.SPI) == 0) && (--cnt));

        cnt = 5000;

        while ((!((md_spi_is_active_flag_rxne(s_spi.SPI)))) && (--cnt));

        flash_id[i] = md_spi_get_data_reg(s_spi.SPI);
    }

    FLASH_CS_SET();

    return ((flash_id[0] << 16) | (flash_id[1] << 8) | (flash_id[2]));  /*Manufacturer ID flash_id [0] and device ID flash_id [1]*/
}

/**
  * @brief  Initializate spi flash pin
  * @retval None.
  */
static void spi_pin_init(void)
{
    md_gpio_inittypedef gpio_init;

    /* SPI CS pin PA15 */
    gpio_init.Pin = MD_GPIO_PIN_15;
    gpio_init.Mode = MD_GPIO_MODE_OUTPUT;
    gpio_init.OutputType = MD_GPIO_OUTPUT_PUSHPULL;
    gpio_init.Pull = MD_GPIO_PULL_UP;
    gpio_init.OutDrive = MD_GPIO_DRIVING_8MA;
    gpio_init.Function = MD_GPIO_AF0;
    md_gpio_init(GPIOA, &gpio_init);

    /* SPI CLK pin PB3 */
    gpio_init.Pin = MD_GPIO_PIN_3;
    gpio_init.Mode = MD_GPIO_MODE_FUNCTION;
    gpio_init.OutputType = MD_GPIO_OUTPUT_PUSHPULL;
    gpio_init.Pull = MD_GPIO_PULL_UP;
    gpio_init.OutDrive = MD_GPIO_DRIVING_8MA;
    gpio_init.Function = MD_GPIO_AF0;
    md_gpio_init(GPIOB, &gpio_init);

    /* SPO MISO pin PB4 */
    gpio_init.Pin = MD_GPIO_PIN_4;
    gpio_init.Mode = MD_GPIO_MODE_FUNCTION;
    gpio_init.OutputType = MD_GPIO_OUTPUT_PUSHPULL;
    gpio_init.Pull = MD_GPIO_PULL_UP;
    gpio_init.OutDrive = MD_GPIO_DRIVING_8MA;
    gpio_init.Function = MD_GPIO_AF0;
    md_gpio_init(GPIOB, &gpio_init);

    /* SPI MOSI pin PB4 */
    gpio_init.Pin = MD_GPIO_PIN_5;
    gpio_init.Mode = MD_GPIO_MODE_FUNCTION;
    gpio_init.OutputType = MD_GPIO_OUTPUT_PUSHPULL;
    gpio_init.Pull = MD_GPIO_PULL_UP;
    gpio_init.OutDrive = MD_GPIO_DRIVING_8MA;
    gpio_init.Function = MD_GPIO_AF0;
    md_gpio_init(GPIOB, &gpio_init);

    md_gpio_set_pin_high(GPIOA, MD_GPIO_PIN_15);
}

/**
  * @brief spi function
  * @retval None.
  */
static void mcu_spi_init(void)
{
    spi_pin_init();

    s_spi.SPI = SPI1;

    md_spi_struct_init(&s_spi.init);
    s_spi.init.Mode = MD_SPI_MODE_MASTER;
    s_spi.init.ClockPhase = MD_SPI_PHASE_1EDGE;
    s_spi.init.ClockPolarity = MD_SPI_POLARITY_LOW;
    s_spi.init.BaudRate = MD_SPI_BAUDRATEPRESCALER_DIV4;
    s_spi.init.BitOrder = MD_SPI_MSB_FIRST;
    s_spi.init.TransferDirection = MD_SPI_FULL_DUPLEX;
    s_spi.init.DataWidth = MD_SPI_FRAME_FORMAT_8BIT;
    s_spi.init.NSS = MD_SPI_NSS_SOFT;
    s_spi.init.CRCCalculation = MD_SPI_CRCCALCULATION_DISABLE;
    s_spi.init.CRCPoly = 0x7;

    md_spi_init(s_spi.SPI, &s_spi.init);                    /* According to the parameter initialization SPI peripherals */
}

/**
  * @brief  flash write enable function.
  * @retval None.
  */
static uint8_t flash_write_enable(void)
{
    uint16_t cnt = 5000, temp;

    /* wait until flash unbusy */
    flash_wait_unbusy();

    /* enable the selected Flash */
    FLASH_CS_CLR();

    /* send write enable command */
    md_spi_send_byte(s_spi.SPI, FLASH_WRITE_ENABLE);

    while ((md_spi_is_active_flag_txe(s_spi.SPI) == 0) && (--cnt));

    cnt = 5000;

    while ((!((md_spi_is_active_flag_rxne(s_spi.SPI)))) && (--cnt));

    temp = md_spi_get_data_reg(s_spi.SPI);
    UNUSED(temp);

    /* release Flash */
    FLASH_CS_SET();

    /* wait until flash unbusy */
    flash_wait_unbusy();

    return 0;
}

/**
  * @brief  low-level flash sector erase function.
  * @param  start_sec: Specified start sector to be erase.
  * @param  end_sec: Specified end sector to be erase.
  * @retval Status.
  */
uint8_t ll_flash_sector_erase(uint32_t start_sec, uint32_t end_sec)
{
    uint8_t  cmd_buf[4];
    uint8_t  i = 0U;
    uint16_t cnt = 5000, temp;
    uint32_t sec_addr;

    /*Flash sector erase command*/
    cmd_buf[0] = FLASH_SEC_ERASE;

    /*Erase Flash sectors */
    for (; start_sec <= end_sec; start_sec++)
    {
        /* group command and address */
        sec_addr = start_sec * SEC_SIZE;
        cmd_buf[1] = (sec_addr >> 16) & 0xFF;   /*24 bit Flash address*/
        cmd_buf[2] = (sec_addr >> 8) & 0xFF;
        cmd_buf[3] = (sec_addr) & 0xFF;

        /* enable flash write */
        flash_write_enable();

        /* erase current sector */
        /* enable the selected Flash */
        FLASH_CS_CLR();

        /*Send the sector erase instructions and Flash address three bytes*/
        for (i = 0; i < sizeof(cmd_buf); i++)
        {
            cnt = 5000;
            md_spi_send_byte(s_spi.SPI, cmd_buf[i]);

            while ((md_spi_is_active_flag_txe(s_spi.SPI) == 0) && (--cnt));

            cnt = 5000;

            while ((!((md_spi_is_active_flag_rxne(s_spi.SPI)))) && (--cnt));

            temp = md_spi_get_data_reg(s_spi.SPI);
            UNUSED(temp);
        }

        /* release Flash */
        FLASH_CS_SET();

        /* wait until flash not in write progress */
        flash_wait_unbusy();
    }

    return 0;
}

/**
  * @brief  low-level flash sector write operation function.
  * @param  src_buf: pointer to the expected written data.
  * @param  sec: start sector to write.
  * @param  sec_num: amount of sector.
  * @retval Status, see @ref ald_status_t.
  */
uint8_t ll_flash_sector_write(const uint8_t *src_buf, uint32_t sec, uint32_t sec_num)
{
    uint8_t  cmd_buf[4];
    uint16_t cnt = 5000, temp;
    uint32_t  i, j, k;
    uint32_t page_addr, sec_addr;

    if (src_buf == NULL)
        return 1;

    /*Flash page program command*/
    cmd_buf[0] = FLASH_PAGE_PROGRAM;

    /*Erase Flash sectors */
    for (i = 0; i < sec_num; i++)
    {
        /* calculate current sector start address */
        sec_addr = sec * SEC_SIZE;

        for (j = 0; j < PAGE_PER_SEC; j++)
        {
            /* group command and address */
            page_addr = sec_addr + PAGE_SIZE * j;
            cmd_buf[1] = (page_addr >> 16) & 0xff;
            cmd_buf[2] = (page_addr >> 8) & 0xff;
            cmd_buf[3] = (page_addr) & 0xff;

            /* enable flash write */
            flash_write_enable();

            /* write current page */
            /* enable the selected Flash */
            FLASH_CS_CLR();

            /*Send the sector erase instructions and Flash address three bytes*/
            for (k = 0; k < sizeof(cmd_buf); k++)
            {
                cnt = 5000;
                md_spi_send_byte(s_spi.SPI, cmd_buf[k]);

                while ((md_spi_is_active_flag_txe(s_spi.SPI) == 0) && (--cnt));

                cnt = 5000;

                while ((!((md_spi_is_active_flag_rxne(s_spi.SPI)))) && (--cnt));

                temp = md_spi_get_data_reg(s_spi.SPI);
                UNUSED(temp);
            }

            /* write the data to the Flash*/
            for (k = 0; k < PAGE_SIZE; k++)
            {
                cnt = 5000;
                md_spi_send_byte(s_spi.SPI, src_buf[i * SEC_SIZE + j * PAGE_SIZE + k]);

                while ((md_spi_is_active_flag_txe(s_spi.SPI) == 0) && (--cnt));

                cnt = 5000;

                while ((!((md_spi_is_active_flag_rxne(s_spi.SPI)))) && (--cnt));

                temp = md_spi_get_data_reg(s_spi.SPI);
                UNUSED(temp);
            }

            /* release Flash */
            FLASH_CS_SET();

            /* wait until flash not in write progress */
            flash_wait_unbusy();
        }
    }

    return 0;
}

/**
  * @brief  low-level flash sector read operation function.
  * @param  str_buf: pointer to the first item of the byte array to store read data.
  * @param  sec: start sector.
  * @param  sec_num: amount of sector to be read.
  * @retval Status, see @ref ald_status_t.
  */
uint8_t ll_flash_sector_read(uint8_t *str_buf, uint32_t sec, uint32_t sec_num)
{

    uint8_t  cmd_buf[4];
    uint16_t cnt = 5000, temp;
    uint32_t  i, j;
    uint32_t sec_addr;

    if (str_buf == NULL)
        return 1;

    /*Flash read command*/
    cmd_buf[0] = FLASH_READ;
    /* group command and address */
    sec_addr = sec * SEC_SIZE;
    cmd_buf[1] = (sec_addr >> 16) & 0xFF;   /*24 bit Flash address*/
    cmd_buf[2] = (sec_addr >> 8) & 0xFF;
    cmd_buf[3] = (sec_addr) & 0xFF;

    /* enable the selected Flash */
    FLASH_CS_CLR();

    /* Send the read instructions and Flash address */
    for (i = 0; i < sizeof(cmd_buf); i++)
    {
        cnt = 5000;
        md_spi_send_byte(s_spi.SPI, cmd_buf[i]);

        while ((md_spi_is_active_flag_txe(s_spi.SPI) == 0) && (--cnt));

        cnt = 5000;

        while ((!((md_spi_is_active_flag_rxne(s_spi.SPI)))) && (--cnt));

        temp = md_spi_get_data_reg(s_spi.SPI);
        UNUSED(temp);
    }

    /* read sectors data */
    for (i = 0; i < sec_num; i++)
    {
        /* read data cycle */
        for (j = 0; j < SEC_SIZE; j++)
        {
            cnt = 5000;
            md_spi_set_data_reg(s_spi.SPI, 0xFF);

            while ((md_spi_is_active_flag_txe(s_spi.SPI) == 0) && (--cnt));

            cnt = 5000;

            while ((!((md_spi_is_active_flag_rxne(s_spi.SPI)))) && (--cnt));

            str_buf[i * SEC_SIZE + j] = md_spi_recv_byte(s_spi.SPI);
        }
    }

    /* release Flash */
    FLASH_CS_SET();

    return 0;
}

/**
  * @brief spi flash init function
  * @retval None.
  */
uint8_t ll_spi_flash_init(void)
{
    mcu_spi_init();

    return 0;
}
/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
