/**
  *********************************************************************************
  *
  * @file    .c
  * @brief  Source file
  *
  * @version V1.0
  * @date    26 Jun 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          26 Jun 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

/* Includes ------------------------------------------------------------------ */
#include <ctype.h>
#include <errno.h>
#include <inttypes.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include "pdfgen.h"

/* Private Macros ------------------------------------------------------------ */
#ifndef M_SQRT2
    #define M_SQRT2 1.41421356237309504880
#endif/*M_SQRT2*/

#define PDF_RGB_R(c) (float)((((c) >> 16) & 0xff) / 255.0)
#define PDF_RGB_G(c) (float)((((c) >> 8) & 0xff) / 255.0)
#define PDF_RGB_B(c) (float)((((c) >> 0) & 0xff) / 255.0)
#define PDF_IS_TRANSPARENT(c) (((c) >> 24) == 0xff)

enum
{
    OBJ_none, /* skipped */
    OBJ_info,
    OBJ_stream,
    OBJ_font,
    OBJ_page,
    OBJ_bookmark,
    OBJ_outline,
    OBJ_catalog,
    OBJ_pages,
    OBJ_image,
    OBJ_link,

    OBJ_count,
};

#define ES_MAX_PDFGEN_OBJ_NUM  (128)
#define ES_MAX_PDFGEN_PAGE_NUM  (16)
#define ES_MAX_PDFGEN_FONT_NUM  (8)

#define ES_DATA_SOURSE_IN_ROM(x)  ((((uint32_t)(x)) < 0x80000)? 0 : 0xFFFFFFFF)

/* Private Variables --------------------------------------------------------- */

static FIL *s_pdf_save_fp = NULL;                   /*Opened files*/
static uint16_t s_es_pdf_page_index_now = 0U;      /*Current page N, page 0 is invalid*/
static uint16_t s_es_pdf_font_now = 0U;            /*Currently the N th font, the 0 th font is invalid*/
static uint32_t s_es_pdf_out_obj_font_num = 0U;    /*Number of existing fonts*/
static int s_es_pdf_out_obj_num = 0;               /*Number of PDF objects output*/
static int s_es_pdf_out_page_num = 0;              /*Existing PDF pages*/
static uint8_t s_es_pdf_have_image = 0;            /*Is there an image in the PDF*/
static uint32_t s_es_pdf_out_obj_location_list[ES_MAX_PDFGEN_OBJ_NUM];     /*The file address of the PDF object*/
static uint16_t s_es_pdf_out_obj_in_pagen[ES_MAX_PDFGEN_OBJ_NUM];          /*PDF object on page N, page 0=not on page*/
static uint8_t s_es_pdf_out_obj_type_list[ES_MAX_PDFGEN_OBJ_NUM];          /*The type of PDF object*/
static uint8_t *s_es_pdf_out_obj_font_list[ES_MAX_PDFGEN_FONT_NUM];         /*The name of the font, stored in ROM by default*/
static float  s_es_pdf_page_width = 500;           /*The width of the PDF page*/
static float  s_es_pdf_page_height = 1000;         /*The length of the PDF page*/

/* Public Variables ---------------------------------------------------------- */
uint32_t g_pdf_out_bytes = 0U;            /*The number of bytes that have been output from the opened file*/

/* Private Constants --------------------------------------------------------- */
/* Private function prototypes ----------------------------------------------- */
/* Private Function ---------------------------------------------------------- */

static inline void es_pdf_obj_out_file_start(uint8_t obj_type, uint8_t obj_in_page)
{
    s_es_pdf_out_obj_location_list[s_es_pdf_out_obj_num] = g_pdf_out_bytes;
    s_es_pdf_out_obj_in_pagen[s_es_pdf_out_obj_num] = (obj_in_page) ? (s_es_pdf_page_index_now) : (0);
    s_es_pdf_out_obj_type_list[s_es_pdf_out_obj_num] = obj_type;

    es_file_printf(s_pdf_save_fp, "%d 0 obj\r\n", (s_es_pdf_out_obj_num + 1));
}

static inline void es_pdf_obj_out_file_end(void)
{
    es_file_printf(s_pdf_save_fp, "endobj\r\n");
    s_es_pdf_out_obj_num++;
}

static inline void es_pdf_stream_out_file_start(uint32_t len)
{
    es_file_printf(s_pdf_save_fp, "<< /Length %zu >>stream\r\n", len);
}

static inline void es_pdf_stream_out_file_end(void)
{
    es_file_printf(s_pdf_save_fp, "\r\nendstream\r\n");
}

void es_pdf_data_sourse_err()
{
    printf("data not in rom\r\n");
}

int es_vsnprintf_len(const char *fmt, ...)
{
    va_list ap;
    int len;

    va_start(ap, fmt);
    len = es_vsnprintf(NULL, 0, fmt, ap);
    va_end(ap);

    return len;
}
void pdf_set_page_size(float width, float height)
{
    s_es_pdf_page_width = width;
    s_es_pdf_page_height = height;
}
int pdf_add_line(struct pdf_doc *pdf, struct pdf_object *page, float x1,
                 float y1, float x2, float y2, float width, uint32_t colour)
{
    int data_len;

    es_pdf_obj_out_file_start(OBJ_stream, 1);

    data_len = 0;

    data_len += es_vsnprintf_len("%f w\r\n", width);
    data_len += es_vsnprintf_len("%f %f m\r\n", x1, y1);
    data_len += es_vsnprintf_len("/DeviceRGB CS\r\n");
    data_len += es_vsnprintf_len("%f %f %f RG\r\n", PDF_RGB_R(colour), PDF_RGB_G(colour), PDF_RGB_B(colour));
    data_len += es_vsnprintf_len("%f %f l S\r\n", x2, y2);

    es_pdf_stream_out_file_start(data_len);

    es_file_printf(s_pdf_save_fp, "%f w\r\n", width);
    es_file_printf(s_pdf_save_fp, "%f %f m\r\n", x1, y1);
    es_file_printf(s_pdf_save_fp, "/DeviceRGB CS\r\n");
    es_file_printf(s_pdf_save_fp, "%f %f %f RG\r\n", PDF_RGB_R(colour), PDF_RGB_G(colour), PDF_RGB_B(colour));
    es_file_printf(s_pdf_save_fp, "%f %f l S\r\n", x2, y2);

    es_pdf_stream_out_file_end();

    es_pdf_obj_out_file_end();

    return 0;
}

int pdf_add_cubic_bezier(struct pdf_doc *pdf, struct pdf_object *page,
                         float x1, float y1, float x2, float y2, float xq1,
                         float yq1, float xq2, float yq2, float width,
                         uint32_t colour)
{
    int data_len;

    es_pdf_obj_out_file_start(OBJ_stream, 1);

    data_len = 0;

    data_len += es_vsnprintf_len("%f w\r\n", width);
    data_len += es_vsnprintf_len("%f %f m\r\n", x1, y1);
    data_len += es_vsnprintf_len("/DeviceRGB CS\r\n");
    data_len += es_vsnprintf_len("%f %f %f RG\r\n", PDF_RGB_R(colour), PDF_RGB_G(colour), PDF_RGB_B(colour));
    data_len += es_vsnprintf_len("%f %f %f %f %f %f c S\r\n", xq1, yq1, xq2, yq2, x2, y2);

    es_pdf_stream_out_file_start(data_len);

    es_file_printf(s_pdf_save_fp, "%f w\r\n", width);
    es_file_printf(s_pdf_save_fp, "%f %f m\r\n", x1, y1);
    es_file_printf(s_pdf_save_fp, "/DeviceRGB CS\r\n");
    es_file_printf(s_pdf_save_fp, "%f %f %f RG\r\n", PDF_RGB_R(colour), PDF_RGB_G(colour), PDF_RGB_B(colour));
    es_file_printf(s_pdf_save_fp, "%f %f %f %f %f %f c S\r\n", xq1, yq1, xq2, yq2, x2, y2);

    es_pdf_stream_out_file_end();

    es_pdf_obj_out_file_end();

    return 0;
}

int pdf_add_ellipse(struct pdf_doc *pdf, struct pdf_object *page, float x,
                    float y, float xradius, float yradius, float width,
                    uint32_t colour, uint32_t fill_colour)
{
    int data_len;
    float lx, ly;

    lx = (4.0f / 3.0f) * (M_SQRT2 - 1) * xradius;
    ly = (4.0f / 3.0f) * (M_SQRT2 - 1) * yradius;

    es_pdf_obj_out_file_start(OBJ_stream, 1);

    data_len = 0;

    if (!PDF_IS_TRANSPARENT(fill_colour))
    {
        data_len += es_vsnprintf_len("/DeviceRGB CS\r\n");
        data_len += es_vsnprintf_len("%f %f %f rg\r\n", PDF_RGB_R(fill_colour), PDF_RGB_G(fill_colour), PDF_RGB_B(fill_colour));
    }

    /* stroke color */
    data_len += es_vsnprintf_len("/DeviceRGB CS\r\n");
    data_len += es_vsnprintf_len("%f %f %f RG\r\n", PDF_RGB_R(colour), PDF_RGB_G(colour), PDF_RGB_B(colour));

    data_len += es_vsnprintf_len("%f w ", width);

    data_len += es_vsnprintf_len("%.2f %.2f m ", (x + xradius), (y));

    data_len += es_vsnprintf_len("%.2f %.2f %.2f %.2f %.2f %.2f c ", (x + xradius), (y - ly), (x + lx), (y - yradius), x, (y - yradius));

    data_len += es_vsnprintf_len("%.2f %.2f %.2f %.2f %.2f %.2f c ", (x - lx), (y - yradius), (x - xradius), (y - ly), (x - xradius), y);

    data_len += es_vsnprintf_len("%.2f %.2f %.2f %.2f %.2f %.2f c ", (x - xradius), (y + ly), (x - lx), (y + yradius), x, (y + yradius));

    data_len += es_vsnprintf_len("%.2f %.2f %.2f %.2f %.2f %.2f c ", (x + lx), (y + yradius), (x + xradius), (y + ly), (x + xradius), y);

    if (PDF_IS_TRANSPARENT(fill_colour))
        data_len += es_vsnprintf_len("%s", "S ");
    else
        data_len += es_vsnprintf_len("%s", "B ");

    es_pdf_stream_out_file_start(data_len);

    if (!PDF_IS_TRANSPARENT(fill_colour))
    {
        es_file_printf(s_pdf_save_fp, "/DeviceRGB CS\r\n");
        es_file_printf(s_pdf_save_fp, "%f %f %f rg\r\n", PDF_RGB_R(fill_colour), PDF_RGB_G(fill_colour), PDF_RGB_B(fill_colour));
    }

    /* stroke color */
    es_file_printf(s_pdf_save_fp, "/DeviceRGB CS\r\n");
    es_file_printf(s_pdf_save_fp, "%f %f %f RG\r\n", PDF_RGB_R(colour), PDF_RGB_G(colour), PDF_RGB_B(colour));

    es_file_printf(s_pdf_save_fp, "%f w ", width);

    es_file_printf(s_pdf_save_fp, "%.2f %.2f m ", (x + xradius), (y));

    es_file_printf(s_pdf_save_fp, "%.2f %.2f %.2f %.2f %.2f %.2f c ", (x + xradius), (y - ly), (x + lx), (y - yradius), x, (y - yradius));

    es_file_printf(s_pdf_save_fp, "%.2f %.2f %.2f %.2f %.2f %.2f c ", (x - lx), (y - yradius), (x - xradius), (y - ly), (x - xradius), y);

    es_file_printf(s_pdf_save_fp, "%.2f %.2f %.2f %.2f %.2f %.2f c ", (x - xradius), (y + ly), (x - lx), (y + yradius), x, (y + yradius));

    es_file_printf(s_pdf_save_fp, "%.2f %.2f %.2f %.2f %.2f %.2f c ", (x + lx), (y + yradius), (x + xradius), (y + ly), (x + xradius), y);

    if (PDF_IS_TRANSPARENT(fill_colour))
        es_file_printf(s_pdf_save_fp, "%s", "S ");
    else
        es_file_printf(s_pdf_save_fp, "%s", "B ");

    es_pdf_stream_out_file_end();

    es_pdf_obj_out_file_end();

    return 0;
}

int pdf_add_circle(struct pdf_doc *pdf, struct pdf_object *page, float xr,
                   float yr, float radius, float width, uint32_t colour,
                   uint32_t fill_colour)
{
    return pdf_add_ellipse(pdf, page, xr, yr, radius, radius, width, colour,
                           fill_colour);
}

int pdf_add_filled_rectangle(struct pdf_doc *pdf, struct pdf_object *page,
                             float x, float y, float width, float height,
                             float border_width, uint32_t colour_fill,
                             uint32_t colour_border)
{
    int data_len;

    es_pdf_obj_out_file_start(OBJ_stream, 1);

    data_len = 0;

    data_len += (es_vsnprintf_len("%f %f %f rg ", PDF_RGB_R(colour_fill), PDF_RGB_G(colour_fill), PDF_RGB_B(colour_fill)));

    if (border_width > 0)
    {
        data_len += (es_vsnprintf_len("%f %f %f RG ", PDF_RGB_R(colour_border), PDF_RGB_G(colour_border), PDF_RGB_B(colour_border)));
        data_len += (es_vsnprintf_len("%f w ", border_width));
        data_len += (es_vsnprintf_len("%f %f %f %f re B ", x, y, width, height));
    }
    else
    {
        data_len += (es_vsnprintf_len("%f %f %f %f re f ", x, y, width, height));
    }

    es_pdf_stream_out_file_start(data_len);

    es_file_printf(s_pdf_save_fp, "%f %f %f rg ", PDF_RGB_R(colour_fill), PDF_RGB_G(colour_fill), PDF_RGB_B(colour_fill));

    if (border_width > 0)
    {
        es_file_printf(s_pdf_save_fp, "%f %f %f RG ", PDF_RGB_R(colour_border), PDF_RGB_G(colour_border), PDF_RGB_B(colour_border));
        es_file_printf(s_pdf_save_fp, "%f w ", border_width);
        es_file_printf(s_pdf_save_fp, "%f %f %f %f re B ", x, y, width, height);
    }
    else
    {
        es_file_printf(s_pdf_save_fp, "%f %f %f %f re f ", x, y, width, height);
    }

    es_pdf_stream_out_file_end();

    es_pdf_obj_out_file_end();

    return 0;
}


int pdf_add_rgb24(struct pdf_doc *pdf, struct pdf_object *page, float x,
                  float y, float display_width, float display_height,
                  const uint8_t *data, uint32_t width, uint32_t height)
{
    if (ES_DATA_SOURSE_IN_ROM(data))
        es_pdf_data_sourse_err();

    s_es_pdf_have_image = 1;
    size_t data_len = (size_t)width * (size_t)height * 3;

    es_pdf_obj_out_file_start(OBJ_image, 0);

    es_file_printf(s_pdf_save_fp,
                   "<<\r\n"
                   "  /Type /XObject\r\n"
                   "  /Name /Image%d\r\n",
                   (s_es_pdf_out_obj_num + 1));

    es_file_printf(s_pdf_save_fp,
                   "  /Subtype /Image\r\n"
                   "  /ColorSpace /DeviceRGB\r\n"
                   "  /Height %d\r\n"
                   "  /Width %d\r\n"
                   "  /BitsPerComponent 8\r\n"
                   "  /Length %zu\r\n"
                   ">>stream\r\n",
                   height, width, data_len + 1);

    es_file_write(s_pdf_save_fp, data, data_len);

    es_pdf_stream_out_file_end();

    es_pdf_obj_out_file_end();

    es_pdf_obj_out_file_start(OBJ_stream, 1);

    data_len = 0;
    data_len += es_vsnprintf_len("q ");
    data_len += es_vsnprintf_len("%f 0 0 %f %f %f cm ", (float)display_width, (float)display_height, (float)x, (float)y);
    data_len += es_vsnprintf_len("/Image%d Do Q", s_es_pdf_out_obj_num);

    es_pdf_stream_out_file_start(data_len);

    es_file_printf(s_pdf_save_fp, "q ");
    es_file_printf(s_pdf_save_fp, "%f 0 0 %f %f %f cm ", (float)display_width, (float)display_height, (float)x, (float)y);
    es_file_printf(s_pdf_save_fp, "/Image%d Do Q", s_es_pdf_out_obj_num);

    es_pdf_stream_out_file_end();

    es_pdf_obj_out_file_end();

    return 0;
}

static int utf8_to_utf32(const char *utf8, int len, uint32_t *utf32)
{
    uint32_t ch;
    uint8_t mask;

    if (len <= 0 || !utf8 || !utf32)
        return -EINVAL;

    ch = *(uint8_t *)utf8;

    if ((ch & 0x80) == 0)
    {
        len = 1;
        mask = 0x7f;
    }
    else if ((ch & 0xe0) == 0xc0 && len >= 2)
    {
        len = 2;
        mask = 0x1f;
    }
    else if ((ch & 0xf0) == 0xe0 && len >= 3)
    {
        len = 3;
        mask = 0xf;
    }
    else if ((ch & 0xf8) == 0xf0 && len >= 4)
    {
        len = 4;
        mask = 0x7;
    }
    else
        return -EINVAL;

    ch = 0;

    for (int i = 0; i < len; i++)
    {
        int shift = (len - i - 1) * 6;

        if (!*utf8)
            return -EINVAL;

        if (i == 0)
            ch |= ((uint32_t)(*utf8++) & mask) << shift;
        else
            ch |= ((uint32_t)(*utf8++) & 0x3f) << shift;
    }

    *utf32 = ch;

    return len;
}

static int utf8_to_pdfencoding(struct pdf_doc *pdf, const char *utf8, int len,
                               uint8_t *res)
{
    uint32_t code;
    int code_len;

    *res = 0;

    code_len = utf8_to_utf32(utf8, len, &code);

    if (code_len < 0)
    {
        printf("Invalid UTF-8 encoding\r\n");
    }

    if (code > 255)
    {
        /* We support *some* minimal UTF-8 characters */
        /* See Appendix D of
           https://www.adobe.com/content/dam/acom/en/devnet/pdf/pdfs/pdf_reference_archives/PDFReference.pdf
           These are all in WinAnsiEncoding*/
        switch (code)
        {
            case 0x160: /* Latin Capital Letter S with Caron */
                *res = 0212;
                break;

            case 0x161: /* Latin Small Letter S with Caron */
                *res = 0232;
                break;

            case 0x17d: /* Latin Capital Letter Z with Caron */
                *res = 0216;
                break;

            case 0x17e: /* Latin Small Letter Z with Caron */
                *res = 0236;
                break;

            case 0x2014: /* emdash */
                *res = 0227;
                break;

            case 0x2018: /* left single quote */
                *res = 0221;
                break;

            case 0x2019: /* right single quote */
                *res = 0222;
                break;

            case 0x201c: /* left double quote */
                *res = 0223;
                break;

            case 0x201d: /* right double quote */
                *res = 0224;
                break;

            case 0x20ac: /* Euro */
                *res = 0200;
                break;

            default:
                printf("Unsupported UTF-8 character: 0x%x 0o%o", (unsigned int)code, (unsigned int)code);
                return -1;
        }
    }
    else
    {
        *res = code;
    }

    return code_len;
}


static int pdf_add_text_spacing(struct pdf_doc *pdf, struct pdf_object *page,
                                const char *text, float size, float xoff,
                                float yoff, uint32_t colour, float spacing)
{
    size_t len = text ? strlen(text) : 0;
    int alpha = (colour >> 24) >> 4;
    int data_len;

    /* Don't bother adding empty/null strings */
    if (!len)
        return 0;

    es_pdf_obj_out_file_start(OBJ_stream, 1);

    data_len =
        (es_vsnprintf_len("BT /GS%d gs ", alpha)) +
        (es_vsnprintf_len("%f %f TD ", xoff, yoff)) +
        (es_vsnprintf_len("/F%d %f Tf ", s_es_pdf_font_now, size)) +
        (es_vsnprintf_len("%f %f %f rg ", PDF_RGB_R(colour), PDF_RGB_G(colour), PDF_RGB_B(colour))) +
        (es_vsnprintf_len("%f Tc (", spacing)) +
        (es_vsnprintf_len(") Tj ET"));

    /* Escape magic characters properly */
    for (size_t i = 0; i < len;)
    {
        int code_len;
        uint8_t pdf_char;
        code_len = utf8_to_pdfencoding(pdf, &text[i], len - i, &pdf_char);

        if (code_len < 0)
        {
            return code_len;
        }

        if (strchr("()\\", pdf_char))
        {
            char buf[3];
            /* Escape some characters */
            buf[0] = '\\';
            buf[1] = pdf_char;
            buf[2] = '\0';
            data_len += strlen(buf);/*es_file_write(pdf_save_fp, buf,strlen(buf));*/
        }
        else if (strrchr("\n\r\t\b\f", pdf_char))
        {
            /* Skip over these characters */
            ;
        }
        else
        {
            data_len ++;/*es_file_write(pdf_save_fp, &pdf_char,1);*/
        }

        i += code_len;
    }

    es_pdf_stream_out_file_start(data_len);

    es_file_printf(s_pdf_save_fp, "BT /GS%d gs ", alpha);
    es_file_printf(s_pdf_save_fp, "%f %f TD ", xoff, yoff);
    es_file_printf(s_pdf_save_fp, "/F%d %f Tf ", s_es_pdf_font_now, size);
    es_file_printf(s_pdf_save_fp, "%f %f %f rg ", PDF_RGB_R(colour), PDF_RGB_G(colour), PDF_RGB_B(colour));
    es_file_printf(s_pdf_save_fp, "%f Tc (", spacing);

    /* Escape magic characters properly */
    for (size_t i = 0; i < len;)
    {
        int code_len;
        uint8_t pdf_char;
        code_len = utf8_to_pdfencoding(pdf, &text[i], len - i, &pdf_char);

        if (code_len < 0)
        {
            return code_len;
        }

        if (strchr("()\\", pdf_char))
        {
            char buf[3];
            /* Escape some characters */
            buf[0] = '\\';
            buf[1] = pdf_char;
            buf[2] = '\0';
            es_file_write(s_pdf_save_fp, buf, strlen(buf));
        }
        else if (strrchr("\n\r\t\b\f", pdf_char))
        {
            /* Skip over these characters */
            ;
        }
        else
        {
            es_file_write(s_pdf_save_fp, &pdf_char, 1);
        }

        i += code_len;
    }

    es_file_printf(s_pdf_save_fp, ") Tj ET");

    es_pdf_stream_out_file_end();

    es_pdf_obj_out_file_end();

    return 0;
}

int pdf_add_text(struct pdf_doc *pdf, struct pdf_object *page,
                 const char *text, float size, float xoff, float yoff,
                 uint32_t colour)
{
    return pdf_add_text_spacing(pdf, page, text, size, xoff, yoff, colour, 0);
}

int pdf_set_font(struct pdf_doc *pdf, const char *font)
{
    uint32_t i;

    if (ES_DATA_SOURSE_IN_ROM(font))
        es_pdf_data_sourse_err();

    s_es_pdf_font_now = 0;

    if (s_es_pdf_out_obj_font_num)
    {
        for (i = 0; i < s_es_pdf_out_obj_font_num; i++)
        {
            if ((strncmp(font, (const char *)(s_es_pdf_out_obj_font_list[i]), strlen(font))) == 0)
            {
                s_es_pdf_font_now = i + 1;
                return 0;
            }
        }
    }

    s_es_pdf_font_now = s_es_pdf_out_obj_font_num + 1;

    s_es_pdf_out_obj_font_list[s_es_pdf_out_obj_font_num] = (uint8_t *)font;
    s_es_pdf_out_obj_font_num++;

    es_pdf_obj_out_file_start(OBJ_font, 0);

    es_file_printf(s_pdf_save_fp,
                   "<<\r\n"
                   "  /Type /Font\r\n"
                   "  /Subtype /Type1\r\n"
                   "  /BaseFont /%s\r\n"
                   "  /Encoding /WinAnsiEncoding\r\n"
                   ">>\r\n",
                   font);

    es_pdf_obj_out_file_end();

    return 0;
}

struct pdf_object *pdf_append_page(struct pdf_doc *pdf)
{
    s_es_pdf_out_page_num++;
    s_es_pdf_page_index_now++;

    return 0;
}

/* Slightly modified djb2 hash algorithm to get pseudo-random ID*/
static uint64_t hash(uint64_t hash, const void *data, size_t len)
{
    const uint8_t *d8 = (const uint8_t *)data;

    for (; len; len--)
    {
        hash = (((hash & 0x03ffffffffffffff) << 5) +
                (hash & 0x7fffffffffffffff)) +
               *d8++;
    }

    return hash;
}

int pdf_save_file_start(struct pdf_info *info, FIL *fp)
{
    /*init*/
    s_pdf_save_fp = fp;
    s_es_pdf_page_index_now = 0;
    s_es_pdf_font_now = 0;
    g_pdf_out_bytes = 0;
    s_es_pdf_out_obj_font_num = 0;
    s_es_pdf_out_obj_num = 0;
    s_es_pdf_out_page_num = 0;
    s_es_pdf_have_image = 0;

    es_file_printf(s_pdf_save_fp, "%%PDF-1.3\r\n");
    /* Hibit bytes */
    es_file_printf(s_pdf_save_fp, "%c%c%c%c%c\r\n", 0x25, 0xc7, 0xec, 0x8f, 0xa2);

    es_pdf_obj_out_file_start(OBJ_info, 0);

    es_file_printf(s_pdf_save_fp, "<<\r\n");

    if (info->creator[0])
        es_file_printf(s_pdf_save_fp, "  /Creator (%s)\r\n", info->creator);

    if (info->producer[0])
        es_file_printf(s_pdf_save_fp, "  /Producer (%s)\r\n", info->producer);

    if (info->title[0])
        es_file_printf(s_pdf_save_fp, "  /Title (%s)\r\n", info->title);

    if (info->author[0])
        es_file_printf(s_pdf_save_fp, "  /Author (%s)\r\n", info->author);

    if (info->subject[0])
        es_file_printf(s_pdf_save_fp, "  /Subject (%s)\r\n", info->subject);

    if (info->date[0])
        es_file_printf(s_pdf_save_fp, "  /CreationDate (D:%s)\r\n", info->date);

    es_file_printf(s_pdf_save_fp, ">>\r\n");

    es_pdf_obj_out_file_end();

    return 0;
}

int pdf_save_file_end(struct pdf_info *info, FIL *fp)
{
    int xref_offset;
    uint64_t id1, id2;
    time_t now = time(NULL);

    /*OBJ_catalog*/
    es_pdf_obj_out_file_start(OBJ_catalog, 0);

    es_file_printf(s_pdf_save_fp, "<<\r\n"
                   "  /Type /Catalog\r\n");

    es_file_printf(s_pdf_save_fp,
                   "  /Pages %d 0 R\r\n"
                   ">>\r\n",
                   (s_es_pdf_out_obj_num + 2));

    es_pdf_obj_out_file_end();

    /*OBJ_pages*/
    es_pdf_obj_out_file_start(OBJ_pages, 0);

    uint16_t page_parents_index = s_es_pdf_out_obj_num + 1;

    es_file_printf(s_pdf_save_fp, "<<\r\n"
                   "  /Type /Pages\r\n"
                   "  /Kids [ ");

    for (int i = 1; i <= s_es_pdf_page_index_now; i++)
    {
        es_file_printf(s_pdf_save_fp, "%d 0 R ", (s_es_pdf_out_obj_num + i + 1));
    }

    es_file_printf(s_pdf_save_fp, "]\r\n");
    es_file_printf(s_pdf_save_fp, "  /Count %d\r\n", s_es_pdf_page_index_now);
    es_file_printf(s_pdf_save_fp, ">>\r\n");

    es_pdf_obj_out_file_end();

    /*OBJ_page*/

    uint16_t pagei, i, j;

    for (pagei = 1; pagei <= s_es_pdf_page_index_now ; pagei++)
    {
        es_pdf_obj_out_file_start(OBJ_page, 0);

        es_file_printf(s_pdf_save_fp,
                       "<<\r\n"
                       "  /Type /Page\r\n"
                       "  /Parent %d 0 R\r\n",
                       page_parents_index);
        es_file_printf(s_pdf_save_fp, "  /MediaBox [0 0 %f %f]\r\n", s_es_pdf_page_width, s_es_pdf_page_height);
        es_file_printf(s_pdf_save_fp, "  /Resources <<\r\n");
        es_file_printf(s_pdf_save_fp, "    /Font <<\r\n");

        j = 1;

        for (i = 0; i < s_es_pdf_out_obj_num; i++)
        {
            if (s_es_pdf_out_obj_type_list[i] == OBJ_font)
                es_file_printf(s_pdf_save_fp, "      /F%d %d 0 R\r\n", j++, i + 1);
        }

        es_file_printf(s_pdf_save_fp, "    >>\r\n");
        /* We trim transparency to just 4-bits*/
        es_file_printf(s_pdf_save_fp, "    /ExtGState <<\r\n");

        for (int i = 0; i < 16; i++)
        {
            es_file_printf(s_pdf_save_fp, "      /GS%d <</ca %f>>\r\n", i,
                           (float)(15 - i) / 15);
        }

        es_file_printf(s_pdf_save_fp, "    >>\r\n");

        if (s_es_pdf_have_image)
        {
            es_file_printf(s_pdf_save_fp, "    /XObject <<");

            for (i = 0; i < s_es_pdf_out_obj_num; i++)
            {
                if (s_es_pdf_out_obj_type_list[i] == OBJ_image)
                    es_file_printf(s_pdf_save_fp, "      /Image%d %d 0 R ", i + 1, i + 1);
            }

            es_file_printf(s_pdf_save_fp, "    >>\r\n");
        }

        es_file_printf(s_pdf_save_fp, "  >>\r\n");

        es_file_printf(s_pdf_save_fp, "  /Contents [\r\n");

        for (i = 0; i < s_es_pdf_out_obj_num; i++)
        {
            if (pagei == s_es_pdf_out_obj_in_pagen[i])
                es_file_printf(s_pdf_save_fp, "%d 0 R\r\n", (i + 1));
        }

        es_file_printf(s_pdf_save_fp, "]\r\n");

        /*Omitted annotations*/

        es_file_printf(s_pdf_save_fp, ">>\r\n");

        es_pdf_obj_out_file_end();
    }

    /* xref */
    xref_offset = g_pdf_out_bytes;/*ftell(fp);*/
    es_file_printf(s_pdf_save_fp, "xref\r\n");
    es_file_printf(s_pdf_save_fp, "0 %d\r\n", s_es_pdf_out_obj_num + 1);
    es_file_printf(s_pdf_save_fp, "0000000000 65535 f\r\n");

    for (int i = 0; i < s_es_pdf_out_obj_num; i++)
    {
        es_file_printf(s_pdf_save_fp, "%10.10d 00000 n\r\n", s_es_pdf_out_obj_location_list[i]);
    }

    es_file_printf(s_pdf_save_fp,
                   "trailer\r\n"
                   "<<\r\n"
                   "/Size %d\r\n",
                   s_es_pdf_out_obj_num + 1);
    es_file_printf(s_pdf_save_fp, "/Root %d 0 R\r\n", (page_parents_index - 1));
    es_file_printf(s_pdf_save_fp, "/Info %d 0 R\r\n", 1);
    /* Generate document unique IDs */
    id1 = hash(5381, info, sizeof(struct pdf_info));
    id1 = hash(id1, &s_es_pdf_out_obj_num, sizeof(s_es_pdf_out_obj_num));
    id2 = hash(5381, &now, sizeof(now));
    es_file_printf(s_pdf_save_fp, "/ID [<%16.16" PRIx64 "> <%16.16" PRIx64 ">]\r\n", id1, id2);
    es_file_printf(s_pdf_save_fp, ">>\r\n"
                   "startxref\r\n");
    es_file_printf(s_pdf_save_fp, "%d\r\n", xref_offset);
    es_file_printf(s_pdf_save_fp, "%%%%EOF\r\n");

    s_pdf_save_fp = NULL;

    return 0;
}
