/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    09 Jun 2023
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          23 Nov 2021     Ginger          the first version
 *          23 Dec 2021     Ginger          Modify the example for PDS board
 *          21 Mar 2022     AE Team         Modify MD Driver
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Define to prevent recursive inclusion -------------------------------------*/
#define MAIN_GLOBALS

/* Includes -------------------------------------------------------------------*/
#include <stdio.h>
#include "main.h"

/** @addtogroup Projects_Examples_MD
  * @{
  */

/* Private types --------------------------------------------------------------*/
md_rcu_init_typedef rcu_initStruct =    /**< RCU init structure */
{
    MD_RCU_MPRE_MCO_DIV1,
    MD_RCU_MSW_MCO_DISABLE,
    MD_RCU_PLLSRC_HOSC,//MD_RCU_PLLSRC_HRC48,
    MD_RCU_PLLCLK_72M,
    MD_RCU_PPRE_HCLK_DIV_1,
    MD_RCU_HPRE_SYSCLK_DIV_1,
    MD_RCU_SW_SYSCLK_PLL0,
    (RCU_CON_HRCON | RCU_CON_HOSCON | RCU_CON_PLL0ON | RCU_CON_HRC48ON_MSK),
};

md_uart_init_typedef uart_initStruct =    /**< UART init structure */
{
    MD_UART_BAUDRATE_115200,
    MD_UART_LCON_LSB_FIRST,
    MD_UART_LCON_PS_NONE,
    MD_UART_LCON_STOP_1,
    MD_UART_LCON_DLS_8,
};

/* Private define -------------------------------------------------------------*/
/* Private macro --------------------------------------------------------------*/
/* Private variable -----------------------------------------------------------*/
uint32_t g_write_fatfs_time_ms = 0U;
/* Private function prototypes ------------------------------------------------*/
void gpio_config(void);
/* Private functions ----------------------------------------------------------*/
void usb_dc_low_level_init(void)
{
    md_rcu_enable_usb(RCU);
		md_mcu_irq_config(USB_IRQn, 3, ENABLE);//USB IRQ
//		md_rcu_enable_usb_reset(RCU);
//		md_rcu_disable_usb_reset(RCU);
}
void es_update_app(void)
{
    uint32_t after_write_fatfs_time_ms;

    if (g_write_fatfs_time_ms)
    {
        after_write_fatfs_time_ms = md_get_tick() - g_write_fatfs_time_ms;

        if ((after_write_fatfs_time_ms > 3000) && (after_write_fatfs_time_ms < 600000))
        {
            get_fatfs_type((uint8_t *)FAT_AND_NEW_APP_FLASH_S);
            get_dir_param((uint8_t *)FAT_AND_NEW_APP_FLASH_S);

            g_write_fatfs_time_ms = 0;
        }
    }
}

void jump2app(void)
{
    __disable_irq();

    md_fc_unlock();
    // REMAP
    // 0x00: EFLASH         -> Remap EFLASH Any 4K
    // 0x01: Bootloader     -> Remap EFLASH 0x00000000
    // 0x02: SRAM           -> Remap SRAM

    // Remap EFLASH
    md_syscfg_set_memory_mapping(SYSCFG, MD_SYSCFG_MEMMOD_MAIN);
    md_syscfg_set_flash_remap_base(SYSCFG, APP_FLASH_S / (4 * 2 * MD_FC_PC_EF_MPAGESZ));
    md_syscfg_enable_memory_remap(SYSCFG);
    md_fc_lock();

    __enable_irq();

    NVIC_SystemReset();
}

uint32_t exist_load_bin(void);

/**
  * @brief  main.
  * @param  None
  * @retval None
  */
int main(void)
{
    md_gpio_inittypedef gpiox;
    __disable_irq();
    md_rcu_pll0_init(RCU, &rcu_initStruct);
    md_rcu_sys_init(RCU, &rcu_initStruct);
    md_rcu_enable_gpioa(RCU);
    md_rcu_enable_gpiob(RCU);
    md_rcu_enable_gpioc(RCU);
    md_rcu_enable_uart1(RCU);
    md_rcu_enable_uart2(RCU);
    gpio_config();
	
		NVIC->ICER[0] = 0xFFFFFFFFUL;
	  md_tick_init(MD_SYSTICK_CLKSRC_HCLK);
		md_tick_set_rvr_reload(TICK, (72000 - 1));
    md_mcu_irq_config(UART1_IRQn, 2, ENABLE);
		md_mcu_irq_config(UART2_IRQn, 2, ENABLE);
    md_uart_init(UART1, &uart_initStruct);
    md_uart_enable_it_rfnempty(UART1);
    md_uart_init(UART2, &uart_initStruct);
    md_uart_enable_it_rfnempty(UART2);
    md_uart_enable_it_tfempty(UART2);
		md_tick_enable_csr_tickie(TICK);
		md_mcu_irq_config(SysTick_IRQn, 3, ENABLE);
    __enable_irq();

		msc_descriptor_init();
	  gpiox.Pin   = MD_GPIO_PIN_13;
    gpiox.Mode  = MD_GPIO_MODE_INPUT;
    gpiox.OutputType = MD_GPIO_OUTPUT_PUSHPULL;
    gpiox.Pull  = MD_GPIO_PULL_FLOATING;
    gpiox.OutDrive = MD_GPIO_DRIVING_8MA;
    gpiox.Function = MD_GPIO_AF0;
	  md_gpio_init(GPIOC, &gpiox);
    md_delay_1ms(1000);
		md_tick_waitms(100,10);

    while (1)
    {
        es_update_app();

				if(md_gpio_get_inputpinset(GPIOC, MD_GPIO_PIN_13) == 0)
				{
					get_fatfs_type((uint8_t *)FAT_AND_NEW_APP_FLASH_S);
					if(exist_load_bin()){//load.binתapp
						printf("main.c->while(1)->jump2app\r\n");
						md_delay_1ms(10);
						jump2app();
					}
				}
    }
}

/**
  * @brief  Configure the GPIO.
  * @param  None
  * @retval None
  */
void gpio_config(void){
	  md_gpio_inittypedef gpiox;
    /* config gpiob pin6 (Tx) */
	  gpiox.Pin   = MD_GPIO_PIN_6;
    gpiox.Mode  = MD_GPIO_MODE_FUNCTION;
    gpiox.OutputType = MD_GPIO_OUTPUT_PUSHPULL;
    gpiox.Pull  = MD_GPIO_PULL_FLOATING;
    gpiox.OutDrive = MD_GPIO_DRIVING_8MA;
    gpiox.Function = MD_GPIO_AF2;
	  md_gpio_init(GPIOB, &gpiox);
    /* config gpiob pin7 (Rx) */
	  gpiox.Pin   = MD_GPIO_PIN_7;
    gpiox.Mode  = MD_GPIO_MODE_FUNCTION;
    gpiox.OutputType = MD_GPIO_OUTPUT_PUSHPULL;
    gpiox.Pull  = MD_GPIO_PULL_UP;
    gpiox.OutDrive = MD_GPIO_DRIVING_8MA;
    gpiox.Function = MD_GPIO_AF2;
	  md_gpio_init(GPIOB, &gpiox);
	
    /* config gpioa pin2 (Tx) */
	  gpiox.Pin   = MD_GPIO_PIN_2;
    gpiox.Mode  = MD_GPIO_MODE_FUNCTION;
    gpiox.OutputType = MD_GPIO_OUTPUT_PUSHPULL;
    gpiox.Pull  = MD_GPIO_PULL_FLOATING;
    gpiox.OutDrive = MD_GPIO_DRIVING_8MA;
    gpiox.Function = MD_GPIO_AF2;
	  md_gpio_init(GPIOA, &gpiox);
    /* config gpioa pin3 (Rx) */
	  gpiox.Pin   = MD_GPIO_PIN_3;
    gpiox.Mode  = MD_GPIO_MODE_FUNCTION;
    gpiox.OutputType = MD_GPIO_OUTPUT_PUSHPULL;
    gpiox.Pull  = MD_GPIO_PULL_UP;
    gpiox.OutDrive = MD_GPIO_DRIVING_8MA;
    gpiox.Function = MD_GPIO_AF2;
	  md_gpio_init(GPIOA, &gpiox);
}

/**
  * @brief  Uart sendchar.
  * @param  arg: char to be sent.
  * @retval data to be sent.
  */
uint8_t  sendchar(uint8_t ch)
{
    while (!(UART1->STAT & (UART_STAT_TFEMPTY)));  // Tx FIFO empty

    UART1->TXDATA = ch;            // Sent byte
    return (ch);
}

/**
  * @} Projects_Examples_MD
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/


