/**
  *********************************************************************************
  *
  * @file    boot_fatfs.c
  * @brief   Fatfs file for DEMO
  *
  * @version V1.0
  * @date    31 Dec 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          31 Dec 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#include "boot_fatfs.h"

/** @addtogroup Bootloader Bootloader
  * @{
  */
/** @defgroup USB_Fatfs Fatfs
  * @brief    USB Fatfs Module
  * @{
  */
/** @defgroup Fatfs_Public_Variables Public Variables
  * @brief    Fatfs Public Variables
  * @{
  */
fat_type_e FATFS_TYPE = TYPE_UKNOW;

/**
  * @}
  */

uint8_t es_iap_program_words(uint32_t addr, uint8_t *buf, uint32_t len, uint8_t en)
{
    uint8_t ret = 0;
		uint32_t i;
    md_fc_ControlTypeDef  FCControlPara;
    FCControlPara.SAddr = addr;
    FCControlPara.SAddrC = ~addr;
    FCControlPara.BCnt = DISK_BLOCK_SIZE;
    FCControlPara.pU32Buf = (uint32_t*)buf;

    if (en)
    {
				for(i = 0;i < len;i += DISK_BLOCK_SIZE)
				{
					md_fc_unlock();
					if (md_fc_page_erase(&FCControlPara) == ERROR)
					{
						md_fc_lock();
						ret = 1;
						break;
					}
					else{
						md_fc_lock();
					}
				}
    }
    FCControlPara.SAddr = addr;
    FCControlPara.SAddrC = ~addr;
    FCControlPara.BCnt = len;
    FCControlPara.pU32Buf = (uint32_t*)buf;
		md_fc_unlock();
    if (md_fc_program(&FCControlPara) != SUCCESS){
				md_fc_lock();
        ret = 2;
		}
		else{
				md_fc_lock();
		}

    if (ret)
        printf("iap write error=%d\r\n",ret);

    return ret;
}

/** @defgroup Fatfs_Private_Functions Private Functions
  * @brief    Fatfs Private Functions
  * @{
  */
/**
  * @brief  transform little port data to big port.
  * @param  dat: data source
  * @param  buf_len: data length
  * @retval None
  */
static uint32_t buf_lb2bb(uint8_t *dat, uint8_t buf_len)
{
    uint32_t temp = 0;
    uint32_t fact = 1;
    uint8_t i = 0;
    uint8_t len = buf_len;

    for (i = 0; i < len; i++)
    {
        temp += dat[i] * fact;
        fact *= 256;
    }

    return temp;
}

/**
  * @brief  get data cluster origination section.
  * @param  src: BDR section
  * @retval origination data section number
  */
static uint32_t get_dataclust_sec(void *src)
{
    uint32_t FirstDataSec   = 0;
    uint32_t RootDirSectors = 0;
    uint32_t RootEntCnt     = 0;
    uint32_t BytesPerSec    = 0;
    uint32_t FATSz          = 0;
    uint32_t RsvdSecCnt     = 0;
    uint32_t NumFATs        = 0;

    dbr_t *pDbr = (dbr_t *)src;
    fat32_dbr_t *pDbr32 = (fat32_dbr_t *)src;

    if (FATFS_TYPE == TYPE_UKNOW)
        return 0;

    RootEntCnt = buf_lb2bb(pDbr->BPB_RootEntCnt, 2);
    BytesPerSec = buf_lb2bb(pDbr->BPB_BytsPerSec, 2);
    RsvdSecCnt = buf_lb2bb(pDbr->BPB_RsvSecCnt, 2);
    NumFATs = buf_lb2bb(pDbr->BPB_NumFATs, 1);

    if (FATFS_TYPE == TYPE_FAT32)
    {
        FATSz = buf_lb2bb(pDbr32->fat.BPB_FATSz32, 4);
    }
    else
    {
        FATSz = buf_lb2bb(pDbr->BPB_FATSz16, 2);
    }

    if (BytesPerSec == 0)
        return 0;

    RootDirSectors = ((RootEntCnt * 32) + (BytesPerSec - 1)) / BytesPerSec;
    FirstDataSec = RsvdSecCnt + (NumFATs * FATSz) + RootDirSectors;

    return FirstDataSec;
}

/**
  * @}
  */

/** @defgroup Fatfs_Public_Functions Public Functions
  * @brief    Fatfs Public Functions
  * @verbatim
 ===============================================================================
              ##### USB flash functions #####
 ===============================================================================
   [..]   This section provides functions allowing to:
      (+) get file system type.
      (+) check specified FAT status.
      (+) Initialize FAT12 file system.
      (+) get specified file information.
      (+) read data from fs.
      (+) write data to fs.
      (+) Write data to app flash specified address.
      (+) Write data to app ram specified address.
      (+) get bin file header struct.
      (+) check bin crc value.
      (+) program software in ram.
      (+) program software in flash.
    @endverbatim
  * @{
  */
/**
  * @brief  get specified FAT Section.
  * @param  src: BDR section
  * @param  crufat: specified FAT
  * @retval specified FAT correspond Section.
  */
uint32_t get_curfat_sec(void *src, uint32_t crufat)
{
    uint32_t FirstDataSec;
    uint32_t curFatSec;
    dbr_t *pDbr = (dbr_t *)src;
    uint32_t SecPerClus;

    SecPerClus = buf_lb2bb(pDbr->BPB_SecperClus, 1);
    FirstDataSec = get_dataclust_sec(src);

    if (crufat > 2)
    {
        curFatSec = FirstDataSec + SecPerClus * (crufat - 2);
    }
    else
    {
        curFatSec = FirstDataSec;
    }

    return curFatSec;
}

/**
  * @brief  get root cluster section.
  * @param  src: BDR section
  * @retval section number of the root cluster.
  */
uint32_t get_rootclust_sec(void *src)
{
    uint32_t RootDirSectors = 0;
    uint32_t FATSz          = 0;
    uint32_t RsvdSecCnt     = 0;
    uint32_t NumFATs        = 0;

    dbr_t *pDbr = (dbr_t *)src;
    fat32_dbr_t *pDbr32 = (fat32_dbr_t *)src;

    if (FATFS_TYPE == TYPE_UKNOW)
        return 0;

    RsvdSecCnt = buf_lb2bb(pDbr->BPB_RsvSecCnt, 2);
    NumFATs = buf_lb2bb(pDbr->BPB_NumFATs, 1);

    if (FATFS_TYPE == TYPE_FAT32)
    {
        FATSz = buf_lb2bb(pDbr32->fat.BPB_FATSz32, 4);
    }
    else
    {
        FATSz = buf_lb2bb(pDbr->BPB_FATSz16, 2);
    }

    RootDirSectors = RsvdSecCnt + (NumFATs * FATSz);

    return RootDirSectors;
}

/**
  * @brief  get fatfs type.
  * @param  src: BDR section
  * @retval None
  */
void get_fatfs_type(void *src)
{
    uint32_t RootDirSectors = 0;
    uint32_t RootEntCnt     = 0;
    uint32_t BytesPerSec    = 0;
    uint32_t FATSz          = 0;
    uint32_t RsvdSecCnt     = 0;
    uint32_t NumFATs        = 0;
    uint32_t TotSec         = 0;
    uint32_t DataSec        = 0;
    uint32_t CntofClust     = 0;
    uint32_t SecPerClus     = 0;

    dbr_t *pDbr = (dbr_t *)src;
    fat32_dbr_t *pDbr32 = (fat32_dbr_t *)src;
    uint8_t *pflag = (uint8_t *)src;

    if (src == NULL)
    {
        FATFS_TYPE = TYPE_UKNOW;
        return;
    }

    if ((pflag[510] != 0x55) || (pflag[511] != 0xAA))
    {
        FATFS_TYPE = TYPE_UKNOW;
        return;
    }

    RootEntCnt = buf_lb2bb(pDbr->BPB_RootEntCnt, 2);
    BytesPerSec = buf_lb2bb(pDbr->BPB_BytsPerSec, 2);

    if (BytesPerSec == 0)
        return;

    RootDirSectors = ((RootEntCnt * 32) + (BytesPerSec - 1)) / BytesPerSec;
    FATSz = buf_lb2bb(pDbr->BPB_FATSz16, 2);
    TotSec = buf_lb2bb(pDbr->BPB_TotSec16, 2);
    NumFATs = buf_lb2bb(pDbr->BPB_NumFATs, 1);
    SecPerClus = buf_lb2bb(pDbr->BPB_SecperClus, 1);
    RsvdSecCnt = buf_lb2bb(pDbr->BPB_RsvSecCnt, 2);

    if (FATSz == 0)
        FATSz = buf_lb2bb(pDbr32->fat.BPB_FATSz32, 4);

    if (TotSec == 0)
        TotSec = buf_lb2bb(pDbr->BPB_TotSec32, 4);

    DataSec = TotSec - (RsvdSecCnt + (NumFATs * FATSz) + RootDirSectors);

    if (SecPerClus == 0)
        return;

    CntofClust = DataSec / SecPerClus;

    if (CntofClust < 4085)
    {
        FATFS_TYPE = TYPE_FAT12;
    }
    else if (CntofClust < 65525)
    {
        FATFS_TYPE = TYPE_FAT16;
    }
    else
    {
        FATFS_TYPE = TYPE_FAT32;
    }

    return;
}

/**
  * @brief  check specified FAT if is eof.
  * @param  FATContent: specified FAT number.
  * @retval 0-success, other value indicates failed.
  */
TypeBool curfat_is_eof(uint32_t FATContent)
{
    if (FATFS_TYPE == TYPE_FAT12)
    {
        if (FATContent >= 0x0FF8)
            return TRUE;
    }
    else if (FATFS_TYPE == TYPE_FAT16)
    {
        if (FATContent >= 0xFFF8)
            return TRUE;
    }
    else if (FATFS_TYPE == TYPE_FAT32)
    {
        if (FATContent >= 0x0FFFFFF8)
            return TRUE;
    }
    else
    {

    }

    return FALSE;
}
/**
  * @brief  check specified FAT if is bat.
  * @param  FATContent: specified FAT number.
  * @retval 0-success, other value indicates failed.
  */
TypeBool curfat_is_bat(uint32_t FATContent)
{
    if (FATFS_TYPE == TYPE_FAT12)
    {
        if (FATContent == 0x0FF7)
            return TRUE;
    }
    else if (FATFS_TYPE == TYPE_FAT16)
    {
        if (FATContent == 0xFFF7)
            return TRUE;
    }
    else if (FATFS_TYPE == TYPE_FAT32)
    {
        if (FATContent == 0x0FFFFFF7)
            return TRUE;
    }
    else
    {

    }

    return FALSE;
}

void jump2boot(void)
{
    __disable_irq();

    md_fc_unlock();
    // REMAP
    // 0x00: EFLASH         -> Remap EFLASH Any 4K
    // 0x01: Bootloader     -> Remap EFLASH 0x00000000
    // 0x02: SRAM           -> Remap SRAM

    // Remap EFLASH
    md_syscfg_set_memory_mapping(SYSCFG, MD_SYSCFG_MEMMOD_MAIN);
    md_syscfg_set_flash_remap_base(SYSCFG, BOOT_FLASH_S / (4 * 2 * MD_FC_PC_EF_MPAGESZ));
    md_syscfg_enable_memory_remap(SYSCFG);
    md_fc_lock();

    __enable_irq();

    NVIC_SystemReset();
}

/**
  * @brief  get specified file information.
  * @param  src: BDR section
  * @param  fName: specified file name
  * @param  f_len: loading specified file length
  * @retval specified file first FAT value.
  */
uint32_t get_dir_param(void *src)
{
    uint8_t *pRootDir = NULL;
    uint8_t tmp = 0;
    uint32_t start_fat = 0;

    uint32_t RootClustSec = 0;
    uint32_t BytesPerSec = 0;
    dbr_t *pDbr = (dbr_t *)src;

    uint32_t FileSz = 0;
    uint8_t *pBin = NULL;
    bin_hdr_t bin_hdr = {0};

    RootClustSec = get_rootclust_sec(src);
    BytesPerSec  = buf_lb2bb(pDbr->BPB_BytsPerSec, 2);

    pRootDir = (uint8_t *)src + (BytesPerSec * RootClustSec);
	

#if defined USBH_FATFS
    pRootDir = src;
#endif

    while (*pRootDir != 0x00)
    {
        if (pRootDir[0] != 0xE5)
        {
            if ((pRootDir[11] & ATTR_ARCHIEVE) == ATTR_ARCHIEVE)
            {
                /*长度有限，只做部分文件名判断*/
                tmp = 1;

                /*文件名为pRootDir*/
                printf("boot_fatfs.c->get_dir_param->find file %s\r\n", pRootDir);
								//printf("boot_fatfs.c->get_dir_param->dataclust=%d\r\n", get_dataclust_sec(src));

                /*文件属性*/
                start_fat = (pRootDir[27] << 8) + pRootDir[26];
                    //printf("boot_fatfs.c->get_dir_param->start_fat=pRootDir[27]|pRootDir[26]=%x\r\n",start_fat);
                FileSz = (pRootDir[31] << 24) + (pRootDir[30] << 16) + (pRootDir[29] << 8) + pRootDir[28];
                    printf("boot_fatfs.c->get_dir_param->FileSz=%d\r\n",FileSz);

                /*获取文件绝对地址*/
                pBin = get_bin_hdr((uint8_t *)FAT_AND_NEW_APP_FLASH_S, &bin_hdr, start_fat);
                    printf("boot_fatfs.c->get_dir_param->bin file's address=%x\r\n",pBin);

                if ((start_fat == 0) || (pBin == NULL)){
                    tmp = 0;
										printf("1.empty file\r\n");
								}

                if ((FileSz & 0x3) || (((uint32_t)(pBin)) & 0x3)){
                    tmp = 0;
								printf("2.addr not aligned\r\n");
								}

                if (tmp)
                {
                    /*跳转到boot*/
                    printf("boot_fatfs.c->get_dir_param->jump2boot\r\n");
										md_delay_1ms(10);
                    jump2boot();
                }
								else{
										printf("boot_fatfs.c->get_dir_param->error\r\n");
								}
            }
        }

        pRootDir += 32;
    }

    return start_fat;
}


/**
  * @brief  judge if exists load.bin.
  * @param  src: BDR section
  * @param  fName: specified file name
  * @param  f_len: loading specified file length
  * @retval specified file first FAT value.
  */
uint32_t exist_load_bin(void)
{
    uint8_t *pRootDir = NULL;
    uint32_t RootClustSec = 0;
    uint32_t BytesPerSec = 0;
    dbr_t *pDbr = (dbr_t *)FAT_AND_NEW_APP_FLASH_S;
    RootClustSec = get_rootclust_sec((uint8_t *)FAT_AND_NEW_APP_FLASH_S);
    BytesPerSec  = buf_lb2bb(pDbr->BPB_BytsPerSec, 2);
    pRootDir = (uint8_t *)FAT_AND_NEW_APP_FLASH_S + (BytesPerSec * RootClustSec);
	

    while (*pRootDir != 0x00)
    {
        if (pRootDir[0] != 0xE5)
        {
            if ((pRootDir[11] & ATTR_ARCHIEVE) == ATTR_ARCHIEVE)
            {
								if((pRootDir[0]=='L')&&(pRootDir[1]=='O')&&(pRootDir[2]=='A')&&(pRootDir[3]=='D')&&(pRootDir[8]=='B')&&(pRootDir[9]=='I')&&(pRootDir[10]=='N')){
									return 1;
								}
								else{
									return 0;
								}
            }
        }
        pRootDir += 32;
    }

    return 0;
}

/**
  * @brief  get bin file header struct.
  * @param  src: pointer of the fatfs DBR section.
  * @param  b_hdr: loader bin file header struct.
  * @param  StFAT: the bin file first FAT.
  * @retval pointer of the program bin address.
  */
uint8_t *get_bin_hdr(void *src, bin_hdr_t *b_hdr, uint32_t StFAT)
{
    uint32_t BytesPerSec = 0;
    uint8_t *pBin = NULL;
    dbr_t *pDbr = (dbr_t *)src;

    if (b_hdr == NULL)
        return pBin;

    BytesPerSec = buf_lb2bb(pDbr->BPB_BytsPerSec, 2);
    pBin = (uint8_t *)src + (get_dataclust_sec(src) + ((StFAT >= 2) ? (StFAT - 2) : (StFAT))) * BytesPerSec;

    return pBin;
}

/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
