/**********************************************************************************
 *
 * @file    exec_proc.c
 * @brief   define the exection functions for state machine
 *
 * @date    10 Mar 2022
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          10 Mar 2022     AE Team         the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ----------------------------------------------------------------- */

#include "exec_proc.h"
#include "main.h"


/* Private Macros ------------------------------------------------------------ */
typedef  void (*FunVoidType)(void);

/* Public Variables ---------------------------------------------------------- */


/* Private Constants --------------------------------------------------------- */


/* Private function prototypes ----------------------------------------------- */


/* Public function prototypes ----------------------------------------------- */


/* Private Variables--------------------------------------------------------- */


/* Public Function ---------------------------------------------------------- */


/* Private Function ---------------------------------------------------------- */

/**
  * @brief  define the peripheral register clear function.
  * @param  None
  * @retval None
  */
static void sfr_reset(void)
{
    md_rcu_enable_gpiod_reset(RCU);
    md_rcu_enable_gpioc_reset(RCU);
    md_rcu_enable_gpiob_reset(RCU);
    md_rcu_enable_gpioa_reset(RCU);
    md_rcu_enable_uart1_reset(RCU);
}

/**
  * @brief  define the page erase function.
  * @param  None
  * @retval None
  */
static fsm_iap_result_t fsm_page_erase(uint32_t PageCode)
{
    volatile fsm_iap_result_t status = IAP_SUCCESS;
    md_fc_ControlTypeDef SErasePara;
//    printf("[%s] %d\r\n", __FUNCTION__, PageCode);

    __disable_irq();

    md_fc_unlock();

    //Erase after page "APP_ADDR_PAGE"
    PageCode += APP_ADDR_PAGE;
    SErasePara.SAddr = PageCode * MD_FC_PC_EF_MPAGESZ;
    SErasePara.SAddrC = ~(PageCode * MD_FC_PC_EF_MPAGESZ);

    if (SUCCESS != md_fc_page_erase(&SErasePara))
        status = IAP_FAIL;

    md_fc_unlock();

    __enable_irq();

    return status;
}

/**
  * @brief  define the words program function.
  * @param  None
  * @retval None
  */
static fsm_iap_result_t fsm_words_program(uint32_t address, uint32_t data_address, uint32_t len)
{
    volatile fsm_iap_result_t status = IAP_SUCCESS;
    uint8_t *pdata;
    md_fc_ControlTypeDef ProgramPara;
    //printf("[%s]address:%x, len:%x\r\n", __FUNCTION__, address, len);

    if (address & 0x3)
        return IAP_FAIL;

    __disable_irq();

    md_fc_unlock();

    pdata = (uint8_t *)malloc(len);
    memcpy(pdata, (uint8_t *)data_address, len);
    ProgramPara.BCnt = len;
    ProgramPara.pU32Buf = (uint32_t *)pdata;
    ProgramPara.SAddr = address + APP_ADDR;
    ProgramPara.SAddrC = ~(address + APP_ADDR);

    if (SUCCESS != md_fc_program(&ProgramPara))
    {
        status = IAP_FAIL;
    }

    md_fc_lock();

    __enable_irq();

    free(pdata);
    return status;
}

/**
  * @brief  define Reverse8 function.
  * @param  None
  * @retval None
  */

uint8_t     Reverse8(uint8_t U8)
{
    U8 = (((U8 & 0xaa) >> 1) | ((U8 & 0x55) << 1));
    U8 = (((U8 & 0xcc) >> 2) | ((U8 & 0x33) << 2));
    U8 = (((U8 & 0xf0) >> 4) | ((U8 & 0x0f) << 4));
    return (U8);
}

/**
  * @brief  define Reverse32 function, used to crc check.
  * @param  None
  * @retval None
  */

uint32_t    Reverse32(uint32_t U32)
{
    U32 = (((U32 & 0xaaaaaaaa) >> 1) | ((U32 & 0x55555555) << 1));
    U32 = (((U32 & 0xcccccccc) >> 2) | ((U32 & 0x33333333) << 2));
    U32 = (((U32 & 0xf0f0f0f0) >> 4) | ((U32 & 0x0f0f0f0f) << 4));
    U32 = (((U32 & 0xff00ff00) >> 8) | ((U32 & 0x00ff00ff) << 8));
    U32 = (((U32 & 0xffff0000) >> 16) | ((U32 & 0x0000ffff) << 16));
    return (U32);
}

/**
  * @brief  define the checksum function, used to crc check.
  * @param  None
  * @retval None
  */
static uint32_t fsm_get_crc32(uint32_t *data_ptr, uint32_t len)
{
    uint32_t    crc32, poly, data32;
    uint8_t     data8;
    uint8_t     ii;
    uint32_t    *pU32Buf;

    pU32Buf = (uint32_t *)((uint32_t)data_ptr + APP_ADDR);
    crc32 = 0xffffffffUL;
    poly = 0x04c11db7;
    len >>= 2;
    while (len--)
    {
        data32 = *pU32Buf++;
        for (ii = 0 ; ii < 4 ; ii++)
        {
            data8 = data32 & 0xff;
            data8 = Reverse8(data8);
            data32 >>= 8;
            data8 ^= (crc32 >> 24);
            crc32 <<= 8;

            if (data8 & 0x80) crc32 ^= 0x690CE0EE; // (poly << 7) ^ (poly << 1)

            if (data8 & 0x40) crc32 ^= 0x34867077; // (poly << 6) ^ (poly << 0)

            if (data8 & 0x20) crc32 ^= 0x9823B6E0; // (poly << 5)

            if (data8 & 0x10) crc32 ^= 0x4C11DB70; // (poly << 4)

            if (data8 & 0x08) crc32 ^= 0x2608EDB8; // (poly << 3)

            if (data8 & 0x04) crc32 ^= 0x130476DC; // (poly << 2)

            if (data8 & 0x02) crc32 ^= 0x09823B6E; // (poly << 1)

            if (data8 & 0x01) crc32 ^= 0x04C11DB7; // (poly << 0)
        }
    }       
    crc32 = ~Reverse32(crc32);

    return crc32;
}

/**
  * @brief  define the check empty function.
  * @param  None
  * @retval None
  */
static fsm_result_t fsm_check_empty(uint32_t *data_ptr, uint32_t len)
{
    uint32_t  i;
//    printf("%s\r\n", __FUNCTION__);
    uint32_t *temp_data_ptr;
    uint32_t temp_len;

    temp_data_ptr = (uint32_t *)((uint32_t)data_ptr + APP_ADDR);
    temp_len = (len - APP_ADDR);

    if ((((uint32_t)temp_data_ptr) % 4 != 0) || (temp_len % 4 != 0))
    {
        return FAIL;
    }

    __disable_irq();

    for (i = 0; i < temp_len; i += 4)
    {
        if (*(temp_data_ptr++) != 0xFFFFFFFF)
        {
            break;
        }
    }

    __enable_irq();

    if (i < temp_len)
        return FAIL;
    else
        return PASS;
}

/**
  * @brief  define the memory read function.
  * @param  None
  * @retval None
  */
static void fsm_read_memo(uint8_t *dest, void *src, uint32_t len)
{
//    printf("%s", __FUNCTION__);

    __disable_irq();

    memcpy(dest, (const void *)((uint32_t)src + APP_ADDR), len);

    __enable_irq();

    return;
}

/**
  * @brief  check specified checksum function.
  * @param  None
  * @retval None
  */
uint8_t verify_valid(void)
{
//    uint32_t  len_hex, crc_cal_result, crc_hex_result;
//    uint32_t i;
//    uint32_t *data_ptr = (uint32_t *)APP_ADDR;
//    
//    if (!md_crc_is_enabled(CRC))
//        init_crc32();
//    
//    if(CRC_CAL_ADDR & 0x03)
//        return FAIL;

//    /* get info from specified address */
//    crc_hex_result = *((uint32_t *)CRC_CAL_ADDR); 
//    len_hex = *((uint32_t *)CRC_CAL_ADDR + 1);
//    
//    if (len_hex > 0x7E000 || len_hex & 0x03 || len_hex == 0)
//            return FAIL;

//    __disable_irq();
//    SYSCFG_UNLOCK();
//    /* set to app flash*/
//    CLEAR_BIT(SYSCFG->MEMRMP, SYSCFG_MEMRMP_BFRMPEN_MSK);
//    SYSCFG_LOCK();

//#ifdef __IAR_SYSTEMS_ICC__  /* IAR Compiler */
//    /*===========================================================================================*/
//    /* MISRA violation of rule 11.4 17.4 -pointer arithmetic is used for CRC calculation */
//#pragma diag_suppress=Pm088,Pm141
//#endif  /* __IAR_SYSTEMS_ICC__ */
//    
//    for (i = 0; i < len_hex / 4; i++)
//        md_crc_write_data(CRC, *data_ptr++);
//        crc_cal_result = md_crc_get_check_result(CRC);

//#ifdef __IAR_SYSTEMS_ICC__  /* IAR Compiler */
//    /*===========================================================================================*/
//    /* MISRA violation of rule 11.4 17.4 -pointer arithmetic is used for CRC calculation */
//#pragma diag_suppress=Pm088,Pm141
//#endif  /* __IAR_SYSTEMS_ICC__ */

//    SYSCFG_UNLOCK();
//    /* set to boot flash*/
//    SET_BIT(SYSCFG->MEMRMP, SYSCFG_MEMRMP_BFRMPEN_MSK);
//    SYSCFG_LOCK();
//    __enable_irq();
//    
//    if (crc_hex_result == crc_cal_result)
//    {
//        return PASS;
//    }
//    else
//    {
//        MD_CRC_RESET();
//        MD_CRC_DISABLE();
//        return FAIL;
//    }

    uint32_t  len_hex, crc_cal_result, crc_hex_result;
    uint32_t i;
    uint32_t *data_ptr = (uint32_t *)0x00000000;

    crc_hex_result = *((uint32_t *)(CRC_CAL_ADDR));     
    len_hex = CRC_CAL_ADDR-APP_ADDR; 
    crc_cal_result = fsm_get_crc32(data_ptr, len_hex); 
    
    printf("crc_hex_result:%x\r\n",crc_hex_result);
    printf("crc_cal_result:%x\r\n",crc_cal_result);
    printf("\r\n");    

    if(crc_hex_result == crc_cal_result)
    {
        return PASS;        
    }
    else
    {
        return FAIL;
    }
} 

/**
  * @brief  define the function used to jump to app program.
  * @param  None
  * @retval None
  */
static void fsm_go(uint32_t para)
{
    uint32_t addr;    
//    printf("%s", __FUNCTION__);

    // Lock FC_CMD Register
    md_fc_lock();
    
    if(para == GO_APP)
    {
        addr = APP_ADDR;
    } 
    else if(para == GO_BOOT) 
    {
        addr = BOOT_ADDR;
    }    

    // REMAP
    // 0x00: EFLASH         -> Remap EFLASH Any 4K
    // 0x01: Bootloader     -> Remap EFLASH 0x00000000
    // 0x02: SRAM           -> Remap SRAM

    // Remap EFLASH
    md_syscfg_set_memory_mapping(SYSCFG, MD_SYSCFG_MEMMOD_MAIN);
    md_syscfg_set_flash_remap_base(SYSCFG, addr / (4 * 2 * MD_FC_PC_EF_MPAGESZ));
    md_syscfg_enable_memory_remap(SYSCFG);
    // Reset MCU
    NVIC_SystemReset();

    while (1);

    return;
}

/**
  * @brief  assign function pointer to all the state machine subfunction.
  * @param  None
  * @retval None
  */
void fsm_exec_func_init(void)
{
    g_isp_data.p_page_erase   = &fsm_page_erase;
    g_isp_data.p_words_progrm = &fsm_words_program;
    g_isp_data.p_get_crc32    = &fsm_get_crc32;
    g_isp_data.p_check_empty  = &fsm_check_empty;
    g_isp_data.p_go           = &fsm_go;
    g_isp_data.p_read_memo    = &fsm_read_memo;
}

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
