/**********************************************************************************
 *
 * @file    lin.c
 * @brief   Main file for lin module
 *
 * @date    23 Sep 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          23 Sep 2021     biyq          the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */
#include "lin.h"

/* Private Macros ------------------------------------------------------------ */

/* Private Variables --------------------------------------------------------- */

/* Public Variables ---------------------------------------------------------- */

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Public Function ---------------------------------------------------------- */

/**
  * @brief:  LIN Bus state change.
  * @param:  None
  * @retval: None
  */
void lin_chstat(lin_node_t *lin_node, node_state_t stat)
{
    lin_node->node_state = stat;

    switch (lin_node->node_state)
    {
        case sleep:
            lin_node->node_enter_sleep();
            break;

        case standby:
            break;

        case normal_slope:
            lin_node->node_exit_sleep();
            lin_node->node_normal_slope_mode();
            break;

        case low_slope:
            lin_node->node_exit_sleep();
            lin_node->node_low_slope_mode();
            break;

        default:
            break;
    }

    return;
}

/**
  * @brief:  Change LIN frame state.
  * @param:  None
  * @retval: None
  */
void lin_chfstat(lin_node_t *lin_node, frame_status_t status)
{
    lin_node->frame_status = status;

    return;
}

/**
  * @brief:  LIN send data.
  * @param:  None
  * @retval: None
  */
uint8_t lin_pid_cal(uint8_t raw_pid)
{
    uint8_t p0;
    uint8_t p1;

    if (raw_pid >= 0x3e)
    {
        /* 0x3e and 0x3f are reserved id, and the valid range of proteced id is from 0x00 to 0x3f */

        return 1;
    }
    else
    {
        /* other valid frame uses the enhanced checksum */
        p0 = ((CAL_BIT(raw_pid, 0) ^ CAL_BIT(raw_pid, 1) ^ CAL_BIT(raw_pid, 2) ^ CAL_BIT(raw_pid, 4)) & 0x01) << 6;
        p1 = (~(CAL_BIT(raw_pid, 1) ^ CAL_BIT(raw_pid, 3) ^ CAL_BIT(raw_pid, 4) ^ CAL_BIT(raw_pid, 5)) & 0x01) << 7;

        return ((raw_pid & 0x3F) | p0 | p1);
    }
}

/**
  * @brief:  Calculate checksum.
  * @param:  None
  * @retval: checksum
  */
uint8_t lin_cal_checksum(lin_node_t *lin_node, uint8_t *buf)
{
    uint32_t i;

    /* if lin node is a slave node, config the checksum initial value as pid */
    if (lin_node->node_mode == slave)
        lin_node->checksum = lin_pid_cal(lin_node->id);

    for (i = 0; i < lin_node->data_len; i++)
    {
        lin_node->checksum += buf[i];

        if (lin_node->checksum > 0xFF)
        {
            lin_node->checksum &= 0xFFUL;
            lin_node->checksum++;
        }
    }

    lin_node->checksum = 0xFF - lin_node->checksum;

    return lin_node->checksum;
}

/**
  * @brief:  process the next logic after detect header break.
  * @param:  None
  * @retval: None
  */
void lin_break_seg_proc(lin_node_t *lin_node)
{
    lin_node->node_recv_ctrl(lin_node);

    if (lin_node->node_mode == master && lin_node->dir == l_read)
    {
        lin_chfstat(lin_node, sync_stage);
    }
    else if (lin_node->node_mode == slave && lin_node->dir == l_read)
    {
        lin_chfstat(lin_node, sync_stage);
    }
    else if (lin_node->node_mode == slave && lin_node->dir == l_write)
    {
        lin_chfstat(lin_node, sync_stage);
    }
    else
    {
        /**************/
    }
}

/**
  * @brief:  LIN send header.
  * @param:  None
  * @retval: None
  */
uint8_t lin_sendheader(lin_node_t *lin_node)
{
    uint8_t pid;

    /* Reset send result and lin break flag */
    lin_node->send_result = pass;
    /* Send lin break requirement */
    lin_node->node_send_bk_req();

    /* Send sync seg */
    lin_node->node_send_byte(SYNC_SEG);

    /* Send PID */
    pid = lin_pid_cal(lin_node->id);

    if (lin_node->id == 0x3c || lin_node->id == 0x3d)
    {
        /* 0x3c and 0x3d are used for diagrequest and diagresponse */
        lin_node->checksum = 0;
    }
    else if (lin_node->id >= 0x3e)
    {
        /* 0x3e and 0x3f are reserved id, and the valid range of proteced id is from 0x00 to 0x3f */
        lin_node->send_result = fail;

        return 1;
    }
    else
    {
        /* other valid frame uses the enhanced checksum */
        lin_node->checksum = pid;
    }

    lin_node->node_send_byte(pid);

    return 0;
}

/**
  * @brief:  send answer to the lin master node.
  * @param:  None
  * @retval: None
  */
void lin_send_answer(lin_node_t *lin_node, uint8_t *buf)
{
    uint32_t i;

    /* Change data tranfer direction to inidicate lin master node start a read operation */
    lin_node->dir = l_read;
    /* Reset send result to on_going */
    lin_node->send_result = on_going;

    /* Send data */
    for (i = 0; i < lin_node->data_len; i++)
    {
        lin_node->node_send_byte(*(buf + i));
    }

    /* Calculate checksum and send checksum*/
    lin_cal_checksum(lin_node, buf);
    lin_node->node_send_byte(lin_node->checksum);

    /* Change send answer result as pass */
    lin_chfstat(lin_node, breakfield_stage);
    lin_node->send_result = pass;

    return;
}

/**
  * @brief:  LIN send frame.
  * @param:  None
  * @retval: None
  */
void lin_sendframe(lin_node_t *lin_node, uint8_t *buf, uint32_t len)
{
    uint32_t i;

    /* Change data tranfer direction to write */
    lin_node->dir = l_write;
    /* Reset send result to on_going */
    lin_node->send_result = on_going;

    /* if master mode, then send frame header */
    if (lin_node->node_mode)
    {
        if (0x01 == lin_sendheader(lin_node))
        {
            lin_node->send_result = fail;
            return;
        }
    }
    else
    {
        /**********/
    }

    lin_node->data_len = len;

    /* Send data */
    for (i = 0; i < lin_node->data_len; i++)
    {
        lin_node->node_send_byte(*(buf + i));
    }

    /* Calculate checksum and send checksum*/
    lin_cal_checksum(lin_node, buf);
    lin_node->node_send_byte(lin_node->checksum);

    /* Change data tranfer direction to read */
    lin_node->send_result = pass;

    return;
}

/**
  * @brief:  LIN receive frame.
  * @param:  None
  * @retval: None
  */
void lin_recvframe(lin_node_t *lin_node, uint32_t len)
{
    /* Check if current node is master in lin */
    if (lin_node->node_mode == slave)
    {
        lin_node->recv_result = fail;
        return;
    }

    /* Prepare for receiving */
    lin_node->node_recv_ctrl(lin_node);
    /* Change data tranfer direction to write */
    lin_node->dir = l_read;

    /* Reset send result to on_going */
    if (lin_node->recv_result == on_going)
    {
        /* last operation has not been completed */
        lin_node->recv_result = fail;
        return;
    }
    else
    {
        lin_node->recv_result = on_going;
    }

    /* if master mode, then send frame header */
    if (lin_node->node_mode)
    {
        if (0x01 == lin_sendheader(lin_node))
        {
            lin_node->recv_result = fail;
            return;
        }
    }
    else
    {
        /**********/
    }

    lin_node->data_len = len;
    /*
    while (lin_node->data_len && lin_node->recv_result == on_going);
    */
    lin_node->recv_result = pass;

    return;
}

/**
  * @brief:  process lin frame.
  * @param:  None
  * @retval: None
  */
void lin_recv_data_proc(lin_node_t *lin_node, uint8_t data)
{
    static uint8_t len = 0;

    /* LIN master node start a read operation or slave node need to receive data from master */
    if ((lin_node->node_mode == slave && lin_node->dir == l_write) || (lin_node->node_mode == master && lin_node->dir == l_read))
    {
        switch (lin_node->frame_status)
        {
            case (breakfield_stage):
                if (data == LBRK_SEG)
                {
                    lin_chfstat(lin_node, sync_stage);
                }
                else
                {
                    lin_node->recv_result = fail;

                    if (lin_node->node_mode == slave)
                        lin_node->node_wait_bk();
                }

                break;

            case (sync_stage):
                if (data == SYNC_SEG)
                {
                    lin_chfstat(lin_node, pid_stage);
                }
                else
                {
                    lin_chfstat(lin_node, breakfield_stage);
                    lin_node->recv_result = fail;

                    if (lin_node->node_mode == slave)
                        lin_node->node_wait_bk();
                }

                break;

            case (pid_stage):
                if (data == lin_pid_cal(lin_node->id))
                {
                    lin_chfstat(lin_node, data_stage);
                }
                else
                {
                    lin_chfstat(lin_node, breakfield_stage);
                    lin_node->recv_result = fail;

                    if (lin_node->node_mode == slave)
                        lin_node->node_wait_bk();
                }

                break;

            case (data_stage):
                lin_node->trans_buf[len++] = data;

                if (lin_node->data_len == len)
                {
                    lin_chfstat(lin_node, checksum_stage);
                    len = 0;
                }

                break;

            case (checksum_stage):
                if (lin_cal_checksum(lin_node, lin_node->trans_buf) == data)
                    lin_node->recv_result = pass;
                else
                    lin_node->recv_result = fail;

                lin_chfstat(lin_node, breakfield_stage);

                if (lin_node->node_mode == slave)
                    lin_node->node_wait_bk();

                break;

            default:
                lin_chfstat(lin_node, breakfield_stage);

                if (lin_node->node_mode == slave)
                    lin_node->node_wait_bk();

                break;
        }
    }
    else if ((lin_node->node_mode == slave && lin_node->dir == l_read))
    {
        switch (lin_node->frame_status)
        {
            case (breakfield_stage):
                if (data == LBRK_SEG)
                {
                    lin_chfstat(lin_node, sync_stage);
                }
                else
                {
                    lin_node->send_result = fail;
                    lin_node->node_wait_bk();
                }

                break;

            case (sync_stage):
                if (data == SYNC_SEG)
                {
                    lin_chfstat(lin_node, pid_stage);
                }
                else
                {
                    lin_node->send_result = fail;
                    lin_chfstat(lin_node, breakfield_stage);
                    lin_node->node_wait_bk();
                }

                break;

            case (pid_stage):
                if (data == lin_pid_cal(lin_node->id))
                {
                    lin_chfstat(lin_node, data_stage);
                }
                else
                {
                    lin_node->send_result = fail;
                    lin_chfstat(lin_node, breakfield_stage);
                    lin_node->node_wait_bk();
                }

                /* Send answer */
                lin_send_answer(lin_node, lin_node->trans_buf);
                lin_node->node_wait_bk();
                break;

            case (checksum_stage):
            default:
                lin_chfstat(lin_node, breakfield_stage);
                lin_node->node_wait_bk();
                break;
        }
    }
}

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
