/**********************************************************************************
 *
 * @file    main.c
 * @brief   Main file for demo
 *
 * @date    23 Sep 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          23 Sep 2021     biyq          the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */

#include <string.h>
#include "main.h"

/* Private Macros ------------------------------------------------------------ */

/* Private Variables --------------------------------------------------------- */
uint8_t ref_buf[8] = {0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 0x88};
uint8_t tx_buf[8] = {0x88, 0x77, 0x66, 0x55, 0x44, 0x33, 0x22, 0x11};
uint8_t rx_buf[8];
uint8_t rx_index;

/* Public Variables ---------------------------------------------------------- */
md_usart_init_t g_h_usart;
lin_node_t node1;

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */
void uart_send_byte(uint8_t u8data);
void uart_recv_ctrl(lin_node_t *lin_node);
void lin_select_normal_slope(void);
void lin_select_low_slope(void);
void lin_exit_sleep(void);
void lin_enter_sleep(void);
void lin_wait_bk(void);


/* Public Function ---------------------------------------------------------- */
void SystemInit()
{}

/**
  * @brief  This function configures the source of the time base.
  *         The time source is configured to have 1ms time base with a dedicated
  *         Tick interrupt priority.
  * @retval None
  */
void md_init_10us_tick(void)
{
    /* Configure the SysTick IRQ */
    SysTick_Config(md_cmu_get_clock() / 100000);
    NVIC_SetPriority(SysTick_IRQn, 3);
    return;
}

/* Private Function ---------------------------------------------------------- */
/**
  * @brief  Init LIN pin
  * @param:  None
  * @retval None
  */
void lin_pin_init(void)
{
    md_gpio_init_t x;
    md_gpio_init_struct(&x);

    /* Initialize tx pin: PA13 */
    x.mode  = MD_GPIO_MODE_OUTPUT;
    x.odos  = MD_GPIO_OPEN_DRAIN;
    x.pupd  = MD_GPIO_PUSH_UP;
    x.odrv = MD_GPIO_OUT_DRIVE_NORMAL;
    x.flt   = MD_GPIO_FILTER_DISABLE;
    x.type  = MD_GPIO_TYPE_CMOS;
    x.func  = MD_GPIO_FUNC_3;
    md_gpio_init(LIN0_PORT, LIN0_TX_PIN, &x);

    /* Initialize rx pin: PA14(initiated for io firstly) */
    x.mode  = MD_GPIO_MODE_INPUT;
    x.pupd = MD_GPIO_PUSH_UP;
    x.flt   = MD_GPIO_FILTER_ENABLE;
    x.func  = MD_GPIO_FUNC_1;
    md_gpio_init(LIN0_PORT, LIN0_RX_PIN, &x);

    /* Initialize inh pin: PB6 */
    x.flt   = MD_GPIO_FILTER_DISABLE;
    x.mode  = MD_GPIO_MODE_INPUT;
    x.pupd  = MD_GPIO_PUSH_DOWN;
    x.func  = MD_GPIO_FUNC_1;
    md_gpio_init(LIN0_PORT, LIN0_INH_PIN, &x);

    /* Initialize slp pin: PB7 */
    x.mode  = MD_GPIO_MODE_OUTPUT;
    x.func  = MD_GPIO_FUNC_1;
    md_gpio_init(LIN0_PORT, LIN0_SLP_PIN, &x);
}

/**
  * @brief:  Initialize the lin.
  * @param:  None
  * @retval: None
  */
void lin_init(void)
{
    /******************** Initialize exti *********************/
    md_gpio_set_interrupt_port(LIN0_PORT, LIN0_RX_PIN);      /* set exit interrupt port */
    md_gpio_enable_trailing_edge_trigger(LIN0_RX_PIN);        /* enable trailing edge trigger */
    md_gpio_enable_external_interrupt(LIN0_RX_PIN);         /* enable exti */
    /* Enable EXTI12_15_IRQn irq */
    md_mcu_irq_config(EXTI12_15_IRQn, 0, ENABLE);

    /******************** Initialize USART0 *********************/
    md_usart_init_struct(&g_h_usart);
    g_h_usart.baud        = BAUDRATE;
    g_h_usart.word_length = MD_USART_WORD_LENGTH_8B;
    g_h_usart.stop_bits   = MD_USART_STOP_BITS_1;
    g_h_usart.parity      = MD_USART_PARITY_NONE;
    g_h_usart.fctl        = MD_USART_HW_FLOW_CTL_NONE;
    g_h_usart.mode        = MD_USART_MODE_TX_RX;
    md_usart_init(USART0, &g_h_usart);

    /* Enable rxne irq and error frame irq */
    md_usart_enable_it_rxne(USART0);
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
    /* Configure system clock */
    md_cmu_clock_config(MD_CMU_CLOCK_HRC, 32000000);
    /* Initialize SysTick Interrupt */
    md_init_10us_tick();

    /* Enable ALL peripheral */
    SYSCFG_UNLOCK();
    md_cmu_enable_perh_all();
    SYSCFG_LOCK();

    /* Init Lin node */
    memset(&node1, 0x00, sizeof(lin_node_t));
    node1.baud = BAUDRATE;
    node1.id = 0x07;
    node1.sync_field = SYNC_SEG;
    node1.node_mode = slave;
    node1.node_send_byte = uart_send_byte;
    node1.node_recv_ctrl = uart_recv_ctrl;
    node1.node_normal_slope_mode = lin_select_normal_slope;
    node1.node_low_slope_mode = lin_select_low_slope;
    node1.node_enter_sleep = lin_enter_sleep;
    node1.node_exit_sleep = lin_exit_sleep;
    node1.node_wait_bk = lin_wait_bk;
    node1.dir = l_write;        /* Lin master write */
    node1.data_len = 8;     /* Lin slave node wait for receiving 8 bytes */

    /* Init Lin config */
    lin_pin_init();
    lin_init();
    /* Due to the fact that tx pin has been pushed up in the above step, so we can only select normal slope */
    lin_chstat(&node1, normal_slope);

    while (1)
    {
        /* strongly recommend to clear idle flag in main loop */
        if (md_usart_is_active_flag_idle(USART0))
            md_usart_clear_flag_idle(USART0);

        if (!memcmp(node1.trans_buf, ref_buf, 8))
        {
            /* Lin master read */
            node1.dir = l_read;
            memcpy(node1.trans_buf, tx_buf, 8);
        }
    }
}

/**
  * @brief  uart send data
  * @retval None
  */
void uart_send_byte(uint8_t u8data)
{
    md_usart_send_data8(USART0, u8data);

    while (!md_usart_is_active_flag_txc(USART0));

    md_usart_clear_flag_txc(USART0);

    return;
}

/**
  * @brief  uart receive irq contrl
  * @retval None
  */
void uart_recv_ctrl(lin_node_t *lin_node)
{
    if (lin_node->node_mode == master && lin_node->dir == l_read)
    {
        md_usart_enable_rx(USART0);
        md_usart_disable_tx(USART0);
    }
    else if (lin_node->node_mode == slave && lin_node->dir == l_write)
    {
        md_usart_enable_rx(USART0);
        md_usart_disable_tx(USART0);
    }
    else if (lin_node->node_mode == slave && lin_node->dir == l_read)
    {
        md_usart_enable_rx(USART0);
        md_usart_enable_tx(USART0);
    }
    else
    {
        md_usart_disable_tx(USART0);
        md_usart_disable_rx(USART0);
    }

    return;
}

/**
  * @brief  use normal slope mode
  * @retval None
  */
void lin_exit_sleep(void)
{
    md_gpio_write_pin(LIN0_PORT, LIN0_SLP_PIN, 1);

    return;
}

/**
  * @brief  use normal slope mode
  * @retval None
  */
void lin_enter_sleep(void)
{
    md_gpio_write_pin(LIN0_PORT, LIN0_SLP_PIN, 0);

    return;
}

/**
  * @brief  use normal slope mode
  * @retval None
  */
void lin_select_normal_slope(void)
{

    return;
}

/**
  * @brief  use low slope mode
  * @retval None
  */

void lin_select_low_slope(void)
{

    return;
}

/**
  * @brief  lin slave node wait for bk
  * @retval None
  */
void lin_wait_bk(void)
{
    md_gpio_init_t x;

    x.mode  = MD_GPIO_MODE_INPUT;
    x.odos  = MD_GPIO_OPEN_DRAIN;
    x.pupd  = MD_GPIO_PUSH_UP;
    x.odrv = MD_GPIO_OUT_DRIVE_NORMAL;
    x.flt   = MD_GPIO_FILTER_ENABLE;
    x.type  = MD_GPIO_TYPE_CMOS;
    x.func  = MD_GPIO_FUNC_1;
    md_gpio_init(LIN0_PORT, LIN0_RX_PIN, &x);

    /* Disable USART0 irq */
    md_usart_recv_data8(USART0);
    md_mcu_irq_config(USART0_IRQn, 0, DISABLE);

    /* enable trailing edge trigger */
    md_gpio_disable_riging_edge_trigger(LIN0_RX_PIN);
    md_gpio_enable_trailing_edge_trigger(LIN0_RX_PIN);

    return;
}

/**
  * @brief  This function is designed to calculate the time difference of breakfield, timebase is 10us.
  * @param: baud rate
  * @retval time difference of one bit
  */
uint32_t time_difference_cal(uint32_t baudrate)
{
    if (baudrate == 0)
        return 0;
    else
        return (100000 / baudrate * 13);
}

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
