/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    30 Apri 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          30 Apri 2021    yanght          the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */
/* Includes ------------------------------------------------------------------ */
#include "main.h"

/** @addtogroup Projects_Examples_MD
  * @{
  */

/** @addtogroup Examples
  * @{
  */
/* Public Variables ---------------------------------------------------------- */
md_usart_init_t g_h_usart;

uint8_t g_tx_buf[512];
uint16_t g_tx_i = 0U;

uint8_t g_rx_buf[10];
uint8_t g_rx_i = 0U;

uint32_t g_data_addr;
uint8_t g_data_len;

uint8_t g_frame_flag = 0U;

/* Private Macros ------------------------------------------------------------ */
/* Private Variables --------------------------------------------------------- */
/* Private Constants --------------------------------------------------------- */
/* Private function prototypes ----------------------------------------------- */
/* Private Function ---------------------------------------------------------- */
/** @addtogroup Projects_Examples_MD
  * @{
  */

/** @addtogroup Examples
  * @{
  */

/**
  * @brief  Initializate system.
  * @retval None
  */
void SystemInit(void)
{

}

/**
  * @brief  Initializate pin of usart module.
  * @retval None
  */
static void usart_pin_init(void)
{
    md_gpio_init_t x;
    md_gpio_init_struct(&x);

    /* Initialize tx pin */
    x.mode = MD_GPIO_MODE_OUTPUT;
    x.odos = MD_GPIO_PUSH_PULL;
    x.pupd = MD_GPIO_PUSH_UP;
    x.odrv = MD_GPIO_OUT_DRIVE_NORMAL;
    x.flt  = MD_GPIO_FILTER_DISABLE;
    x.type = MD_GPIO_TYPE_CMOS;
    x.func = MD_GPIO_FUNC_3;
    md_gpio_init(USART0_TX_PORT, USART0_TX_PIN, &x);

    /* Initialize rx pin */
    x.mode = MD_GPIO_MODE_INPUT;
    x.odos = MD_GPIO_PUSH_PULL;
    x.pupd = MD_GPIO_PUSH_UP;
    x.odrv = MD_GPIO_OUT_DRIVE_NORMAL;
    x.flt  = MD_GPIO_FILTER_DISABLE;
    x.type = MD_GPIO_TYPE_CMOS;
    x.func = MD_GPIO_FUNC_3;
    md_gpio_init(USART0_RX_PORT, USART0_RX_PIN, &x);

    return;
}

/**
  * @brief:  Initialize the usart.
  * @param:  None
  * @retval: None
  */
static void usart_init(void)
{
    md_usart_init_struct(&g_h_usart);
    /* Initialize USART0 */
    g_h_usart.baud        = 115200;
    g_h_usart.word_length = MD_USART_WORD_LENGTH_8B;
    g_h_usart.stop_bits   = MD_USART_STOP_BITS_1;
    g_h_usart.parity      = MD_USART_PARITY_NONE;
    g_h_usart.fctl        = MD_USART_HW_FLOW_CTL_NONE;
    g_h_usart.mode        = MD_USART_MODE_TX_RX;
    md_usart_init(USART0, &g_h_usart);

    md_usart_clear_flag_txc(USART0);
    md_usart_enable_it_txc(USART0);
    md_usart_enable_it_rxne(USART0);
}


uint8_t ask_data_handle(void)
{
    if (g_rx_buf[0] != STA1 || g_rx_buf[1] != STA2 || g_rx_buf[2] != FRAME_LEN || g_rx_buf[8] != END1 || g_rx_buf[9] != END2)
    {
        return 0;
    }

    g_data_addr = (g_rx_buf[3] << 24) | (g_rx_buf[4] << 16) | (g_rx_buf[5] << 8) | (g_rx_buf[6]);
    g_data_len = g_rx_buf[7];

    return 1;
}

void answer_data_send(void)
{
    int i, j, k = 0;
    uint8_t data_buf[512];

    memcpy(data_buf, (void *)g_data_addr, g_data_len);

    g_tx_buf[0] = STA1;
    g_tx_buf[1] = STA2;
    g_tx_buf[2] = FRAME_LEN + g_data_len;
    g_tx_buf[3] = g_rx_buf[3];
    g_tx_buf[4] = g_rx_buf[4];
    g_tx_buf[5] = g_rx_buf[5];
    g_tx_buf[6] = g_rx_buf[6];
    g_tx_buf[7] = g_data_len;

    for (i = 0; i < (g_data_len / 4) ; i++)
    {
        for (j = 3 ; j >= 0 ; j--)
        {
            g_tx_buf[8 + i * 4 + j ] = data_buf[k++];
        }
    }

    for (i = 0; i < (g_data_len % 4); i++)
    {
        g_tx_buf[8  + g_data_len - i - 1] = data_buf[k++];
    }

    g_tx_buf[8 + g_data_len] = END1;
    g_tx_buf[9 + g_data_len] = END2;

    md_usart_send_data8(USART0, g_tx_buf[0]);
}


/**
  * @brief:  Main program.
  * @param:  None
  * @retval: None
  */
int main()
{
    /* Configure system clock */
    md_cmu_clock_config(MD_CMU_CLOCK_HRC, 32000000);
    /* Initialize SysTick Interrupt */
    md_init_1ms_tick();

    /* Enable ALL peripheral */
    SYSCFG_UNLOCK();
    md_cmu_enable_perh_all();
    SYSCFG_LOCK();
    /* Enable and Set priority for USART_IRQn */
    md_mcu_irq_config(USART0_IRQn, 1, ENABLE);

    usart_pin_init();

    /* Init USART */
    usart_init();

    while (1)
    {
        /* strongly recommend to clear idle flag in main loop */
        if (md_usart_is_active_flag_idle(USART0))
            md_usart_clear_flag_idle(USART0);

        if (g_frame_flag == 1)
        {
            if (ask_data_handle() == 1)
            {
                answer_data_send();
            }

            g_frame_flag = 0;
            memset(g_rx_buf, 0, 10);
            g_tx_i = 0;
        }
    }
}

/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
