/**********************************************************************************
 *
 * @file    main.c
 * @brief   Main file for DEMO
 *
 * @date    30 Apri 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          30 Apri 2021    yanght          the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */

#include <string.h>
#include "main.h"
void SystemInit()
{

}
/* Private Macros ------------------------------------------------------------ */

/* Private Variables --------------------------------------------------------- */

/* Public Variables ---------------------------------------------------------- */

/* The lower 9 bits of DMA_CTRLBASE regisitor must be 0 */
md_dma_descriptor_t g_dma0_ctrl_base[2] __attribute__((aligned(512)));
md_dma_config_t g_tx_config, g_rx_config;

md_usart_init_t g_h_usart;
uint8_t g_rx_buf[32];

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Function ---------------------------------------------------------- */

/** @addtogroup Projects_Examples_MD
  * @{
  */

/** @addtogroup Examples
  * @{
  */

/**
  * @brief  Initializate pin of uart module.
  * @retval None
  */
static void usart_pin_init(void)
{
    md_gpio_init_t x;
    md_gpio_init_struct(&x);

    /* Initialize tx pin */
    x.mode = MD_GPIO_MODE_OUTPUT;
    x.odos = MD_GPIO_PUSH_PULL;
    x.pupd = MD_GPIO_PUSH_UP;
    x.odrv = MD_GPIO_OUT_DRIVE_NORMAL;
    x.flt  = MD_GPIO_FILTER_DISABLE;
    x.type = MD_GPIO_TYPE_CMOS;
    x.func = MD_GPIO_FUNC_3;
    md_gpio_init(USART0_TX_PORT, USART0_TX_PIN, &x);

    /* Initialize rx pin */
    x.mode = MD_GPIO_MODE_INPUT;
    x.odos = MD_GPIO_PUSH_PULL;
    x.pupd = MD_GPIO_PUSH_UP;
    x.odrv = MD_GPIO_OUT_DRIVE_NORMAL;
    x.flt  = MD_GPIO_FILTER_DISABLE;
    x.type = MD_GPIO_TYPE_CMOS;
    x.func = MD_GPIO_FUNC_3;
    md_gpio_init(USART0_RX_PORT, USART0_RX_PIN, &x);

    return;
}

/**
  * @brief:  Initialize the usart.
  * @param:  None
  * @retval: None
  */
static void usart_init(void)
{
    md_usart_init_struct(&g_h_usart);
    /* Initialize USART0 */
    g_h_usart.baud        = 115200;
    g_h_usart.word_length = MD_USART_WORD_LENGTH_8B;
    g_h_usart.stop_bits   = MD_USART_STOP_BITS_1;
    g_h_usart.parity      = MD_USART_PARITY_NONE;
    g_h_usart.fctl        = MD_USART_HW_FLOW_CTL_NONE;
    g_h_usart.mode        = MD_USART_MODE_TX_RX;
    md_usart_init(USART0, &g_h_usart);

}

/**
  * @brief  dma_start_tx init.
  * @param  None
  * @retval None
  */
void dma_send_tx(DMA_TypeDef *DMAx, USART_TypeDef *USARTx, uint8_t *buf, uint16_t size, uint8_t channel)
{
    memset(&g_tx_config, 0x0, sizeof(g_tx_config));
    md_dma_set_ctrlbase((uint32_t)&g_dma0_ctrl_base);

    md_dma_enable();
    md_dma_config_struct(&g_tx_config);
    g_tx_config.primary    = ENABLE;
    g_tx_config.channel    = channel;
    g_tx_config.burst      = ENABLE;
    g_tx_config.high_prio  = DISABLE;
    g_tx_config.src        = (void *)(buf);
    g_tx_config.dst        = (void *)(&USARTx->DATA);
    g_tx_config.src_inc    = MD_DMA_DATA_INC_BYTE;
    g_tx_config.dst_inc    = MD_DMA_DATA_INC_NONE;
    g_tx_config.data_width = MD_DMA_DATA_SIZE_BYTE;
    g_tx_config.R_power    = MD_DMA_R_POWER_1;
    g_tx_config.size       = size;
    g_tx_config.msel       = USARTx == USART0 ? MD_DMA_MSEL_USART0 : MD_DMA_MSEL_USART1;
    g_tx_config.msigsel    = MD_DMA_MSIGSEL_USART_TXEMPTY;
    g_tx_config.interrupt  = ENABLE;	
	
    if ((g_h_usart.word_length == MD_USART_WORD_LENGTH_9B) && (g_h_usart.parity == MD_USART_PARITY_NONE))
    {
        g_tx_config.src_inc    = MD_DMA_DATA_INC_HALFWORD;
        g_tx_config.data_width = MD_DMA_DATA_SIZE_HALFWORD;
    }

    md_dma_config_base(DMAx, MD_DMA_CYCLE_CTRL_BASIC, &g_tx_config);

    return;
}

/**
  * @brief  dma_start_rx init.
  * @param  None
  * @retval None
  */
void dma_recv_rx(DMA_TypeDef *DMAx, USART_TypeDef *USARTx, uint8_t *buf, uint16_t size, uint8_t channel)
{
    memset(&g_rx_config, 0x0, sizeof(g_rx_config));
    md_dma_set_ctrlbase((uint32_t)&g_dma0_ctrl_base);

    md_dma_enable();
    md_dma_config_struct(&g_rx_config);
    g_rx_config.primary    = ENABLE;
    g_rx_config.channel    = channel;
    g_rx_config.burst      = ENABLE;
    g_rx_config.high_prio  = DISABLE;
    g_rx_config.src        = (void *)(&USARTx->DATA);
    g_rx_config.dst        = (void *)(buf);
    g_rx_config.src_inc    = MD_DMA_DATA_INC_NONE;
    g_rx_config.dst_inc    = MD_DMA_DATA_INC_BYTE;
    g_rx_config.data_width = MD_DMA_DATA_SIZE_BYTE;
    g_rx_config.R_power    = MD_DMA_R_POWER_1;
    g_rx_config.size       = size;
    g_rx_config.msel       = USARTx == USART0 ? MD_DMA_MSEL_USART0 : MD_DMA_MSEL_USART1;
    g_rx_config.msigsel    = MD_DMA_MSIGSEL_USART_RNR;
    g_rx_config.interrupt  = ENABLE;
	
    if ((g_h_usart.word_length == MD_USART_WORD_LENGTH_9B) && (g_h_usart.parity == MD_USART_PARITY_NONE))
    {
        g_rx_config.dst_inc    = MD_DMA_DATA_INC_HALFWORD;
        g_rx_config.data_width = MD_DMA_DATA_SIZE_HALFWORD;
    }

    md_dma_config_base(DMAx, MD_DMA_CYCLE_CTRL_BASIC, &g_rx_config);

    return;
}

/**
  * @brief:  Main program.
  * @param:  None
  * @retval: None
  */
int main()
{
    /* Configure system clock */
    md_cmu_clock_config(MD_CMU_CLOCK_HRC, 32000000);
    /* Initialize SysTick Interrupt */
    md_init_1ms_tick();

    /* Enable ALL peripheral */
    SYSCFG_UNLOCK();
    md_cmu_enable_perh_all();
    SYSCFG_LOCK();

    /* Init USART */
    usart_pin_init();
    usart_init();

    NVIC_SetPriority(DMA_IRQn, 2);
    NVIC_EnableIRQ(DMA_IRQn);
    memset(g_dma0_ctrl_base, 0x0, sizeof(g_dma0_ctrl_base));

    /* DMA receive config */
    dma_recv_rx(DMA0, USART0, g_rx_buf, 32, 1);
    /* DMA send config */
    dma_send_tx(DMA0, USART0, g_rx_buf, 32, 0);

    /* Enable DMA channel */
    md_dma_clear_flag_done(MD_DMA_CH_1);
    md_dma_enable_channel(MD_DMA_CH_1);
    md_usart_enable_dma_req_rx(USART0);

    while (1)
    {
        /* strongly recommend to clear idle flag in main loop */
        if (md_usart_is_active_flag_idle(USART0))
            md_usart_clear_flag_idle(USART0);
    }
}

/**
  * @}
  */
/**
  * @}
  */


/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
