/**********************************************************************************
 *
 * @file    spi_flash.c
 * @brief   spi_flash C file
 *
 * @date    30 Apri 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          30 Apri 2021    yanght          the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */
/* Includes ------------------------------------------------------------------ */
#include "spi_flash.h"

/* Private Macros ------------------------------------------------------------ */
/* Public Variables ---------------------------------------------------------- */
/* Private Constants --------------------------------------------------------- */

/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */
/* Private Variables --------------------------------------------------------- */
static md_spi_init_t s_spi;
/* Private function prototypes ----------------------------------------------- */
static void delay(int i);
static void spi_pin_init(void);
/* Private Function ---------------------------------------------------------- */
int32_t md_spi_send_byte_fast(SPI_TypeDef *SPIx, uint8_t data)
{
    uint16_t cnt = 5000U, temp;

    SPIx->DATA = data;

    while (((SPIx->STAT & SPI_STAT_TXBE_MSK) == 0) && (--cnt));

    cnt = 5000;

    while (((SPIx->STAT & SPI_STAT_RXBNE_MSK) == 0) && (--cnt));

    temp = SPIx->DATA;
    UNUSED(temp);

    return cnt == 0 ? -1 : 0;
}

uint8_t md_spi_recv_byte_fast(SPI_TypeDef *SPIx, int *status)
{
    uint16_t cnt = 5000U;

    if (READ_BIT(SPIx->CON1, SPI_CON1_MSTREN_MSK) == SPI_CON1_MSTREN_MSK)
    {
        SPIx->DATA = 0xFF;

        while (((SPIx->STAT & SPI_STAT_TXBE_MSK) == 0) && (--cnt));
    }

    cnt = 5000;

    while (((SPIx->STAT & SPI_STAT_RXBNE_MSK) == 0) && (--cnt));

    *status = cnt == 0 ? -1 : 0;

    return (uint8_t)SPIx->DATA;
}
/**
  * @brief  flash sector erase function
  * @param  addr: Specific address which sector to be erase.
  * @retval Status.
  */
md_status_t flash_sector_erase(uint32_t addr)
{
    uint8_t cmd_buf[4];
    uint8_t i = 0U;

    cmd_buf[0] = FLASH_ERASE;       /*Flash sector erase command*/
    cmd_buf[1] = (addr >> 16) & 0xff;   /*24 bit Flash address*/
    cmd_buf[2] = (addr >> 8) & 0xff;
    cmd_buf[3] = addr & 0xff;

    FLASH_CS_CLR();     /*Choose lower, the selected Flash*/

    if (md_spi_send_byte_fast(SPI0, FLASH_WRITE_ENABLE) != MD_OK)  /*First write send enabling instruction*/
    {
        FLASH_CS_SET();
        return MD_ERROR;
    }

    FLASH_CS_SET();    /*Pick up and release of Flash*/

    delay(100);
    FLASH_CS_CLR();

    for (i = 0; i < sizeof(cmd_buf); i++)     /*Send the sector erase instructions and Flash address three bytes*/
    {
        if (md_spi_send_byte_fast(SPI0, cmd_buf[i]) != MD_OK)
        {
            FLASH_CS_SET();
            return MD_ERROR;
        }
    }

    FLASH_CS_SET();

    return MD_OK;
}

/**
  * @brief  transmit an amount of data in blocking mode.
  * @param  addr: Specific address which to be write.
  * @param  buf: Pointer to data buffer
  * @param  size: Amount of data to be sent
  * @retval Status, see @ref md_status_t.
  */
md_status_t flash_write(uint32_t addr, char *buf, uint8_t size)
{
    uint8_t cmd_buf[4];
    uint8_t i = 0U;

    if (buf == NULL)
        return MD_ERROR;

    cmd_buf[0] = FLASH_PROGRAM;
    cmd_buf[1] = (addr >> 16) & 0xff;
    cmd_buf[2] = (addr >> 8) & 0xff;
    cmd_buf[3] = addr & 0xff;

    FLASH_CS_CLR();     /*Choose lower, the selected Flash*/

    if (md_spi_send_byte_fast(SPI0, FLASH_WRITE_ENABLE) != MD_OK)
    {
        FLASH_CS_SET();
        return MD_ERROR;
    }

    FLASH_CS_SET();    /*Pick up and release of Flash*/

    delay(100);
    FLASH_CS_CLR();

    for (i = 0; i < sizeof(cmd_buf); i++)     /*Send the programming instructions and Flash address three bytes*/
    {
        if (md_spi_send_byte_fast(SPI0, cmd_buf[i]) != MD_OK)
        {
            FLASH_CS_SET();
            return MD_ERROR;
        }
    }

    for (i = 0; i < size; i++)  /*To write the data sent to the Flash*/
    {
        if (md_spi_send_byte_fast(SPI0, buf[i]) != MD_OK)
        {
            FLASH_CS_SET();
            return MD_ERROR;
        }
    }

    FLASH_CS_SET();

    return MD_OK;
}

/**
  * @brief  Receive an amount of data in blocking mode.
  * @param  addr: address of flash where want to read.
  * @param  buf: Pointer to data buffer
  * @param  size: Amount of data to be received
  * @retval Status, see @ref md_status_t.
  */
md_status_t flash_read(uint32_t addr, char *buf, uint16_t size)
{
    uint8_t cmd_buf[4];
    uint8_t i = 0U;
    int r_flag = 0;

    if (buf == NULL)
        return MD_ERROR;

    cmd_buf[0] = FLASH_READ;
    cmd_buf[1] = (addr >> 16) & 0xff;
    cmd_buf[2] = (addr >> 8) & 0xff;
    cmd_buf[3] = addr & 0xff;

    FLASH_CS_CLR();     /*Choose lower, the selected Flash*/

    for (i = 0; i < sizeof(cmd_buf); i++)   /*Send the editor & reader instructions and Flash address three bytes*/
    {
        if (md_spi_send_byte_fast(SPI0, cmd_buf[i]) != MD_OK)
        {
            FLASH_CS_SET();     /*Pick up and release of Flash*/
            return MD_ERROR;
        }
    }

    for (i = 0; i < size; i++)  /*Send the programming instructions and Flash address three bytes*/
    {
        buf[i] = md_spi_recv_byte_fast(SPI0, &r_flag);

        if (r_flag != MD_OK)
        {
            FLASH_CS_SET();
            return MD_ERROR;
        }
    }

    FLASH_CS_SET();

    return MD_OK;
}

/**
  * @brief  wait until flash unbusy.
  * @retval Status, see @ref md_status_t.
  */
md_status_t flash_wait_unbusy(void)
{
    uint8_t status;
    int r_flag = 0;

    FLASH_CS_CLR(); /*Choose lower, the selected Flash*/

    if (md_spi_send_byte_fast(SPI0, (uint8_t)FLASH_STATUS) != MD_OK)   /*Send to read status command*/
    {
        FLASH_CS_SET();     /*Pick up and release of Flash*/
        return MD_ERROR;
    }

    do
    {
        status = md_spi_recv_byte_fast(SPI0, &r_flag);

        if (r_flag != MD_OK)
        {
            FLASH_CS_SET();
            return MD_ERROR;
        }
    }
    while (status & 0x01);

    FLASH_CS_SET();

    return MD_OK;
}

/**
  * @brief  Read flash id in blocking mode.
  * @retval flash id.
  */
uint32_t flash_read_id(void)
{
    uint8_t i;
    int r_flag = 0;
    uint8_t flash_id[4] = {0U};

    flash_id[0] = FLASH_ID;

    FLASH_CS_CLR(); /*Choose lower, the selected Flash*/

    for (i = 0; i < sizeof(flash_id); i++)
    {
        if (md_spi_send_byte_fast(SPI0, flash_id[i]) != MD_OK)
        {
            FLASH_CS_SET();     /*Pick up and release of Flash*/
            return MD_ERROR;
        }
    }

    for (i = 0; i < 3; i++)
    {
        flash_id[i] = md_spi_recv_byte_fast(SPI0, &r_flag);

        if (r_flag != MD_OK)
        {
            FLASH_CS_SET();
            return MD_ERROR;
        }
    }

    FLASH_CS_SET();

    return ((flash_id[0] << 16) | (flash_id[1] << 8) | (flash_id[2]));  /*Manufacturer ID flash_id [0] and device ID flash_id [1]*/
}

/**
  * @brief spi function
  * @retval None.
  */
void mcu_spi_init(void)
{
    spi_pin_init();

    s_spi.mode      = MD_SPI_MODE_MASTER;   /*SPI host mode*/
    s_spi.baud      = MD_SPI_BAUD_8;        /*48 MHZ clock / 4 = 12 MHZ*/
    s_spi.data_size = MD_SPI_DATA_SIZE_8;   /*8 bit pattern*/
    s_spi.polarity  = MD_SPI_CPOL_HIGH;     /*Free high level*/
    s_spi.phase     = MD_SPI_CPHA_SECOND;   /*The second edge receiving data*/
    s_spi.first_bit = MD_SPI_FIRSTBIT_MSB;  /*Send the MSB first*/
    s_spi.dir       = MD_SPI_DIRECTION_2LINES;
    s_spi.ss_en     = DISABLE;
    s_spi.crc_calc  = DISABLE;
    s_spi.crc_poly  = 0;

    md_spi_init(SPI0, &s_spi); /*According to the parameter initialization SPI peripherals*/
}

/**
  * @brief  Initializate spi flash pin
  * @retval None.
  */
static void spi_pin_init(void)
{
    md_gpio_init_t l_gpio;

    l_gpio.type  = MD_GPIO_TYPE_CMOS;
    l_gpio.odos  = MD_GPIO_PUSH_PULL;
    l_gpio.pupd  = MD_GPIO_PUSH_UP;
    l_gpio.odrv  = MD_GPIO_OUT_DRIVE_NORMAL;
    l_gpio.flt   = MD_GPIO_FILTER_DISABLE;

    l_gpio.func  = MD_GPIO_FUNC_1;
    l_gpio.mode  = MD_GPIO_MODE_OUTPUT;
    md_gpio_init(GPIOA, MD_GPIO_PIN_0, &l_gpio);    /*Initialize PA15 for selected pin*/
    FLASH_CS_SET();   /*Choose the output high, release of Flash*/

    l_gpio.func  = MD_GPIO_FUNC_2;
    l_gpio.mode  = MD_GPIO_MODE_OUTPUT;
    md_gpio_init(GPIOA, MD_GPIO_PIN_10, &l_gpio);    /*Initialize the PA10 for clock output pin*/

    l_gpio.func  = MD_GPIO_FUNC_2;
    l_gpio.mode  = MD_GPIO_MODE_OUTPUT;
    md_gpio_init(GPIOA, MD_GPIO_PIN_11, &l_gpio);    /*Initialize PA11 for MOSI pin*/

    l_gpio.func  = MD_GPIO_FUNC_2;
    l_gpio.mode  = MD_GPIO_MODE_INPUT;
    md_gpio_init(GPIOA, MD_GPIO_PIN_12, &l_gpio);    /*Initialize PA12 for MISO pins*/
}

/**
  * @brief  delay some time.
  * @retval None.
  */
static void delay(int i)
{
    while (i--) ;
}

/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
