/**********************************************************************************
 *
 * @file    main.c
 * @brief   Main file for demo
 *
 * @date    30 Apri 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          30 Apri 2021    yanght          the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */

#include <string.h>
#include "main.h"
void SystemInit()
{}

/* Private Macros ------------------------------------------------------------ */

#define HIGH_FIRST     /*The clock polarity and clock phase*/

/* Private Variables --------------------------------------------------------- */

/* Public Variables ---------------------------------------------------------- */

md_spi_init_t g_h_spi;
md_dma_descriptor_t g_dma0_ctrl_base[2] __attribute__((aligned(512)));
md_dma_config_t g_tx_config, g_rx_config;

uint8_t g_send_buf[20] = {0x01U, 0x12U, 0x23U, 0x34U, 0x45U, 0x56U, 0x67U, 0x78U, 0x89U, 0x9AU, 0x01U, 0x12U, 0x23U, 0x34U, 0x45U, 0x56U, 0x67U, 0x78U, 0x89U, 0x9AU};
uint8_t g_send_len = sizeof(g_send_buf);
uint8_t g_recv_buf[20] = {0U};
uint8_t g_recv_len = sizeof(g_recv_buf);
uint8_t g_complete;

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Function ---------------------------------------------------------- */

/*Host mode communication pins initialization function*/
void spi_pin_init(void)
{
    md_gpio_init_t x;

    /*Host SCK pin initialization, configuration for the output mode*/
    x.mode = MD_GPIO_MODE_OUTPUT;
    x.odos = MD_GPIO_PUSH_PULL;
    x.pupd = MD_GPIO_PUSH_UP;
    x.odrv = MD_GPIO_OUT_DRIVE_NORMAL;
    x.flt  = MD_GPIO_FILTER_DISABLE;
    x.type = MD_GPIO_TYPE_CMOS;
    x.func = MD_GPIO_FUNC_2;
    md_gpio_init(GPIO_PORT, SPI_SCK_PIN, &x);

    /*Host MISO pins initialization, configuration for the input mode*/
    x.mode = MD_GPIO_MODE_INPUT;
    x.odos = MD_GPIO_PUSH_PULL;
    x.pupd = MD_GPIO_PUSH_UP;
    x.odrv = MD_GPIO_OUT_DRIVE_NORMAL;
    x.flt  = MD_GPIO_FILTER_DISABLE;
    x.type = MD_GPIO_TYPE_CMOS;
    x.func = MD_GPIO_FUNC_2;
    md_gpio_init(GPIO_PORT, SPI_MISO_PIN, &x);

    /*Host MOSI pin initialization, configuration for the output mode*/
    x.mode = MD_GPIO_MODE_OUTPUT;
    x.odos = MD_GPIO_PUSH_PULL;
    x.pupd = MD_GPIO_PUSH_UP;
    x.odrv = MD_GPIO_OUT_DRIVE_NORMAL;
    x.flt  = MD_GPIO_FILTER_DISABLE;
    x.type = MD_GPIO_TYPE_CMOS;
    x.func = MD_GPIO_FUNC_2;
    md_gpio_init(GPIO_PORT, SPI_MOSI_PIN, &x);

    /*Host NSS pin initialization, configuration for the output mode*/
    x.mode = MD_GPIO_MODE_OUTPUT;
    x.odos = MD_GPIO_PUSH_PULL;
    x.pupd = MD_GPIO_PUSH_UP;
    x.odrv = MD_GPIO_OUT_DRIVE_NORMAL;
    x.flt  = MD_GPIO_FILTER_DISABLE;
    x.type = MD_GPIO_TYPE_CMOS;
    x.func = MD_GPIO_FUNC_1;        /* GPIO */
    md_gpio_init(GPIO_PORT, SPI_NSS_PIN, &x);
    md_gpio_set_pin_high(GPIO_PORT, SPI_NSS_PIN);
}

/*SPI configuration function*/
void spi_struct_user(md_spi_init_t *init)
{
    init->mode      = MD_SPI_MODE_MASTER;       /*Host mode*/
    init->dir       = MD_SPI_DIRECTION_2LINES;      /*Full duplex*/
    init->data_size = MD_SPI_DATA_SIZE_8;
    init->baud      = MD_SPI_BAUD_32;

#ifdef HIGH_FIRST       /*Free clock high level, the first clock edge sampling data*/
    init->phase     = MD_SPI_CPHA_FIRST;
    init->polarity  = MD_SPI_CPOL_HIGH;

#elif defined HIGH_SECOND       /*Free clock high level, the second clock edge sampling data*/
    init->phase     = MD_SPI_CPHA_SECOND;
    init->polarity  = MD_SPI_CPOL_HIGH;

#elif defined LOW_FIRST     /*Free clock low level, the first clock edge sampling data*/
    init->phase     = MD_SPI_CPHA_FIRST;
    init->polarity  = MD_SPI_CPOL_LOW;

#elif defined LOW_SECOND        /*Free clock low level, the second clock edge sampling data*/
    init->phase     = MD_SPI_CPHA_SECOND;
    init->polarity  = MD_SPI_CPOL_LOW;
#endif  /*HIGH_FIRST*/

    init->first_bit = MD_SPI_FIRSTBIT_MSB;      /*High in the former*/
    init->ss_en     = DISABLE;
    init->crc_calc  = DISABLE;
    init->crc_poly  = 0;
}

/*The DMA TX initialization function*/
void config_dma_struct_tx(void)
{
    memset(&g_tx_config, 0x0, sizeof(g_tx_config));
    md_dma_set_ctrlbase((uint32_t)&g_dma0_ctrl_base);

    /*To enable the DMA*/
    md_dma_enable();

    /*User configuration DMA*/
    md_dma_config_struct(&g_tx_config);
    g_tx_config.data_width = MD_DMA_DATA_SIZE_BYTE;
    g_tx_config.src_inc    = MD_DMA_DATA_INC_BYTE;       /*The source data address byte increments*/
    g_tx_config.dst_inc    = MD_DMA_DATA_INC_NONE;       /*The target data address no increment*/
    g_tx_config.R_power    = MD_DMA_R_POWER_1;
    g_tx_config.primary    = ENABLE;
    g_tx_config.burst      = DISABLE;
    g_tx_config.high_prio  = DISABLE;
    g_tx_config.interrupt  = ENABLE;
    g_tx_config.channel    = 0;
    g_tx_config.src        = (void *)(g_send_buf);         /*The source data*/
    g_tx_config.dst        = (void *)(&SPI0->DATA);      /*The target data*/
    g_tx_config.size       = g_send_len;                   /*The length of the data*/
    g_tx_config.msel       = MD_DMA_MSEL_SPI0;
    g_tx_config.msigsel    = MD_DMA_MSIGSEL_SPI_TXEMPTY; /*SPI0 send*/

}

/*DMA_RX initialization function*/
void config_dma_struct_rx(void)
{
    memset(&g_rx_config, 0x0, sizeof(g_rx_config));
    md_dma_set_ctrlbase((uint32_t)&g_dma0_ctrl_base);

    /*To enable the DMA*/
    md_dma_enable();

    /*User configuration DMA*/
    md_dma_config_struct(&g_rx_config);
    g_rx_config.data_width = MD_DMA_DATA_SIZE_BYTE;
    g_rx_config.src_inc    = MD_DMA_DATA_INC_NONE;       /*Source data is no address increment*/
    g_rx_config.dst_inc    = MD_DMA_DATA_INC_BYTE;       /*The target data address byte increments*/
    g_rx_config.R_power    = MD_DMA_R_POWER_1;
    g_rx_config.primary    = ENABLE;
    g_rx_config.burst      = DISABLE;
    g_rx_config.high_prio  = DISABLE;
    g_rx_config.interrupt  = ENABLE;
    g_rx_config.channel    = 1;
    g_rx_config.src        = (void *)(&SPI0->DATA);      /*The source data*/
    g_rx_config.dst        = (void *)(g_recv_buf);         /*The target data*/
    g_rx_config.size       = g_recv_len;                   /*The length of the data*/
    g_rx_config.msel       = MD_DMA_MSEL_SPI0;
    g_rx_config.msigsel    = MD_DMA_MSIGSEL_SPI_RNR;     /*SPI0 receive*/

}

/*The main function*/
int main()
{
    /* Configure system clock */
    md_cmu_clock_config(MD_CMU_CLOCK_HRC, 32000000);
    /* Initialize SysTick Interrupt */
    md_init_1ms_tick();

    /* Enable ALL peripheral */
    SYSCFG_UNLOCK();
    md_cmu_enable_perh_all();
    SYSCFG_LOCK();

    /*Pin initialization*/
    spi_pin_init();

    /*Empty the SPI initialization structure*/
    memset(&g_h_spi, 0, sizeof(g_h_spi));

    /*The SPI initialization structure is set to the default values*/
    md_spi_struct_init(&g_h_spi);

    /*User configuration SPI*/
    spi_struct_user(&g_h_spi);

    /*SPI initialization*/
    md_spi_init(SPI0, &g_h_spi);
    md_spi_disable_tx_dma(SPI0);
    md_spi_disable_rx_dma(SPI0);
    /*Can make the SPI*/
    md_spi_enable(SPI0);

    NVIC_SetPriority(DMA_IRQn, 0);
    NVIC_EnableIRQ(DMA_IRQn);
    memset(g_dma0_ctrl_base, 0x0, sizeof(g_dma0_ctrl_base));
    config_dma_struct_tx();
    config_dma_struct_rx();

    while (1)
    {
        g_complete = 0;
        /*Lower NSS pin to select the external from the machine*/
        md_gpio_set_pin_low(GPIO_PORT, SPI_NSS_PIN);
        md_dma_config_base(DMA0, MD_DMA_CYCLE_CTRL_BASIC, &g_tx_config);
        md_dma_config_base(DMA0, MD_DMA_CYCLE_CTRL_BASIC, &g_rx_config);
        md_dma_enable_channel(MD_DMA_CH_1);
        md_dma_enable_channel(MD_DMA_CH_0);
        md_spi_enable_tx_dma(SPI0);
        md_spi_enable_rx_dma(SPI0);

        while (g_complete != 1);

        md_gpio_set_pin_high(GPIO_PORT, SPI_NSS_PIN);

        md_delay_1ms(3000);
    }
}

/**
  * @} Examples_SPI
  */

/**
  * @} Projects_Examples_MD
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
