/**********************************************************************************
 *
 * @file    irq.c
 * @brief   Interrupt handler
 *
 * @date    29 Apri 2023
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          30 May 2021     zhuxf           the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */
#include "main.h"

/* Private Macros ------------------------------------------------------------ */

/* Private Variables --------------------------------------------------------- */
volatile static uint8_t s_second = 59;
volatile static uint32_t s_wwdt_load_value_10ms = 320;

/* Public Variables ---------------------------------------------------------- */

/* Private Constants --------------------------------------------------------- */

/** @addtogroup Projects_Examples_MD
  * @{
  */

/** @addtogroup Examples
  * @{
  */

/* Private function prototypes ----------------------------------------------- */

/* Private Function ---------------------------------------------------------- */

/**
  * @brief  Hardfault IRQ handler, already defined in startup file
  * @retval None
  */

/**
  * @brief  SysTick IRQ handler
  * @retval None
  */
void SysTick_Handler(void)
{
    md_inc_tick();
    return;
}

/**
  * @brief  CMU IRQ handler
  * @retval None
  */
void CMU_Handler(void)
{
    if ((md_cmu_is_enabled_hosc_stp_interrupt()) && (md_cmu_is_active_flag_hosc_stp()))
    {
        SYSCFG_UNLOCK();
        md_cmu_clear_flag_hosc_stp();
        SYSCFG_LOCK();
        /* system rescue code in hosc fault  */
    }

    return;
}

/**
  * @brief  NMI handler
  * @retval None
  */
void NMI_Handler(void)
{
    if ((md_cmu_is_enabled_hosc_nmi_interrupt()) && (md_cmu_is_active_flag_hosc_stp()))
    {
        SYSCFG_UNLOCK();
        md_cmu_clear_flag_hosc_stp();
        SYSCFG_LOCK();
        /* system rescue code in hosc fault  */
    }

    return;
}

void WWDT_Handler(void)
{

    uint8_t i;

    /* Feed dog */
    WWDT_UNLOCK();
    md_wwdt_clear_interrupt();
    WWDT_LOCK();

    md_gpio_toggle_pin_output(LED_PORT, LED_PIN);    /* IO翻转 */

    s_second++;        /* 秒计数 */

    if (s_second >= 60)
    {
        s_second = 0;      /* 第0秒执行校准LRC */
        
        SYSCFG_UNLOCK();
        md_rmu_enable_wwdt_reset();
        SYSCFG_LOCK();

        md_wwdt_init(s_wwdt_load_value_10ms * 100 * 4, MD_WWDT_WIN_25, ENABLE);
        WWDT_UNLOCK();
        md_wwdt_enable();
        WWDT_LOCK();

        for (i = 0 ; i < 100; i++)      /* 等待10ms */
        {
            md_delay_1us(100);
        }

        /* 获取10ms内 WWDT的计数值 */
        s_wwdt_load_value_10ms = (s_wwdt_load_value_10ms * 100) - (md_wwdt_get_count_value() & 0x3FFFFFFF);

        SYSCFG_UNLOCK();
        md_rmu_enable_wwdt_reset();
        SYSCFG_LOCK();

        /* 配置WWDT继续计数990ms后产生中断 */
        md_wwdt_init(s_wwdt_load_value_10ms * 99 * 4, MD_WWDT_WIN_25, ENABLE);
        WWDT_UNLOCK();
        md_wwdt_enable();
        WWDT_LOCK();

    }
    else if (s_second == 1)   /* 第1秒使用第0秒的校准值初始化WWDT */
    {
        SYSCFG_UNLOCK();
        md_rmu_enable_wwdt_reset();
        SYSCFG_LOCK();

        md_wwdt_init(s_wwdt_load_value_10ms * 100 * 4, MD_WWDT_WIN_25, ENABLE);
        WWDT_UNLOCK();
        md_wwdt_enable();
        WWDT_LOCK();
    }

    md_delay_1us(200);     /* 等待WWDT设置完成 */
}

/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
