/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    30 Oct 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          30 Oct 2021     biyq            the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */
/* Includes ------------------------------------------------------------------ */
#include "main.h"
#include "stdio.h"

/* Public Variables ---------------------------------------------------------- */
/* Private Macros ------------------------------------------------------------ */
/* Private Variables --------------------------------------------------------- */
static uint32_t crc_buf[8] = {1U, 2U, 3U, 4U, 5U, 6U, 7U, 8U};
static uint32_t crc_result = 0x0U;

/* Private Constants --------------------------------------------------------- */
/* Private function prototypes ----------------------------------------------- */
void gpio_init(void);
void usart_init(void);

/* Private Function ---------------------------------------------------------- */
/**
  * @brief  Initializate system.
  * @retval None
  */
void SystemInit(void)
{

}

/**
  * @brief:  Main program.
  * @param:  None
  * @retval: None
  */
int main(void)
{
    /* Configure system clock */
    md_cmu_clock_config(MD_CMU_CLOCK_HRC, 32000000);
    /* Initialize SysTick Interrupt */
    md_init_1ms_tick();

    /* Enable ALL peripheral */
    SYSCFG_UNLOCK();
    md_cmu_enable_perh_all();
    SYSCFG_LOCK();

    /* Init GPIO */
    gpio_init();

    /* Init UART */
    usart_init();

    /* Init CRC32 */
    init_crc32_module();

    /* Calculate CRC32 value for array crc_buf */
    crc_result = get_crc32_calculation_result(crc_buf, (sizeof(crc_buf) / 4));

    /* Delay 7S and output value */
    md_delay_1ms(7000);
    printf("CRC32 calculation result for 0x01 ~ 0x08 is: 0x%.8X\n", crc_result);

    while (1)
    {
        md_gpio_toggle_pin_output(LED_PORT, LED_PIN);
        md_delay_1ms(1000);
    }
}

/**
  * @brief  Initialize gpio for led.
  * @retval None
  */
void gpio_init(void)
{
    md_gpio_init_t x;

    /* Initialize led1 pin */
    x.mode  = MD_GPIO_MODE_OUTPUT;
    x.odos  = MD_GPIO_PUSH_PULL;
    x.pupd  = MD_GPIO_PUSH_UP;
    x.odrv = MD_GPIO_OUT_DRIVE_NORMAL;
    x.flt   = MD_GPIO_FILTER_DISABLE;
    x.type  = MD_GPIO_TYPE_CMOS;
    x.func  = MD_GPIO_FUNC_1;
    md_gpio_init(LED_PORT, LED_PIN, &x);

    return;
}

/**
  * @brief  Initialize gpio for uart.
  * @retval None
  */
void usart_gpio_init(void)
{

    md_gpio_init_t g_gpio_init_s;
    md_gpio_init_struct(&g_gpio_init_s);

    /* Initialize tx pin: PA13 */
    g_gpio_init_s.type  = MD_GPIO_TYPE_CMOS;
    g_gpio_init_s.func = MD_GPIO_FUNC_3;
    md_gpio_init(GPIOA, MD_GPIO_PIN_13, &g_gpio_init_s);

    /* Initialize rx pin: PA14 */
    g_gpio_init_s.mode = MD_GPIO_MODE_INPUT;
    md_gpio_init(GPIOA, MD_GPIO_PIN_14, &g_gpio_init_s);
}

/**
* @brief  Initialize uart as default configuration: baud: 115200, data bit: 8, stop bit: 1, no parity and flow control.
  * @retval None
  */
void usart_init(void)
{
    md_usart_init_t g_uart_init_s;
    md_usart_init_struct(&g_uart_init_s);

    usart_gpio_init();
    md_usart_init(USART0, &g_uart_init_s);
}

/**
  * @brief  Rewrite fput function.
  * @retval None
  */
int fputc(int c, FILE *f)
{
    while (!md_usart_is_active_flag_txemp(USART0));

    md_usart_send_data8(USART0, c);
    return c;
}


/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
