/**********************************************************************************
 *
 * @file    crc32.c
 * @brief   crc32 C file
 *
 * @date    30 Oct 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          30 Oct 2021     biyq            the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */

#include <string.h>
#include "crc32.h"

/* Private Macros ------------------------------------------------------------ */
/* Private Constants --------------------------------------------------------- */
/* Private function prototypes ----------------------------------------------- */
/* Private Variables --------------------------------------------------------- */
/* Public Variables --------------------------------------------------------- */

/** @addtogroup Projects_Examples_MD
  * @{
  */

/** @addtogroup Examples
  * @{
  */

/* Private Function ---------------------------------------------------------- */
/**
  * @brief  Reset the CRC peripheral.
  * @retval None
  */
#if   defined ( __ICCARM__ )
    #pragma optimize=none
    __root void  set_iar_crc_reset_reg(void)@".get_iar_cusum32"
#elif defined ( __CC_ARM )
    void  set_keil_crc_reset_reg(void)__attribute((section("crc32_section")));
    void  set_keil_crc_reset_reg(void)
#endif
{
    CRC->DATA = 0x0;
    CRC->CR   = 0x2;
    CRC->SEED = 0xFFFFFFFF;

    return;
}

/**
  * @brief  Init CRC module.
  * @retval None.
  */
#if   defined ( __ICCARM__ )
    #pragma optimize=none
    __root void  set_iar_crc_init(md_crc_init_t *init)@".get_iar_cusum32"
#elif defined ( __CC_ARM )
    void  set_keil_crc_init(md_crc_init_t *init)__attribute((section("crc32_section")));
    void  set_keil_crc_init(md_crc_init_t *init)
#endif
{
    uint32_t tmp = 0;

#if   defined ( __ICCARM__ )
    set_iar_crc_reset_reg();
#elif defined ( __CC_ARM )
    set_keil_crc_reset_reg();
#endif


    tmp |= ((init->chs_rev << CRC_CR_CHSREV_POS) | (init->data_inv << CRC_CR_DATINV_POS) |
            (init->chs_inv << CRC_CR_CHSINV_POS) | (init->mode << CRC_CR_MODE_POSS) |
            (init->len << CRC_CR_DATLEN_POSS) | (init->data_rev << CRC_CR_DATREV_POS) |
            (init->order << CRC_CR_BYTORD_POS));
    WRITE_REG(CRC->CR, tmp);
    WRITE_REG(CRC->SEED, init->seed);
    MD_CRC_RESET();
    MD_CRC_ENABLE();

    return;
}

/**
  * @brief  Init structure which describe CRC configuration.
  * @retval None.
  */
#if   defined ( __ICCARM__ )
    #pragma optimize=none
    __root void  set_iar_crc_init_struct(md_crc_init_t *init)@".get_iar_cusum32"
#elif defined ( __CC_ARM )
    void  set_keil_crc_init_struct(md_crc_init_t *init)__attribute((section("crc32_section")));
    void  set_keil_crc_init_struct(md_crc_init_t *init)
#endif
{
    init->mode     = MD_CRC_MODE_CCITT;
    init->len      = MD_CRC_DATASIZE_8;
    init->order    = MD_CRC_BYTORD_LOW;
    init->data_rev = DISABLE;
    init->data_inv = DISABLE;
    init->chs_rev  = DISABLE;
    init->chs_inv  = DISABLE;
    init->seed     = 0x0;

    return;
}

/**
  * @brief  Init CRC module as standard CRC32.
  * @retval None.
  */
#if   defined ( __ICCARM__ )
    #pragma optimize=none
    __root void  init_crc32_module(void)@".get_iar_cusum32"
#elif defined ( __CC_ARM )
    void  init_crc32_module(void);
    void  init_crc32_module(void)
#endif
{
    md_crc_init_t g_init;
#if   defined ( __ICCARM__ )
    set_iar_crc_init_struct(&g_init);
#elif defined ( __CC_ARM )
    set_keil_crc_init_struct(&g_init);
#endif
    g_init.mode = MD_CRC_MODE_32;
    g_init.seed = 0xFFFFFFFF;
    g_init.chs_inv = ENABLE;
    g_init.chs_rev = ENABLE;
    g_init.data_inv = DISABLE;
    g_init.data_rev = ENABLE;

#if   defined ( __ICCARM__ )
    set_iar_crc_init(&g_init);
#elif defined ( __CC_ARM )
    set_keil_crc_init(&g_init);
#endif

    return;
}

/**
  * @brief  Calculate result accoding to crc algorithm.
  * @retval CRC32 calculation result.
  */
#if   defined ( __ICCARM__ )
    #pragma optimize=none
    __root uint32_t  get_crc32_calculation_result(uint32_t *buf, uint32_t size)@".get_iar_cusum32"
#elif defined ( __CC_ARM )
    uint32_t  get_crc32_calculation_result(uint32_t *buf, uint32_t size);
    uint32_t  get_crc32_calculation_result(uint32_t *buf, uint32_t size)
#endif
{
    uint32_t i;

    for (i = 0; i < size; i++)
       CRC->DATA = buf[i];

    return (CRC->CHECKSUM);
}

/**
  * @}
  */
/**
  * @}
  */
