/**********************************************************************************
 *
 * @file    i2c.c
 * @brief   i2c C file
 *
 * @date    25 Aug 2022
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          25 Aug 2022     biyq            the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */
#include "main.h"
#include "ens210.h"
#include "drv_soft_i2c.h"
#include "stdio.h"

/* Private Macros ------------------------------------------------------------ */

/* CRC POLY */
#define CRC7WIDTH	(7)
#define CRC7POLY	(0x89)
#define CRC7IVEC	(0x7F)
/* CRC DATA */
#define DATA7WIDTH	(17)
#define DATA7MASK	((1UL << DATA7WIDTH) - 1)
#define DATA7MSB	(1UL << (DATA7WIDTH - 1))

/* Private Variables --------------------------------------------------------- */

/* Public Variables ---------------------------------------------------------- */

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Function ---------------------------------------------------------- */

/* Public Function ---------------------------------------------------------- */
/**
  * @brief  Write data to sensor via software I2C.
  * @param  sensor_addr: sensor address.
  *			reg_addr: target register address.
  *			tx_buf: data buffer pointer which is to sent.
  *			len: length of expected data.
  * @retval  0 for success, and 1 for fail.
  */
uint8_t sw_i2c_sensor_write(uint8_t sensor_addr, uint8_t reg_addr, uint8_t *tx_buf, uint32_t len)
{
    struct drv_soft_i2c_bus_device *i2c_bus = &_i2c_device0; 
    struct drv_soft_i2c_msg temp_msg[2];
	 
    temp_msg[0].addr = sensor_addr;
    temp_msg[0].len  = 1;
    temp_msg[0].buf = &reg_addr;
    temp_msg[0].flags = DRV_SOFT_I2C_WR|DRV_SOFT_I2C_NO_STOP;
	
    temp_msg[1].addr = sensor_addr;
    temp_msg[1].len  = len;
    temp_msg[1].buf = tx_buf;
    temp_msg[1].flags = DRV_SOFT_I2C_WR;
	
    if (i2c_bit_xfer(i2c_bus, temp_msg, 2) == 2)
		return 0;
	else
		return 1;
}

/**
  * @brief  Read data from sensor register via software I2C.
  * @param  sensor_addr: sensor address.
  *			reg_addr: target register address.
  *			rx_buf: buffer which store the expected data.
  *			len: length of expected data.
  * @retval 0 for success, and 1 for fail.
  */
uint8_t sw_i2c_sensor_read(uint8_t sensor_addr, uint8_t reg_addr, uint8_t *rx_buf, uint32_t len)
{
    struct drv_soft_i2c_bus_device *i2c_bus = &_i2c_device0; 
    struct drv_soft_i2c_msg temp_msg[2];
	
    temp_msg[0].addr = sensor_addr;
    temp_msg[0].len  = 1;
    temp_msg[0].buf = &reg_addr;
    temp_msg[0].flags = DRV_SOFT_I2C_WR|DRV_SOFT_I2C_NO_STOP;
	
    memset(rx_buf, 0xFF, len);
    temp_msg[1].addr = sensor_addr;
    temp_msg[1].len  = len;
    temp_msg[1].buf = rx_buf;
    temp_msg[1].flags = DRV_SOFT_I2C_RD;
    
    if (i2c_bit_xfer(i2c_bus, temp_msg, 2) == 2)
		return 0;
	else
		return 1;
}

/**
  * @brief  Initialization for software I2C.
  * @param  baund: the baund of I2C.
  * @retval None
  */
void sw_i2c_sensor_init(void)
{
	uint8_t data[10];
	drv_soft_i2c_init();
	
    sw_i2c_sensor_read(SENSOR_ADDR, 0x10, data, 1);
    printf("[ENS210] Read SYSCTRL = 0x%X\r\n", data[0]);
    
    sw_i2c_sensor_read(SENSOR_ADDR, 0x11, data, 1);
    printf("[ENS210] Read STAT = 0x%X\r\n", data[0]);
    
    sw_i2c_sensor_read(SENSOR_ADDR, 0x00, data, 2);
    printf("[ENS210] Read PART_ID = 0x%X\r\n", (data[1]<<8) + data[0]);
    
	return;
}

/**
  * @brief  Read data from sensor register via software I2C.
  * @param  rx_buf: buffer which store the expected data.
  * @param  len: length of data.
  * @retval 0: success, other:fail.
  */
uint8_t sw_i2c_sensor_crc_verify(uint8_t *rx_buf, uint8_t len)
{
	uint32_t bit, pol, i, value;
	uint32_t result = 0;
	
	for (i = 0; i < len; i += 3)
	{
		if ((rx_buf[i + 2] & 0x01) != 0x01)
		{
			result |= (i + 3) / 3;
			continue;
		}
		
		value =  (uint32_t)(0x01 << 16);
		value |= (uint32_t)((uint32_t)rx_buf[i + 1] << 8);
		value |= (uint32_t)rx_buf[i];
		
		pol = CRC7POLY;
		pol = pol << (DATA7WIDTH - CRC7WIDTH - 1);
		pol = pol << CRC7WIDTH;
		bit = DATA7MSB;
		bit = bit << CRC7WIDTH; 
		value <<= CRC7WIDTH; 
		value |= CRC7IVEC;
		while( bit & (DATA7MASK << CRC7WIDTH) ) 
		{
			if (bit & value) 
				value ^= pol; 
			bit >>= 1;
			pol >>= 1;
		}
		
		if (value != (uint8_t)(rx_buf[i + 2] >> 1))
		{
			result |= (i + 3) / 3;
			continue;
		}
	}
	
	return result;
}

/**
  * @} Examples_I2C
  */

/**
  * @} Projects_Examples_MD
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
