/**********************************************************************************
 *
 * @file    drv_soft_i2c.c
 * @brief   drv_soft_i2c C file
 *
 * @date    29 Aug 2022
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          29 Aug 2022     biyq            the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */
#include "drv_soft_i2c.h"

/* Private Macros ------------------------------------------------------------ */

#define SET_SDA(ops, val)   ops->set_sda(ops->data, val)
#define SET_SCL(ops, val)   ops->set_scl(ops->data, val)
#define GET_SDA(ops)        ops->get_sda(ops->data)
#define GET_SCL(ops)        ops->get_scl(ops->data)

#define SDA_L(ops)          SET_SDA(ops, 0)
#define SDA_H(ops)          SET_SDA(ops, 1)
#define SCL_L(ops)          SET_SCL(ops, 0)


/* Private function prototypes ----------------------------------------------- */

/* Private Variables --------------------------------------------------------- */
/**
  * @brief  Define the global variable for software i2c operation.
  */
struct drv_soft_i2c_bus_device_ops soft_i2c_ops = 
{
    i2c_bit_xfer,
    NULL,
    NULL
};

/**
  * @brief  Define software i2c pin information.
  */
struct drv_soft_i2c_pins soft_i2c0_pins =
{
    {
		SOFT_I2C0_SCL_PORT,
		SOFT_I2C0_SCL_PIN
	},
    {
		SOFT_I2C0_SDA_PORT,
		SOFT_I2C0_SDA_PIN
	}
};

/**
  * @brief  Define software i2c device.
  */
struct drv_soft_i2c_bus_device _i2c_device0;

/**
  * @brief  Define software i2c pin information.
  */
struct drv_soft_i2c_bit_ops es32_i2c_bit_ops;

/* Public Variables ---------------------------------------------------------- */

/* Private Constants --------------------------------------------------------- */

/* Private Function ---------------------------------------------------------- */
/**
  * @brief  Change pin state.
  * @param  ptr: pointer of es32_i2c_pin_bit.
  * @param  state: 0 for change to low level, and 1 for high level.
  * @retval NONE.
  */
static void _i2c_pin_bit_set(struct drv_soft_i2c_pin *ptr, uint8_t state)
{
    if (state)
        md_gpio_set_pin_high(ptr->gpio, ptr->pin_num);
    else
        md_gpio_set_pin_low(ptr->gpio, ptr->pin_num);
	
	return;
}

/**
  * @brief  Get pin state.
  * @param  ptr: pointer of es32_i2c_pin_bit.
  * @retval 0 for low level, and high for high level.
  */
static uint8_t _i2c_pin_bit_get(struct drv_soft_i2c_pin *ptr)
{
	uint8_t res = md_gpio_get_input_data(ptr->gpio, ptr->pin_num);
	
    return res;
}

/**
  * @brief  ticks level delay function.
  * @param  None.
  * @retval None.
  */
void i2c_bit_delay(uint32_t udelay_time)
{
	md_delay_1us(udelay_time);
	
	return;
}

/****************************************   Common frame function start **********************************************/
/**
  * @brief  Software I2C delay time.
  * @param  ops: pointer of drv_soft_i2c_bit_ops.
  * @retval NONE.
  */
void i2c_delay2(struct drv_soft_i2c_bit_ops *ops)
{
    ops->delay(ops->delay_time);
}

/**
  * @brief  Software I2C delay time, half time delay than function i2c_delay2.
  * @param  ops: pointer of drv_soft_i2c_bit_ops.
  * @retval NONE.
  */
void i2c_delay(struct drv_soft_i2c_bit_ops *ops)
{
    ops->delay((ops->delay_time + 1) >> 1);
}

/**
  * @brief  Set pin state.
  * @param  ptr: pointer of es32_i2c_pin_bit.
  * @param  state: 0 for change to low level, and 1 for high level.
  * @retval 0 for low level, and high for high level.
  */
void drv_soft_i2c_set_sda(void *ptr, uint8_t state)
{
    struct drv_soft_i2c_pins *i2c_pins = (struct drv_soft_i2c_pins*)ptr;
    _i2c_pin_bit_set(&i2c_pins->sda, state);
}

/**
  * @brief  Get pin state.
  * @param  ptr: pointer of es32_i2c_pin_bit.
  * @param  state: 0 for change to low level, and 1 for high level.
  * @retval 0 for low level, and high for high level.
  */
void drv_soft_i2c_set_scl(void *ptr, uint8_t state)
{
    struct drv_soft_i2c_pins *i2c_pins = (struct drv_soft_i2c_pins*)ptr;
    _i2c_pin_bit_set(&i2c_pins->scl, state);
}

/**
  * @brief  Get pin state.
  * @param  ptr: pointer of es32_i2c_pin_bit.
  * @retval 0 for low level, and high for high level.
  */
uint8_t drv_soft_i2c_get_sda(void *ptr)
{
    struct drv_soft_i2c_pins *i2c_pins = (struct drv_soft_i2c_pins*)ptr;
    return _i2c_pin_bit_get(&i2c_pins->sda);
}

/**
  * @brief  Get pin state.
  * @param  ptr: pointer of es32_i2c_pin_bit.
  * @retval 0 for low level, and high for high level.
  */
uint8_t drv_soft_i2c_get_scl(void *ptr)
{
    struct drv_soft_i2c_pins *i2c_pins = (struct drv_soft_i2c_pins*)ptr;
    return _i2c_pin_bit_get(&i2c_pins->scl);
}

/**
  * @brief  Release software I2C SCL.
  * @param  ops: pointer of drv_soft_i2c_bit_ops.
  * @retval 0: ACK, 1:NACK.
  */
static int32_t SCL_H(struct drv_soft_i2c_bit_ops *ops)
{
    uint32_t start;

    SET_SCL(ops, 1);

    if (!ops->get_scl)
        goto done;

    start = md_get_tick();
    while (!GET_SCL(ops))
    {
        if ((md_get_tick() - start) > ops->timeout)
            return -DRV_SOFT_I2C_TIMEOUT;
        i2c_delay(ops);
    }

done:
    i2c_delay(ops);

    return DRV_SOFT_I2C_OK;
}

/**
  * @brief  Set software I2C start condition.
  * @param  ops: pointer of drv_soft_i2c_bit_ops.
  * @retval None.
  */
static void i2c_start(struct drv_soft_i2c_bit_ops *ops)
{
    SDA_L(ops);
    i2c_delay(ops);
    SCL_L(ops);
	
	return;
}

/**
  * @brief  Set software I2C restart condition.
  * @param  ops: pointer of drv_soft_i2c_bit_ops.
  * @retval None.
  */
static void i2c_restart(struct drv_soft_i2c_bit_ops *ops)
{
    SDA_H(ops);
    SCL_H(ops);
    i2c_delay(ops);
    SDA_L(ops);
    i2c_delay(ops);
    SCL_L(ops);
	
	return;
}

/**
  * @brief  Set software I2C stop condition.
  * @param  ops: pointer of drv_soft_i2c_bit_ops.
  * @retval None.
  */
static void i2c_stop(struct drv_soft_i2c_bit_ops *ops)
{
    SDA_L(ops);
    i2c_delay(ops);
    SCL_H(ops);
    i2c_delay(ops);
    SDA_H(ops);
    i2c_delay2(ops);
	
	return;
}

/**
  * @brief  Software I2C wait for an ack.
  * @param  ops: pointer of drv_soft_i2c_bit_ops.
  * @retval 0: ACK, 1:NACK.
  */
static int32_t i2c_wait_ack(struct drv_soft_i2c_bit_ops *ops)
{
    uint8_t ack;

    SDA_H(ops);
    i2c_delay(ops);

    if (SCL_H(ops) < 0)
    {
        return -DRV_SOFT_I2C_TIMEOUT;
    }

    ack = GET_SDA(ops);

    SCL_L(ops);

    return ack;
}

/**
  * @brief  Software I2C send ack or nack.
  * @param  bus: pointer of drv_soft_i2c_bus_device.
  *			ack: 0 for ack, and 1 for nack
  * @retval 0 for success, or the error code.
  */
static int32_t i2c_send_ack_or_nack(struct drv_soft_i2c_bus_device *bus, int ack)
{
    struct drv_soft_i2c_bit_ops *ops = (struct drv_soft_i2c_bit_ops *)bus->priv;

    if (!ack)
        SET_SDA(ops, 0);
	
    i2c_delay(ops);
    if (SCL_H(ops) < 0)
    {
        return -DRV_SOFT_I2C_TIMEOUT;
    }
    SCL_L(ops);

    return DRV_SOFT_I2C_OK;
}

/**
  * @brief  Software I2C send a byte.
  * @param  bus: pointer of drv_soft_i2c_bus_device.
  *			data: the data which should be sent out.
  * @retval 0:OK.
  */
static int32_t i2c_write_byte(struct drv_soft_i2c_bus_device *bus, uint8_t data)
{
    int32_t i;
    uint8_t bit;

    struct drv_soft_i2c_bit_ops *ops = (struct drv_soft_i2c_bit_ops *)bus->priv;

    for (i = 7; i >= 0; i--)
    {
        SCL_L(ops);
        bit = (data >> i) & 1;
        SET_SDA(ops, bit);
        i2c_delay(ops);
        if (SCL_H(ops) < 0)
        {
            return -DRV_SOFT_I2C_TIMEOUT;
        }
    }
    SCL_L(ops);
    i2c_delay(ops);

    return i2c_wait_ack(ops);
}

/**
  * @brief  Software I2C read a byte.
  * @param  bus: pointer of drv_soft_i2c_bus_device.
  * @retval byte data read from i2c bus.
  */
static int32_t i2c_read_byte(struct drv_soft_i2c_bus_device *bus)
{
    uint8_t i;
    uint8_t data = 0;
    struct drv_soft_i2c_bit_ops *ops = (struct drv_soft_i2c_bit_ops *)bus->priv;

    SDA_H(ops);
    i2c_delay(ops);
    for (i = 0; i < 8; i++)
    {
        data <<= 1;

        if (SCL_H(ops) < 0)
        {
            return -DRV_SOFT_I2C_TIMEOUT;
        }

        if (GET_SDA(ops))
            data |= 1;
        SCL_L(ops);
        i2c_delay(ops);
    }

    return data;
}

/**
  * @brief  Software I2C send data bytes.
  * @param  bus: pointer of drv_soft_i2c_bus_device.
  *			msg: pointer of drv_soft_i2c_msg.
  * @retval the byte number of data sent out, or the error code.
  */
static uint32_t i2c_send_bytes(struct drv_soft_i2c_bus_device *bus, struct drv_soft_i2c_msg *msg)
{
    int32_t ret;
    uint32_t bytes = 0;
    const uint8_t *ptr = msg->buf;
    int32_t count = msg->len;
    uint16_t ignore_nack = msg->flags & DRV_SOFT_I2C_IGNORE_NACK;

    while (count > 0)
    {
        ret = i2c_write_byte(bus, *ptr);

        if ((ret == 0) || (ignore_nack && (ret == 1)))
        {
            count--;
            ptr++;
            bytes++;
        }
        else if (ret == 1)
        {
            return 0;
        }
        else
        {
            return ret;
        }
    }

    return bytes;
}

/**
  * @brief  Software I2C receive data bytes.
  * @param  bus: pointer of drv_soft_i2c_bus_device.
  *			msg: pointer of drv_soft_i2c_msg.
  * @retval byte data read from i2c bus.
  */
static uint32_t i2c_recv_bytes(struct drv_soft_i2c_bus_device *bus, struct drv_soft_i2c_msg *msg)
{
    int32_t val;
    int32_t bytes = 0;
    uint8_t *ptr = msg->buf;
    int32_t count = msg->len;
    const uint32_t flags = msg->flags;

    while (count > 0)
    {
        val = i2c_read_byte(bus);
        if (val >= 0)
        {
            *ptr = val;
            bytes++;
        }
        else
        {
            break;
        }

        ptr++;
        count--;

        if (!(flags & DRV_SOFT_I2C_NO_READ_ACK))
        {
            val = i2c_send_ack_or_nack(bus, (count == 0 ? 1 : 0));
            if (val < 0)
                return val;
        }
    }

    return bytes;
}

/**
  * @brief  Software I2C send address.
  * @param  bus: pointer of drv_soft_i2c_bus_device.
  *			addr: I2C chip address.
  *			retries: retry times.
  * @retval 0 for success, or the error code.
  */
static int32_t i2c_send_address(struct drv_soft_i2c_bus_device *bus, uint8_t addr, int32_t retries)
{
    struct drv_soft_i2c_bit_ops *ops = (struct drv_soft_i2c_bit_ops *)bus->priv;
    int32_t i;
    int ret = 0;

    for (i = 0; i <= retries; i++)
    {
        ret = i2c_write_byte(bus, addr);
        if (ret == 0 || i == retries)
            break;
		
        i2c_stop(ops);
        i2c_delay2(ops);
        i2c_start(ops);
    }

    return ret;
}

/**
  * @brief  Software I2C send address.
  * @param  bus: pointer of drv_soft_i2c_bus_device.
  *			msg: pointer of drv_soft_i2c_msg.
  * @retval byte data read from i2c bus.
  */
static int32_t i2c_bit_send_address(struct drv_soft_i2c_bus_device *bus, struct drv_soft_i2c_msg *msg)
{
    uint8_t addr1, addr2;
    int32_t retries;
    int32_t ret;
    uint16_t flags = msg->flags;
    uint16_t ignore_nack = msg->flags & DRV_SOFT_I2C_IGNORE_NACK;
    struct drv_soft_i2c_bit_ops *ops = (struct drv_soft_i2c_bit_ops *)bus->priv;

    retries = ignore_nack ? 0 : bus->retries;

    if (flags & DRV_SOFT_I2C_ADDR_10BIT)
    {
        addr1 = 0xf0 | ((msg->addr >> 7) & 0x06);
        addr2 = msg->addr & 0xff;

        ret = i2c_send_address(bus, addr1, retries);
        if ((ret == 1) && !ignore_nack)
        {
            return -DRV_SOFT_I2C_IO_ERR;
        }

        ret = i2c_write_byte(bus, addr2);
        if ((ret == 1) && !ignore_nack)
        {
            return -DRV_SOFT_I2C_IO_ERR;
        }
        if (flags & DRV_SOFT_I2C_RD)
        {
            i2c_restart(ops);
            addr1 |= 0x01;
            ret = i2c_send_address(bus, addr1, retries);
            if ((ret == 1) && !ignore_nack)
            {
				return -DRV_SOFT_I2C_IO_ERR;
            }
        }
    }
    else
    {
        /* 7-bit addr */
        addr1 = msg->addr << 1;
        if (flags & DRV_SOFT_I2C_RD)
            addr1 |= 1;
        ret = i2c_send_address(bus, addr1, retries);
        if ((ret == 1) && !ignore_nack)
            return -DRV_SOFT_I2C_IO_ERR;
    }

    return DRV_SOFT_I2C_OK;
}

/**
  * @brief  Software I2C communication.
  * @param  bus: pointer of drv_soft_i2c_bus_device.
  *			msg: pointer of drv_soft_i2c_msg.
  *			num: the number of data which should be sent out or receive.
  * @retval the number of msg which send out or receive successful.
  */
int32_t i2c_bit_xfer(struct drv_soft_i2c_bus_device *bus, struct drv_soft_i2c_msg msgs[], uint32_t num)
{
    struct drv_soft_i2c_msg *msg;
    struct drv_soft_i2c_bit_ops *ops = (struct drv_soft_i2c_bit_ops *)bus->priv;
    int32_t i, ret;
    uint16_t ignore_nack;

    if (num == 0)
		return 0;

    for (i = 0; i < num; i++)
    {
        msg = &msgs[i];
        ignore_nack = msg->flags & DRV_SOFT_I2C_IGNORE_NACK;
        if (!(msg->flags & DRV_SOFT_I2C_NO_START))
        {
            if (i)
            {
                i2c_restart(ops);
            }
            else
            {
                i2c_start(ops);
            }
            ret = i2c_bit_send_address(bus, msg);
            if ((ret != DRV_SOFT_I2C_OK) && !ignore_nack)
            {
                goto out;
            }
        }
        if (msg->flags & DRV_SOFT_I2C_RD)
        {
            ret = i2c_recv_bytes(bus, msg);
            if (ret < msg->len)
            {
                if (ret >= 0)
                    ret = -DRV_SOFT_I2C_IO_ERR;
                goto out;
            }
        }
        else
        {
            ret = i2c_send_bytes(bus, msg);
            if (ret < msg->len)
            {
                if (ret >= 0)
                    ret = -DRV_SOFT_I2C_IO_ERR;
                goto out;
            }
        }
    }
    ret = i;

out:
    if (!(msg->flags & DRV_SOFT_I2C_NO_STOP))
    {
        i2c_stop(ops);
    }

    return ret;
}

/**
  * @brief  Initialize the software I2C bus object.
  * @param  bus:pointer of drv_soft_i2c_bus_device.
  * @retval NONE.
  */
void setup_soft_i2c_bus(struct drv_soft_i2c_bus_device *bus)
{
    bus->ops = &soft_i2c_ops;
    bus->retries = 3;
    bus->timeout = 1000;
	
	return;
}

/****************************************   Common frame function end **********************************************/

/**
  * @brief  Initialize the software I2C.
  * @param  NONE.
  * @retval NONE.
  */
uint8_t drv_soft_i2c_init(void)
{
    md_gpio_init_t a;
	
    memset((void *)&_i2c_device0, 0, sizeof(struct drv_soft_i2c_bus_device));
    _i2c_device0.priv = &es32_i2c_bit_ops;
	
    setup_soft_i2c_bus(&_i2c_device0);
	
    /*The host SCL pins initialization*/
	es32_i2c_bit_ops.get_scl = drv_soft_i2c_get_scl;
	es32_i2c_bit_ops.set_scl = drv_soft_i2c_set_scl;
	es32_i2c_bit_ops.get_sda = drv_soft_i2c_get_sda;
	es32_i2c_bit_ops.set_sda = drv_soft_i2c_set_sda;
	es32_i2c_bit_ops.delay = i2c_bit_delay;
	es32_i2c_bit_ops.data = &soft_i2c0_pins;
	es32_i2c_bit_ops.delay_time = 2;
	es32_i2c_bit_ops.timeout = 1000;
	
    /*The host SCL pins initialization*/
    a.mode = MD_GPIO_MODE_OUTPUT;
    a.odos = MD_GPIO_OPEN_DRAIN;    /* Open-Drain output*/
    a.pupd = MD_GPIO_PUSH_UP;
    a.odrv = MD_GPIO_OUT_DRIVE_STRONG;
    a.flt  = MD_GPIO_FILTER_DISABLE;
    a.type = MD_GPIO_TYPE_CMOS;
    a.func = MD_GPIO_FUNC_1;
    md_gpio_init(SOFT_I2C0_SCL_PORT, SOFT_I2C0_SCL_PIN, &a);
    md_gpio_init(SOFT_I2C0_SDA_PORT, SOFT_I2C0_SDA_PIN, &a);
    
    _i2c_pin_bit_set(&soft_i2c0_pins.scl, 1);
    _i2c_pin_bit_set(&soft_i2c0_pins.sda, 1);
    
    return 0;
}
