/**********************************************************************************
 *
 * @file    drv_soft_i2c.h
 * @brief   header file of drv_soft_i2c.c
 *
 * @date    29 Aug 2022
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          29 Aug 2022     biyq            the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

#ifndef   __DRV_SOFT_I2C_H__
#define   __DRV_SOFT_I2C_H__


/* Includes ------------------------------------------------------------------ */
#include <stdint.h>
#include <string.h>
#include <md_conf.h>


/* Exported Types ------------------------------------------------------------ */

/**
  * @brief  Define software i2c bit operation.
  */
struct drv_soft_i2c_bit_ops
{
    void	*data;            						/*  */
    void	(*set_sda)(void *data, uint8_t state);
    void	(*set_scl)(void *data, uint8_t state);
    uint8_t (*get_sda)(void *data);
    uint8_t (*get_scl)(void *data);
    void	(*delay)(uint32_t us);					/* in tick */
    uint32_t delay_time;  							/* scl and sda line delay */
    uint32_t timeout;								/* in tick */
};

/* Forward declaration */
struct drv_soft_i2c_bus_device_ops;
/**
  * @brief  Define software i2c device.
  */
struct drv_soft_i2c_bus_device
{
    const struct drv_soft_i2c_bus_device_ops *ops;
    uint16_t	flags;
    uint32_t	timeout;
    uint32_t	retries;
    void		*priv;
};

/**
  * @brief  Define software i2c msg constitution.
  */
struct drv_soft_i2c_msg
{
    uint16_t	addr;
    uint16_t	flags;
    uint16_t	len;
    uint8_t		*buf;
};

/**
  * @brief  Define software i2c device operation.
  */
struct drv_soft_i2c_bus_device_ops
{
    int32_t (*master_xfer)(struct drv_soft_i2c_bus_device *bus, struct drv_soft_i2c_msg msgs[], uint32_t num);
    int32_t (*slave_xfer)(struct drv_soft_i2c_bus_device *bus, struct drv_soft_i2c_msg msgs[], uint32_t num);
    int32_t (*i2c_bus_control)(struct drv_soft_i2c_bus_device *bus, uint32_t, uint32_t);
};

/**
  * @brief  Define software i2c pin constitution information.
  */
struct drv_soft_i2c_pin{
    GPIO_TypeDef	*gpio;
    uint32_t		pin_num;
};

/**
  * @brief  Define software i2c pin information struction.
  */
struct drv_soft_i2c_pins{
    struct drv_soft_i2c_pin scl;
    struct drv_soft_i2c_pin sda;
};

/* Exported Macros ----------------------------------------------------------- */
/* I2C IO defination */
#define SOFT_I2C0_SCL_PORT  GPIOA
#define SOFT_I2C0_SCL_PIN   MD_GPIO_PIN_5
#define SOFT_I2C0_SDA_PORT  GPIOA
#define SOFT_I2C0_SDA_PIN   MD_GPIO_PIN_6  

/* Software I2C execution result */
#define DRV_SOFT_I2C_OK						(0)       /* No error */
#define DRV_SOFT_I2C_ERR					(1)       /* Error happens */
#define DRV_SOFT_I2C_TIMEOUT				(2)       /* Timed out */
#define DRV_SOFT_I2C_IO_ERR					(8)       /* IO error */

/* Software I2C paramenter */
#define DRV_SOFT_I2C_WR						(0x0000)
#define DRV_SOFT_I2C_RD						(1u << 0)
#define DRV_SOFT_I2C_ADDR_10BIT				(1u << 2)  /* 10 bit chip address */
#define DRV_SOFT_I2C_NO_START				(1u << 4)
#define DRV_SOFT_I2C_IGNORE_NACK			(1u << 5)
#define DRV_SOFT_I2C_NO_READ_ACK			(1u << 6)  /* Do not ACK, when I2C reading process*/
#define DRV_SOFT_I2C_NO_STOP				(1u << 7)


/* Exported Variables -------------------------------------------------------- */


/* Exported Constants -------------------------------------------------------- */

/* Exported Functions -------------------------------------------------------- */

/* Exported Types ------------------------------------------------------------ */

/* Exported Macros ----------------------------------------------------------- */

/* Exported Variables -------------------------------------------------------- */
extern struct drv_soft_i2c_bus_device _i2c_device0;

/* Exported Constants -------------------------------------------------------- */

/* Exported Functions -------------------------------------------------------- */
int32_t i2c_bit_xfer(struct drv_soft_i2c_bus_device *bus, struct drv_soft_i2c_msg msgs[], uint32_t num);
uint8_t drv_soft_i2c_init(void);

#endif     /*__DRV_SOFT_I2C_H__*/

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
