/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    30 Apri 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          30 Apri 2021    yanght          the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */

#include <string.h>
#include "main.h"
void SystemInit(void)
{}
/* Private Macros ------------------------------------------------------------ */

/* Private Variables --------------------------------------------------------- */

/* Public Variables ---------------------------------------------------------- */

md_i2c_init_t g_i2c_h;
md_dma_descriptor_t g_dma0_ctrl_base[2] __attribute__((aligned(512)));
md_dma_config_t g_dma_tx_cfg, g_dma_rx_cfg;

uint8_t g_recv_buf[20];
uint32_t g_recv_len = sizeof(g_recv_buf);

uint8_t g_send_buf[20] = {0x01U, 0x12U, 0x23U, 0x34U, 0x45U, 0x56U, 0x67U, 0x78U, 0x89U, 0x9AU, 0x01U, 0x12U, 0x23U, 0x34U, 0x45U, 0x56U, 0x67U, 0x78U, 0x89U, 0x9AU};
uint32_t g_send_len = sizeof(g_send_buf);

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Function ---------------------------------------------------------- */

/*From machine mode communication pins initialization function*/
void i2c_pin_init(void)
{
    md_gpio_init_t a;

    /*From the machine initialization SCL pins*/
    a.mode = MD_GPIO_MODE_OUTPUT;
    a.odos = MD_GPIO_OPEN_DRAIN;    /*Open drain output*/
    a.pupd = MD_GPIO_PUSH_UP;
    a.odrv = MD_GPIO_OUT_DRIVE_NORMAL;
    a.flt  = MD_GPIO_FILTER_DISABLE;
    a.type = MD_GPIO_TYPE_CMOS;
    a.func = MD_GPIO_FUNC_3;
    md_gpio_init(I2C_SCL_PORT, I2C_SCL_PIN, &a);

    /*From the machine initialization SDA pin*/
    a.mode = MD_GPIO_MODE_OUTPUT;
    a.odos = MD_GPIO_OPEN_DRAIN;    /*Open drain output*/
    a.pupd = MD_GPIO_PUSH_UP;
    a.odrv = MD_GPIO_OUT_DRIVE_NORMAL;
    a.flt  = MD_GPIO_FILTER_DISABLE;
    a.type = MD_GPIO_TYPE_CMOS;
    a.func = MD_GPIO_FUNC_3;
    md_gpio_init(I2C_SDA_PORT, I2C_SDA_PIN, &a);

    return;
}

/*I2C initialization function*/
void initialize_i2c_slave(void)
{
    /*Will the I2C initialization structure is set to the default values*/
    md_i2c_struct_init(&g_i2c_h);

    /*User configuration I2C*/
    g_i2c_h.addr_mode = MD_I2C_ADDR_7BIT;
    g_i2c_h.clk_speed = 400000;
    g_i2c_h.dual_addr = MD_I2C_DUALADDR_DISABLE;
    g_i2c_h.duty = MD_I2C_DUTYCYCLE_16_9;
    g_i2c_h.general_call = MD_I2C_GENERALCALL_DISABLE;
    g_i2c_h.stretch = MD_I2C_STRETCH_ENABLE;
    g_i2c_h.own_addr1 =  0x2D << 1;

    /*Initialize the I2C*/
    md_i2c_init(I2C0, &g_i2c_h);

    /*Make can receive complete automatic reply*/
    md_i2c_enable_ack(I2C0);

    md_i2c_enable_it_evt(I2C0);

}

/*Send the DMA function*/
void dma_tx(void)
{
    memset(&g_dma_tx_cfg, 0x0, sizeof(g_dma_tx_cfg));
    md_dma_set_ctrlbase((uint32_t)&g_dma0_ctrl_base);

    /*To enable the DMA*/
    md_dma_enable();

    /*User configuration DMA*/
    md_dma_config_struct(&g_dma_tx_cfg);
    g_dma_tx_cfg.data_width = MD_DMA_DATA_SIZE_BYTE;
    g_dma_tx_cfg.src_inc    = MD_DMA_DATA_INC_BYTE;       /*The source data address byte increments*/
    g_dma_tx_cfg.dst_inc    = MD_DMA_DATA_INC_NONE;       /*No address incremental target data*/
    g_dma_tx_cfg.R_power    = MD_DMA_R_POWER_1;
    g_dma_tx_cfg.primary    = ENABLE;
    g_dma_tx_cfg.burst      = DISABLE;
    g_dma_tx_cfg.high_prio  = DISABLE;
    g_dma_tx_cfg.interrupt  = ENABLE;
    g_dma_tx_cfg.channel      = 0;
    g_dma_tx_cfg.src          = (void *)(g_send_buf);       /*The source data*/
    g_dma_tx_cfg.dst          = (void *)(&I2C0->DATA);        /*The target data*/
    g_dma_tx_cfg.size         = g_send_len;       /*The length of the data*/
    g_dma_tx_cfg.msel         = MD_DMA_MSEL_I2C0;
    g_dma_tx_cfg.msigsel      = MD_DMA_MSIGSEL_I2C_TXEMPTY;       /*I2C0 send*/

}

/*The DMA receive function*/
void dma_rx(void)
{
    memset(&g_dma_rx_cfg, 0x0, sizeof(g_dma_rx_cfg));
    md_dma_set_ctrlbase((uint32_t)&g_dma0_ctrl_base);

    /*To enable the DMA*/
    md_dma_enable();

    /*User configuration DMA*/
    md_dma_config_struct(&g_dma_rx_cfg);
    g_dma_rx_cfg.data_width = MD_DMA_DATA_SIZE_BYTE;
    g_dma_rx_cfg.src_inc    = MD_DMA_DATA_INC_NONE;       /*Source data is no address increment*/
    g_dma_rx_cfg.dst_inc    = MD_DMA_DATA_INC_BYTE;       /*The target data address byte increments*/
    g_dma_rx_cfg.R_power    = MD_DMA_R_POWER_1;
    g_dma_rx_cfg.primary    = ENABLE;
    g_dma_rx_cfg.burst      = DISABLE;
    g_dma_rx_cfg.high_prio  = DISABLE;
    g_dma_rx_cfg.interrupt  = ENABLE;
    g_dma_rx_cfg.channel      = 1;
    g_dma_rx_cfg.src          = (void *)(&I2C0->DATA);        /*The source data*/
    g_dma_rx_cfg.dst          = (void *)(g_recv_buf);       /*The target data*/
    g_dma_rx_cfg.size         = g_recv_len;       /*The length of the data*/
    g_dma_rx_cfg.msel         = MD_DMA_MSEL_I2C0;
    g_dma_rx_cfg.msigsel      = MD_DMA_MSIGSEL_I2C_RNR;       /*I2C0 receive*/

}

/*The main function*/
int main(void)
{
    /* Configure system clock */
    md_cmu_clock_config(MD_CMU_CLOCK_HRC, 32000000);
    /* Initialize SysTick Interrupt */
    md_init_1ms_tick();

    /* Enable ALL peripheral */
    SYSCFG_UNLOCK();
    md_cmu_enable_perh_all();
    SYSCFG_LOCK();
    /*Interrupt priority level*/
    md_mcu_irq_config(I2C0_IRQn, 0, ENABLE);

    /*Pin initialization of communication*/
    i2c_pin_init();

    /*I2C initialization*/
    initialize_i2c_slave();

    md_mcu_irq_config(DMA_IRQn, 2, ENABLE);
    memset(g_dma0_ctrl_base, 0x0, sizeof(g_dma0_ctrl_base));
    dma_rx();
    dma_tx();

    while (1)
    {

    }

}

/**
  * @} Examples_I2C
  */

/**
  * @} Projects_Examples_MD
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
