/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    30 Apri 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          30 Apri 2021    yanght          the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */
/* Includes ------------------------------------------------------------------ */
#include "main.h"

/* Private Macros ------------------------------------------------------------ */
#define BUFFER_SIZE  10
#define CAPTURE_CH1_PORT GPIOA
#define CAPTURE_CH1_PIN  MD_GPIO_PIN_6

/* Private Variables --------------------------------------------------------- */
md_dma_descriptor_t dma_ctrl_base[2] __attribute__((aligned(128)));
md_dma_config_t g_dma_config;
/* duty/cycle data buffer */
uint16_t g_duty_buffer[BUFFER_SIZE] __attribute__((aligned(4))) = {0U};
uint16_t g_cycle_buffer[BUFFER_SIZE] __attribute__((aligned(4))) = {0U};

/* Public Variables ---------------------------------------------------------- */

/* Private Constants --------------------------------------------------------- */
/* Private function prototypes ----------------------------------------------- */
/* Private Function ---------------------------------------------------------- */

/**
  * @brief  Initializate system.
  * @retval None
  */
void SystemInit(void)
{

}

/**
  * @brief  Initialize Capture port.
  * @param  None
  * @retval None
  */
static void init_gpio(void)
{
    md_gpio_init_t gpio_init;

    /* Initialize CH1IN */
    md_gpio_init_struct(&gpio_init);         /* initialize the gpio_init */
    gpio_init.mode = MD_GPIO_MODE_INPUT;     /* input */
    gpio_init.odos = MD_GPIO_PUSH_PULL;
    gpio_init.pupd = MD_GPIO_FLOATING;       /* no pull up and pull down */
    gpio_init.odrv = MD_GPIO_OUT_DRIVE_NORMAL;
    gpio_init.flt = MD_GPIO_FILTER_DISABLE;  /* no filter */
    gpio_init.type = MD_GPIO_TYPE_CMOS;      /* CMOS */
    gpio_init.func = MD_GPIO_FUNC_5;         /* GPIO function */
    md_gpio_init(CAPTURE_CH1_PORT, CAPTURE_CH1_PIN, &gpio_init);

    return;
}

/**
  * @brief:  Initialize GP16C2T0 to timely overflow funtion
  * @param:  None
  * @retval: None
  */
static void init_timer(void)
{
    md_timer_base_init_t gp16c2t_init;
    md_timer_ic_init_t ic_init;

    md_timer_base_struct_init(&gp16c2t_init);   /* initialize the gp16c2t_init  */
    gp16c2t_init.prescaler = 32 - 1;            /* clk_count: 1MHz */
    gp16c2t_init.clk_div = MD_TIMER_CLOCK_DIV1; /* working clock of dead time and filter */
    gp16c2t_init.mode = MD_TIMER_CNT_MODE_UP;   /* count up */
    gp16c2t_init.period = 0xFFFF;               /* period is max count clock */
    gp16c2t_init.re_cnt = 1 - 1;                /* 0 repeat count */
    md_timer_base_set_config(GP16C2T0, &gp16c2t_init);

    /* initialize the ic_init  */
    md_timer_ic_struct_init(&ic_init);
    ic_init.filter      = 0;                           /* 0 filter */
    ic_init.polarity    = MD_TIMER_IC_POLARITY_RISE;   /* capture rising edge */
    ic_init.psc         = MD_TIMER_IC_PSC_DIV1;        /* no prescaler in capture channel */
    ic_init.sel         = MD_TIMER_IC_SEL_DIRECT;      /* capture this channel */
    md_timer_ic_init(GP16C2T0, MD_TIMER_CHANNEL_1, &ic_init);
    md_timer_ic_struct_init(&ic_init);
    ic_init.filter      = 0;
    ic_init.polarity    = MD_TIMER_IC_POLARITY_FALL;   /* capture falling edge */
    ic_init.psc         = MD_TIMER_IC_PSC_DIV1;        /* no prescaler in capture channel */
    ic_init.sel         = MD_TIMER_IC_SEL_INDIRECT;    /* capture adjacent channel */
    md_timer_ic_init(GP16C2T0, MD_TIMER_CHANNEL_2, &ic_init);

    md_timer_set_slave_mode_smods(GP16C2T0, 4);        /* 0x4: Reset Mode */
    md_timer_set_slave_trigger_tssel(GP16C2T0, 5);     /* 0x5: Filtered Timer Input 1 */

    md_timer_set_cc_dma_select_ccdmasel(GP16C2T0, 0);  /* CCn DMA request sent when CCx event occurs */
    md_timer_enable_cc1dma(GP16C2T0);        /* enable CC1 DMA request */
    md_timer_enable_cc2dma(GP16C2T0);        /* enable CC2 DMA request */

    md_timer_enable_cc1it_interrupt(GP16C2T0);         /* enable CC1 interrupt request */
    md_timer_enable_cc2it_interrupt(GP16C2T0);         /* enable CC1 interrupt request */
    md_timer_enable_uit_interrupt(GP16C2T0);           /* enable update interrupt request */

    md_mcu_irq_config(GPTIMC0_IRQn, 0, ENABLE);

    /* enable GP16C2T0 */
    md_timer_enable_counter_cnten(GP16C2T0);

    return;
}
/**
  * @brief:  Use MD library to config DMA.
  * @param:  None
  * @retval: None
  */
static void init_dma(void)
{
    md_dma_set_ctrlbase((uint32_t)dma_ctrl_base);
    md_dma_config_struct(&g_dma_config);

    md_dma_enable();
    g_dma_config.src      = (void *)(&GP16C2T0->CCVAL1);  /* Source data begin pointer */
    g_dma_config.dst      = (void *)g_cycle_buffer;         /* Destination data begin pointer */
    g_dma_config.size     = BUFFER_SIZE;                  /* The total number of DMA transfers that DMA cycle contains */
    g_dma_config.data_width = MD_DMA_DATA_SIZE_HALFWORD;  /* Data width */
    g_dma_config.src_inc  = MD_DMA_DATA_INC_NONE;         /* Source increment type */
    g_dma_config.dst_inc  = MD_DMA_DATA_INC_HALFWORD;     /* Destination increment type */
    g_dma_config.R_power  = MD_DMA_R_POWER_1;             /* Control how many DMA transfers can occur before re-arbitrates */
    g_dma_config.primary  = ENABLE;                       /* Use primary descriptor or alternate descriptor */
    g_dma_config.burst    = DISABLE;                      /* Uses the alternate data structure when complete a DMA cycle */
    g_dma_config.high_prio    = DISABLE;                  /* High priority or default priority */
    g_dma_config.interrupt    = ENABLE;                   /* Enable/disable interrupt */
    g_dma_config.msel     = MD_DMA_MSEL_GP16C2T0;           /* Input source to DMA channel */
    g_dma_config.msigsel  = MD_DMA_MSIGSEL_TIMER_CH1;     /* Input signal to DMA channel */
    g_dma_config.channel  = 0;                            /* Channel index */
    md_dma_config_base(DMA0, MD_DMA_CYCLE_CTRL_BASIC, &g_dma_config);

    md_dma_set_ctrlbase((uint32_t)dma_ctrl_base);
    md_dma_config_struct(&g_dma_config);
    g_dma_config.src      = (void *)(&GP16C2T0->CCVAL2);  /* Source data begin pointer */
    g_dma_config.dst      = (void *)g_duty_buffer;          /* Destination data begin pointer */
    g_dma_config.size     = BUFFER_SIZE;                  /* The total number of DMA transfers that DMA cycle contains */
    g_dma_config.data_width = MD_DMA_DATA_SIZE_HALFWORD;  /* Data width */
    g_dma_config.src_inc  = MD_DMA_DATA_INC_NONE;         /* Source increment type */
    g_dma_config.dst_inc  = MD_DMA_DATA_INC_HALFWORD;     /* Destination increment type */
    g_dma_config.R_power  = MD_DMA_R_POWER_1;             /* Control how many DMA transfers can occur before re-arbitrates */
    g_dma_config.primary  = ENABLE;                       /* Use primary descriptor or alternate descriptor */
    g_dma_config.burst    = DISABLE;                      /* Uses the alternate data structure when complete a DMA cycle */
    g_dma_config.high_prio    = DISABLE;                  /* High priority or default priority */
    g_dma_config.interrupt    = ENABLE;                   /* Enable/disable interrupt */
    g_dma_config.msel     = MD_DMA_MSEL_GP16C2T0;           /* Input source to DMA channel */
    g_dma_config.msigsel  = MD_DMA_MSIGSEL_TIMER_CH2;     /* Input signal to DMA channel */
    g_dma_config.channel  = 1;                            /* Channel index */
    md_dma_config_base(DMA0, MD_DMA_CYCLE_CTRL_BASIC, &g_dma_config);

    md_dma_enable_channel(MD_DMA_CH_0);        /* enable DMA channel 0 */
    md_dma_enable_channel(MD_DMA_CH_1);        /* enable DMA channel 1 */

    return;
}
/**
  * @brief:  Main program.
  * @param:  None
  * @retval: None
  */
int main()
{
    /* Configure system clock */
    md_cmu_clock_config(MD_CMU_CLOCK_HRC, 32000000);
    /* Initialize SysTick Interrupt */
    md_init_1ms_tick();

    /* Enable ALL peripheral */
    SYSCFG_UNLOCK();
    md_cmu_enable_perh_all();
    SYSCFG_LOCK();
    /* Initialize pin */
    init_gpio();
    /* Initialize timer */
    init_timer();
    /* Initialize dma */
    init_dma();

    while (1)
    {
        if (md_dma_is_active_flag_done(MD_DMA_CH_1) && md_dma_is_active_flag_done(MD_DMA_CH_0))
        {
            md_dma_clear_flag_done(MD_DMA_CH_0);
            md_dma_clear_flag_done(MD_DMA_CH_1);

            init_dma();
        }
    }
}

/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
