/**********************************************************************************
 *
 * @file    main.c
 * @brief   ADC0_SingleConversion C file
 *
 * @date    30 Apri 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          2 June 2023     zhuxf           the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */
#include <string.h>
#include "main.h"
void SystemInit()
{}
/* Private Macros ------------------------------------------------------------ */
#define TESTPORT (*(uint32_t *)0x40080100)
#define TESTCON (*(uint32_t *)0x40080104)
/* Private Constants --------------------------------------------------------- */
/* Private function prototypes ----------------------------------------------- */
static void usart_init(void);
static void usart_gpio_init(void);
static uint32_t sample_ldo1v66(void);

/* Private Variables --------------------------------------------------------- */

/** @addtogroup Projects_Examples_MD
  * @{
  */

/* Public Variables ---------------------------------------------------------- */
md_adc_init_t g_adc_init;
uint32_t g_adc_result;

/* Private Function ---------------------------------------------------------- */

/** @addtogroup Public_Function
  * @{
  */

/**
  * @brief  Configure the ADC0.
  * @param  None
  * @retval None
  */
void adc_base_config()
{
    md_adc_struct_init(&g_adc_init);

    /* Set ADC0 Base Configuration:  */
    g_adc_init.align    = MD_ADC_DATAALIGN_RIGHT;     /* Specifies ADC0 data alignment */
    g_adc_init.data_bit = MD_ADC_CONV_BIT_12;
    g_adc_init.div      = MD_ADC_CKDIV_16;            /* ADC0CLK divider */
    g_adc_init.n_ref    = MD_ADC_NEG_REF_VSS;
    g_adc_init.p_ref    = MD_ADC_POS_REF_VDD;         /* The positive reference voltage*/
    md_adc_init(ADC0, &g_adc_init);

    /* Trigger the NCHE FALG mode */
    md_adc_set_eoc_selection_nchesel(ADC0, MD_ADC_NCHESEL_MODE_ALL);
    /*Enable ADC0 sequencers scan mode*/
    md_adc_scan_mode_enable_scanen(ADC0);
    /* Set ADC0 group REG sequencer length and scan direction,ADC0 conversion 1 channel */
    md_adc_set_normal_channel_length_nsl(ADC0, MD_ADC_NCH_NR_1);
    /* Set ADC0 continuous conversion mode on ADC0 group REG.*/
    md_adc_continuous_conversion_disable_cm(ADC0);
    /*Set ADC0 group REG sequence: channel on the selected scan sequence rank.*/
    md_adc_set_normal_1st_conv_ns1(ADC0, MD_ADC_CHANNEL_15);
    md_adc_set_smpt1_cht(ADC0, MD_ADC_SAMPLETIME_15, MD_ADC_CHANNEL_15);

    /*Enable interruption ADC0 group REG end of sequence conversions.*/
    md_adc_conv_end_interrupt_enable_ncheie(ADC0);

    return;
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
    /* Configure system clock */
    md_cmu_clock_config(MD_CMU_CLOCK_HRC, 32000000);
    /* Initialize SysTick Interrupt */
    md_init_1ms_tick();

    /* Enable ALL peripheral */
    SYSCFG_UNLOCK();
    md_cmu_enable_perh_all();
    SYSCFG_LOCK();

    /* Enable the selected ADC0 instance.*/
    md_adc_converter_enable_adcen(ADC0);
    /* Disable the selected ADC0 instance.*/
    md_adc_converter_disable_adcen(ADC0);

    /* Usart Configuration */
    usart_init();
    /* ADC0 Base Feature Configuration */
    adc_base_config();

    /*Enable the selected ADC0 instance.*/
    md_adc_converter_enable_adcen(ADC0);

    while (1)
    {
        /* enable ldo 1.66v */
        SYSCFG_UNLOCK();
        TESTPORT = 0x5A962841;
        TESTPORT = 0xE7CB69A5;
        TESTCON = 0x3F;
        TESTPORT = 0;
        SYSCFG_LOCK();

        /* sample ldo 1.66v */
        g_adc_result = sample_ldo1v66();

        /* disable ldo 1.66v */
        SYSCFG_UNLOCK();
        TESTPORT = 0x5A962841;
        TESTPORT = 0xE7CB69A5;
        TESTCON = 0x7;
        TESTPORT = 0;
        SYSCFG_LOCK();

        /* Reverse the VREF voltage */
        g_adc_result = (4096 * 1660) / g_adc_result;
        printf("VREF is %d mV\r\n", g_adc_result);

        md_delay_1ms(100);
    }
}

/**
  * @brief  Configure the usart Pins.
  * @param  None
  * @retval None
  */
void usart_gpio_init()
{

    md_gpio_init_t g_gpio_init_s;

    md_gpio_init_struct(&g_gpio_init_s);

    g_gpio_init_s.mode = MD_GPIO_MODE_OUTPUT;
    g_gpio_init_s.func = MD_GPIO_FUNC_3;

    md_gpio_init(USART_PORT, USART_TX_PIN, &g_gpio_init_s);

    g_gpio_init_s.mode = MD_GPIO_MODE_INPUT;
    md_gpio_init(USART_PORT, USART_RX_PIN, &g_gpio_init_s);
}

/**
  * @brief  Configure the usart.
  * @param  None
  * @retval None
  */
void usart_init()
{

    md_usart_init_t g_usart_init_s;

    md_cmu_enable_perh_usart0();

    usart_gpio_init();

    md_usart_init_struct(&g_usart_init_s);

    md_usart_init(USART0, &g_usart_init_s);
}

/**
  * @brief  sample ldo 1.66v.
  * @param  None
  * @retval ldo 1.66v sample value
  */
uint32_t sample_ldo1v66(void)
{
    uint8_t i;
    uint32_t max, min, sum, adc_value;

    max = 0;
    min = 0xFFF;
    sum = 0;
    
    md_adc_set_normal_1st_conv_ns1(ADC0, MD_ADC_CHANNEL_15);
    for (i = 0 ; i < 10; i++)
    {
        /* Enable ADC0 group REG conversion trigger source internal (SW start)*/
        md_adc_set_normal_channel_conv_start_nchtrg(ADC0);

        while (md_adc_get_stat_nche(ADC0) == 0);

        md_adc_set_clr_nche(ADC0);
        adc_value = md_adc_get_normal_channel_val(ADC0);
        sum += adc_value;

        if (adc_value < min)
        {
            min = adc_value;
        }

        if (adc_value > max)
        {
            max = adc_value;
        }
    }

    adc_value = (sum - max - min) / 8;

    return adc_value;
}

int fputc(int c, FILE *f)
{
    while ((READ_BIT(USART0->STAT, USART_STAT_TXEMPIF_MSK)) != USART_STAT_TXEMPIF_MSK);

    USART0->DATA = (uint8_t)c;
    return c;
}

/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
