/**********************************************************************************
 *
 * @file    main.c
 * @brief   ADC0_DMA C file
 *
 * @date    30 Apri 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          30 Apri 2021    biyq            the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */
#include <string.h>
#include "main.h"

/* Private Macros ------------------------------------------------------------ */
/* Private Constants --------------------------------------------------------- */
/* Private function prototypes ----------------------------------------------- */
/* Private Variables --------------------------------------------------------- */
md_dma_descriptor_t g_dma0_ctrl_base[1] __attribute__((aligned(128)));
md_dma_config_t g_config;
md_adc_init_t   g_adc_init;
uint16_t        g_adc_result[4] = {0U};

/* Public Variables ---------------------------------------------------------- */

/* Public Function ---------------------------------------------------------- */
void SystemInit()
{

}

/* Private Function ---------------------------------------------------------- */
/**
  * @brief  Configure the ADC0 Pins.
  * @param  None
  * @retval None
  */
void gpio_pin_g_config(void)
{
    md_gpio_init_t x;
    md_gpio_init_struct(&x);

    /* Initialize adc gpio pin */
    x.mode  = MD_GPIO_MODE_CLOSE;
    x.pupd  = MD_GPIO_FLOATING;
    x.odos  = MD_GPIO_PUSH_PULL;
    x.odrv  = MD_GPIO_OUT_DRIVE_NORMAL;
    x.flt   = MD_GPIO_FILTER_DISABLE;
    x.type  = MD_GPIO_TYPE_CMOS;
    x.func  = MD_GPIO_FUNC_0;

    md_gpio_init(ADC_CHANNEL0_GPIO_PORT, ADC_CHANNEL0_PIN, &x);
    md_gpio_init(ADC_CHANNEL1_GPIO_PORT, ADC_CHANNEL1_PIN, &x);
    md_gpio_init(ADC_CHANNEL2_GPIO_PORT, ADC_CHANNEL2_PIN, &x);
    md_gpio_init(ADC_CHANNEL3_GPIO_PORT, ADC_CHANNEL3_PIN, &x);

    return;
}

/**
  * @brief  Configure the ADC0 Pins.
  * @param  None
  * @retval None
  */
void adc_base_g_config(void)
{
    md_adc_struct_init(&g_adc_init);
    /* Set ADC0 Base Configuration:  */
    g_adc_init.align    = MD_ADC_DATAALIGN_RIGHT;   /* Specifies ADC0 data alignment */
    g_adc_init.data_bit = MD_ADC_CONV_BIT_12;
    g_adc_init.div      = MD_ADC_CKDIV_4;
    g_adc_init.n_ref    = MD_ADC_NEG_REF_VSS;
    g_adc_init.p_ref    = MD_ADC_POS_REF_VDD;
    md_adc_init(ADC0, &g_adc_init);

    /* Trigger the NCHE FALG mode */
    md_adc_set_eoc_selection_nchesel(ADC0, MD_ADC_NCHESEL_MODE_ALL);
    /*Enable ADC0 sequencers scan mode*/
    md_adc_scan_mode_enable_scanen(ADC0);
    /* Set ADC0 group REG sequencer length and scan direction,ADC0 conversion 4 channel */
    md_adc_set_normal_channel_length_nsl(ADC0, MD_ADC_NCH_NR_4);
    /* Set ADC continuous conversion mode on ADC group REG.*/
    md_adc_continuous_conversion_enable_cm(ADC0);

    /* Enable dma require */
    md_adc_dma_enable(ADC0);

    /*Set ADC0 group REG sequence: channel on the selected scan sequence rank.*/
    md_adc_set_normal_1st_conv_ns1(ADC0, MD_ADC_CHANNEL_0);
    md_adc_set_normal_2st_conv_ns2(ADC0, MD_ADC_CHANNEL_1);
    md_adc_set_normal_3st_conv_ns3(ADC0, MD_ADC_CHANNEL_2);
    md_adc_set_normal_4st_conv_ns4(ADC0, MD_ADC_CHANNEL_3);
    /* Config sample time */
    md_adc_set_smpt1_cht(ADC0, MD_ADC_SAMPLETIME_15, MD_ADC_CHANNEL_0);
    md_adc_set_smpt1_cht(ADC0, MD_ADC_SAMPLETIME_15, MD_ADC_CHANNEL_1);
    md_adc_set_smpt1_cht(ADC0, MD_ADC_SAMPLETIME_15, MD_ADC_CHANNEL_2);
    md_adc_set_smpt1_cht(ADC0, MD_ADC_SAMPLETIME_15, MD_ADC_CHANNEL_3);

    return;
}

/**
  * @brief  dma_init.
  * @param  None
  * @retval None
  */
void dma_init(void)
{
    memset(&g_config, 0x0, sizeof(md_dma_config_t));
    md_dma_enable();

    md_dma_set_ctrlbase((uint32_t)&g_dma0_ctrl_base);
    md_dma_config_struct(&g_config);
    g_config.channel    = 0;
    g_config.dst        = (void *)(g_adc_result);
    g_config.src        = (void *)(&ADC0->NCHDR);
    g_config.size       = 4;
    g_config.R_power    = MD_DMA_R_POWER_1;
    g_config.data_width = MD_DMA_DATA_SIZE_HALFWORD;
    g_config.src_inc    = MD_DMA_DATA_INC_NONE;
    g_config.dst_inc    = MD_DMA_DATA_INC_HALFWORD;
    g_config.primary    = DISABLE;
    g_config.burst      = DISABLE;
    g_config.high_prio  = DISABLE;
    g_config.interrupt  = ENABLE;
    g_config.msel       = MD_DMA_MSEL_ADC;
    g_config.msigsel    = MD_DMA_MSIGSEL_ADC;
    md_dma_config_base(DMA0, MD_DMA_CYCLE_CTRL_PINGPONG, &g_config);

    md_dma_config_struct(&g_config);
    g_config.channel    = 0;
    g_config.dst        = (void *)(g_adc_result);
    g_config.src        = (void *)(&ADC0->NCHDR);
    g_config.size       = 4;
    g_config.R_power    = MD_DMA_R_POWER_1;
    g_config.data_width = MD_DMA_DATA_SIZE_HALFWORD;
    g_config.src_inc    = MD_DMA_DATA_INC_NONE;
    g_config.dst_inc    = MD_DMA_DATA_INC_HALFWORD;
    g_config.primary    = ENABLE;
    g_config.burst      = DISABLE;
    g_config.high_prio  = DISABLE;
    g_config.interrupt  = ENABLE;
    g_config.msel       = MD_DMA_MSEL_ADC;
    g_config.msigsel    = MD_DMA_MSIGSEL_ADC;
    md_dma_config_base(DMA0, MD_DMA_CYCLE_CTRL_PINGPONG, &g_config);
    md_dma_clear_flag_done(MD_DMA_CH_0);
    md_dma_enable_channel(MD_DMA_CH_0);

    /* Set priority for ADC_IRQn and Enable ADC_IRQn */
    NVIC_SetPriority(DMA_IRQn, 0);
    NVIC_EnableIRQ(DMA_IRQn);

    return;
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
    uint8_t     i;
    volatile uint32_t   g_conv_result[4] = {0U};

    /* Configure system clock */
    md_cmu_clock_config(MD_CMU_CLOCK_HRC, 48000000);
    /* Initialize SysTick Interrupt */
    md_init_1ms_tick();

    /* Enable ALL peripheral */
    SYSCFG_UNLOCK();
    md_cmu_enable_perh_all();
    SYSCFG_LOCK();

    /* Enable the selected ADC0 instance.*/
    md_adc_converter_enable_adcen(ADC0);
    /* Disable the selected ADC0 instance.*/
    md_adc_converter_disable_adcen(ADC0);

    memset(g_dma0_ctrl_base, 0x0, sizeof(g_dma0_ctrl_base));
    /* Config DMA */
    dma_init();

    /* Pin Configuration */
    gpio_pin_g_config();
    /* ADC0 Base Feature Configuration */
    adc_base_g_config();

    /*Enable the selected ADC0 instance.*/
    md_adc_converter_enable_adcen(ADC0);

    /* Enable ADC0 group REG conversion trigger source internal (SW start)*/
    md_adc_set_normal_channel_conv_start_nchtrg(ADC0);

    while (1)
    {
        for (i = 0; i < 4; i++)
            g_conv_result[i] = (uint32_t)(g_adc_result[i]) * VDD  / 4096;
    }
}


/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
