/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    30 Apri 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          30 Apri 2021    yanght          the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */
/* Includes ------------------------------------------------------------------ */
#include "main.h"

/* Private Macros ------------------------------------------------------------ */
#define BUFFER_SIZE  10

#define CAPTURE_CH1_PORT GPIOA
#define CAPTURE_CH1_PIN  MD_GPIO_PIN_0
/* Private Variables --------------------------------------------------------- */
/* Public Variables ---------------------------------------------------------- */
md_dma_descriptor_t dma_ctrl_base[2] __attribute__((aligned(512)));
md_dma_config_t g_dma_config;
uint8_t g_cnt = 0U;

/* duty/cycle data buffer */
uint16_t g_duty_buffer[BUFFER_SIZE] __attribute__((aligned(2))) = {0U};
uint16_t g_cycle_buffer[BUFFER_SIZE] __attribute__((aligned(2))) = {0U};

/* Private Constants --------------------------------------------------------- */
/* Private function prototypes ----------------------------------------------- */
/* Private Function ---------------------------------------------------------- */

/**
  * @brief  Initializate system.
  * @retval None
  */
void SystemInit(void)
{

}

/**
  * @brief  Initialize Capture port.
  * @param  None
  * @retval None
  */
static void init_gpio(void)
{
    md_gpio_init_t gpio_init;

    /* Initialize CH1IN */
    md_gpio_init_struct(&gpio_init);         /* initialize the gpio_init */
    gpio_init.mode = MD_GPIO_MODE_INPUT;     /* input */
    gpio_init.odos = MD_GPIO_PUSH_PULL;
    gpio_init.pupd = MD_GPIO_FLOATING;       /* no pull up and pull down */
    gpio_init.odrv = MD_GPIO_OUT_DRIVE_NORMAL;
    gpio_init.flt  = MD_GPIO_FILTER_DISABLE;  /* no filter */
    gpio_init.type = MD_GPIO_TYPE_CMOS;       /* CMOS */
    gpio_init.func = MD_GPIO_FUNC_5;         /* GPIO function */
    md_gpio_init(CAPTURE_CH1_PORT, CAPTURE_CH1_PIN, &gpio_init);
}

/**
  * @brief:  Initialize AD16C4T to timely overflow funtion
  * @param:  None
  * @retval: None
  */
static void init_timer(void)
{
    md_timer_base_init_t ad16c4t_init;
    md_timer_ic_init_t ic_init;

    md_timer_base_struct_init(&ad16c4t_init);   /* initialize the ad16c4t_init  */
    ad16c4t_init.prescaler = 32 - 1;            /* clk_count: 1MHz */
    ad16c4t_init.clk_div = MD_TIMER_CLOCK_DIV1; /* working clock of dead time and filter */
    ad16c4t_init.mode = MD_TIMER_CNT_MODE_UP;   /* count up */
    ad16c4t_init.period = 0xFFFF;               /* period is max count clock */
    ad16c4t_init.re_cnt = 1 - 1;                /* 0 repeat count */
    md_timer_base_set_config(AD16C4T, &ad16c4t_init);

    /* initialize the ic_init  */
    md_timer_ic_struct_init(&ic_init);
    ic_init.filter      = 0;                            /* 0 filter */
    ic_init.polarity    = MD_TIMER_IC_POLARITY_RISE;    /* capture rising edge */
    ic_init.psc         = MD_TIMER_IC_PSC_DIV1;         /* no prescaler in capture channel */
    ic_init.sel         = MD_TIMER_IC_SEL_DIRECT;       /* capture this channel */
    md_timer_ic_init(AD16C4T, MD_TIMER_CHANNEL_1, &ic_init);
    md_timer_ic_struct_init(&ic_init);
    ic_init.filter      = 0;
    ic_init.polarity    = MD_TIMER_IC_POLARITY_FALL;    /* capture falling edge */
    ic_init.psc         = MD_TIMER_IC_PSC_DIV1;         /* no prescaler in capture channel */
    ic_init.sel         = MD_TIMER_IC_SEL_INDIRECT;     /* capture adjacent channel */
    md_timer_ic_init(AD16C4T, MD_TIMER_CHANNEL_2, &ic_init);

    md_timer_set_slave_mode_smods(AD16C4T, 4);          /* 0x4: Reset Mode */
    md_timer_set_slave_trigger_tssel(AD16C4T, 5);       /* 0x5: I1˲ */

    md_timer_set_cc_dma_select_ccdmasel(AD16C4T, 0);    /* CCn DMA request sent when CCx event occurs */
    md_timer_enable_cc1dma(AD16C4T);                    /* enable CC1 DMA request */
    md_timer_enable_cc2dma(AD16C4T);                    /* enable CC2 DMA request */

    md_timer_enable_cc1it_interrupt(AD16C4T);           /* enable CC1 interrupt request */
    md_timer_enable_cc2it_interrupt(AD16C4T);           /* enable CC1 interrupt request */
    md_timer_enable_uit_interrupt(AD16C4T);             /* enable update interrupt request */

    md_mcu_irq_config(ADTIM0_CC_IRQn, 0, ENABLE);
    md_mcu_irq_config(ADTIM0_BRK_UP_TRIGCOM_IRQn, 0, ENABLE);

    /* enable AD16C4T */
    md_timer_enable_counter_cnten(AD16C4T);

}
/**
  * @brief:  Use MD library to config DMA.
  * @param:  None
  * @retval: None
  */
void init_dma(void)
{
    md_dma_set_ctrlbase((uint32_t)dma_ctrl_base);
    md_dma_config_struct(&g_dma_config);

    md_dma_enable();
    g_dma_config.src      = (void *)(&AD16C4T->CCVAL1);  /* Source data begin pointer */
    g_dma_config.dst      = (void *)g_cycle_buffer;         /* Destination data begin pointer */
    g_dma_config.size     = BUFFER_SIZE;                  /* The total number of DMA transfers that DMA cycle contains */
    g_dma_config.data_width = MD_DMA_DATA_SIZE_HALFWORD;  /* Data width */
    g_dma_config.src_inc  = MD_DMA_DATA_INC_NONE;         /* Source increment type */
    g_dma_config.dst_inc  = MD_DMA_DATA_INC_HALFWORD;     /* Destination increment type */
    g_dma_config.R_power  = MD_DMA_R_POWER_1;             /* Control how many DMA transfers can occur before re-arbitrates */
    g_dma_config.primary  = ENABLE;                       /* Use primary descriptor or alternate descriptor */
    g_dma_config.burst    = DISABLE;                      /* Uses the alternate data structure when complete a DMA cycle */
    g_dma_config.high_prio    = DISABLE;                  /* High priority or default priority */
    g_dma_config.interrupt    = ENABLE;                   /* Enable/disable interrupt */
    g_dma_config.msel     = MD_DMA_MSEL_AD16C4T;           /* Input source to DMA channel */
    g_dma_config.msigsel  = MD_DMA_MSIGSEL_TIMER_CH1;     /* Input signal to DMA channel */
    g_dma_config.channel  = 0;                            /* Channel index */
    md_dma_config_base(DMA0, MD_DMA_CYCLE_CTRL_BASIC, &g_dma_config);

    g_dma_config.src      = (void *)(&AD16C4T->CCVAL2);  /* Source data begin pointer */
    g_dma_config.dst      = (void *)g_duty_buffer;          /* Destination data begin pointer */
    g_dma_config.size     = BUFFER_SIZE;                  /* The total number of DMA transfers that DMA cycle contains */
    g_dma_config.data_width = MD_DMA_DATA_SIZE_HALFWORD;  /* Data width */
    g_dma_config.src_inc  = MD_DMA_DATA_INC_NONE;         /* Source increment type */
    g_dma_config.dst_inc  = MD_DMA_DATA_INC_HALFWORD;     /* Destination increment type */
    g_dma_config.R_power  = MD_DMA_R_POWER_1;             /* Control how many DMA transfers can occur before re-arbitrates */
    g_dma_config.primary  = ENABLE;                       /* Use primary descriptor or alternate descriptor */
    g_dma_config.burst    = DISABLE;                      /* Uses the alternate data structure when complete a DMA cycle */
    g_dma_config.high_prio    = DISABLE;                  /* High priority or default priority */
    g_dma_config.interrupt    = ENABLE;                   /* Enable/disable interrupt */
    g_dma_config.msel     = MD_DMA_MSEL_AD16C4T;           /* Input source to DMA channel */
    g_dma_config.msigsel  = MD_DMA_MSIGSEL_TIMER_CH2;     /* Input signal to DMA channel */
    g_dma_config.channel  = 1;                            /* Channel index */
    md_dma_config_base(DMA0, MD_DMA_CYCLE_CTRL_BASIC, &g_dma_config);

    md_dma_enable_channel(MD_DMA_CH_0);        /* enable DMA channel 0 */
    md_dma_enable_channel(MD_DMA_CH_1);        /* enable DMA channel 1 */

    /* Set priority for DMA_IRQn */
    NVIC_SetPriority(DMA_IRQn, 0);
    /* Enable DMA_IRQn */
    NVIC_EnableIRQ(DMA_IRQn);
}

/**
  * @brief:  Main program.
  * @param:  None
  * @retval: None
  */
int main()
{
    /* Configure system clock */
    md_cmu_clock_config(MD_CMU_CLOCK_HRC, 32000000);
    /* Initialize SysTick Interrupt */
    md_init_1ms_tick();

    /* Enable ALL peripheral */
    SYSCFG_UNLOCK();
    md_cmu_enable_perh_all();
    SYSCFG_LOCK();
    /* Initialize pin */
    init_gpio();
    /* Initialize timer */
    init_timer();
    /* Initialize dma */
    init_dma();

    while (1)
    {
        if (g_cnt == 2)
        {
            init_dma();
            g_cnt = 0;
        }

        /* If DMA status 6 is detected, the DMA request is restarted. */
        if (READ_BITS(DMA0->STATUS, DMA_STATUS_STATUS_MSK, DMA_STATUS_STATUS_POSS) == 0x6)
        {
            /* DMA CH0 */
            if(((*(uint32_t *)0x4000CE18) & 0x1) == 0x1)
            {
                md_timer_disable_cc1dma(AD16C4T);
                md_timer_enable_cc1dma(AD16C4T);
            }

            /* DMA CH1 */
            if(((*(uint32_t *)0x4000CE18) & 0x2) == 0x2)
            {
                md_timer_disable_cc2dma(AD16C4T);
                md_timer_enable_cc2dma(AD16C4T);
            }
        }
    }
}

/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
