/**********************************************************************************
 *
 * @file    main.c
 * @brief   Main file for DEMO
 *
 * @date    30 Apri 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          30 Apri 2021    yanght          the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */

#include <string.h>
#include "main.h"
#include "iap_rom.h"

/* Private Macros ------------------------------------------------------------ */

/* Private Variables --------------------------------------------------------- */
typedef  void (*FunVoidType)(void);

/* Public Variables ---------------------------------------------------------- */

md_usart_init_t g_h_usart;

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Function ---------------------------------------------------------- */

void SystemInit(void)
{

}

/**
  * @brief  Initializate pin of usart module.
  * @retval None
  */
static void usart_pin_init(void)
{
    md_gpio_init_t x;
    md_gpio_init_struct(&x);

    /* Initialize tx pin */
    x.mode = MD_GPIO_MODE_OUTPUT;
    x.odos = MD_GPIO_PUSH_PULL;
    x.pupd = MD_GPIO_PUSH_UP;
    x.odrv = MD_GPIO_OUT_DRIVE_NORMAL;
    x.flt  = MD_GPIO_FILTER_DISABLE;
    x.type = MD_GPIO_TYPE_CMOS;
    x.func = MD_GPIO_FUNC_3;
    md_gpio_init(USART_TX_PORT, USART_TX_PIN, &x);

    /* Initialize rx pin */
    x.mode = MD_GPIO_MODE_INPUT;
    x.odos = MD_GPIO_PUSH_PULL;
    x.pupd = MD_GPIO_PUSH_UP;
    x.odrv = MD_GPIO_OUT_DRIVE_NORMAL;
    x.flt  = MD_GPIO_FILTER_DISABLE;
    x.type = MD_GPIO_TYPE_CMOS;
    x.func = MD_GPIO_FUNC_3;
    md_gpio_init(USART_RX_PORT, USART_RX_PIN, &x);

    return;
}

/**
  * @brief:  Initialize the usart.
  * @param:  None
  * @retval: None
  */
static void usart_init(void)
{
    md_usart_init_struct(&g_h_usart);
    /* Initialize USART0 */
    g_h_usart.baud        = 115200;
    g_h_usart.word_length = MD_USART_WORD_LENGTH_9B;
    g_h_usart.stop_bits   = MD_USART_STOP_BITS_1;
    g_h_usart.parity      = MD_USART_PARITY_EVEN;
    g_h_usart.fctl        = MD_USART_HW_FLOW_CTL_NONE;
    g_h_usart.mode        = MD_USART_MODE_TX_RX;

    md_usart_init(USART0, &g_h_usart);

    md_usart_enable_it_rxne(USART0);
}

/**
  * @brief:  Init gpio for LED.
  * @param:  None
  * @retval: None
  */
static void led_init(void)
{
    md_gpio_init_t gpio_init;

    /* Initialize LED pin */
    md_gpio_init_struct(&gpio_init);      /* initialize the gpio_init */
    gpio_init.mode  = MD_GPIO_MODE_OUTPUT;
    gpio_init.odos  = MD_GPIO_PUSH_PULL;
    gpio_init.pupd  = MD_GPIO_FLOATING;
    gpio_init.odrv  = MD_GPIO_OUT_DRIVE_NORMAL;
    gpio_init.type  = MD_GPIO_TYPE_CMOS;
    gpio_init.func  = MD_GPIO_FUNC_1;
    md_gpio_init(LED_PORT, LED_PIN, &gpio_init);
    
    md_gpio_write_pin(LED_PORT, LED_PIN, 1);

    return;
}

/**
  * @brief  define the peripheral register clear function.
  * @param  None
  * @retval None
  */
static void sfr_reset(void)
{
    SYSCFG_UNLOCK();
    md_rmu_enable_gpio_reset();
    md_rmu_enable_usart0_reset();
    SYSCFG_LOCK();
}

/**
  * @brief  define the function used to jump to app program.
  * @param  None
  * @retval None
  */
void fsm_go(uint32_t para)
{
    FunVoidType JumpToApplication = NULL;
    uint32_t m_JumpAddress;
    uint32_t addr;

    __disable_irq();

    if(para == GO_APP)
    {
        addr = APP_ADDR;
    }
    else if(para == GO_BOOT) 
    {
        addr = BOOT_ADDR;
        WRITE_REG(MSC->FLASHKEY, 0x8ACE0246);
        WRITE_REG(MSC->FLASHKEY, 0x9BDF1357);
        IAPROM_PAGE_ERASE(CRC_CAL_PAGE_ADDR, ~CRC_CAL_PAGE_ADDR, 0);
        WRITE_REG(MSC->FLASHKEY, 0);
        WRITE_REG(MSC->FLASHKEY, 0);
    }
    /* reset registers of peripherals */
    sfr_reset();

    /* disable all peripherals' clock */
    SYSCFG_UNLOCK();
    md_cmu_disable_perh_all();
    SYSCFG_LOCK();

    /* disable all peripherals which may cause an interrupt,
    and clear all possible undisposed interrupt flag */ 
    NVIC->ICER[0] = 0xFFFFFFFF;
    NVIC->ICPR[0] = 0xFFFFFFFF;

    /* disable systick and clear the pending bit */
    SysTick->CTRL = 0;
    SCB->ICSR |= SCB_ICSR_PENDSTCLR_Msk;

    /* set start adress to app/boot flash*/
    if(para == GO_APP)
        MD_BOOT_FROM_FLASH();
    else if(para == GO_BOOT)
        MD_BOOT_FROM_BOOT_FLASH();

    /* interrupt vector remap */
    SYSCFG_UNLOCK();
    SYSCFG->MEMRMP = 0x10000;
    SYSCFG->VTOR = addr;
    SYSCFG_LOCK();

    __enable_irq();

    m_JumpAddress = *(volatile uint32_t *)((addr & 0xFFFFFF00) + 4);
    JumpToApplication = (FunVoidType) m_JumpAddress;
    
    /* init stack top */
    __set_MSP(*(volatile uint32_t *)(addr & 0xFFFFFF00));
    /* jump to app/boot flash */
    JumpToApplication();
}

/**
  * @brief:  Main program.
  * @param:  None
  * @retval: None
  */
int main(void)
{
    /* Configure system clock */
    md_cmu_clock_config(MD_CMU_CLOCK_HRC, 32000000);
    /* Initialize SysTick Interrupt */
    md_init_1ms_tick();

    /* Enable ALL peripheral */
    SYSCFG_UNLOCK();
    md_cmu_enable_perh_all();
    SYSCFG_LOCK();

    md_mcu_irq_config(USART0_IRQn, 0, ENABLE);

    led_init();

    /* Init USART */
    usart_pin_init();
    usart_init();

    while (1)
    {
        if(g_flag)
        {
            g_flag = 0;

            if(g_rx_buf[1] == 0xA1 && g_rx_buf[2] == 0x5E && g_rx_buf[3] == 0xFF
            && g_rx_buf[4] == 0xFF && g_rx_buf[5] == 0xFF && g_rx_buf[6] == 0xFD
            && g_rx_buf[7] == 0xFA)
            {
                md_usart_send_data8(USART0, POS_ACK);
                while(!md_usart_is_active_flag_txc(USART0));
                fsm_go(GO_BOOT);
            }
            else
            {
                md_usart_send_data8(USART0, NEG_ACK);
                while(!md_usart_is_active_flag_txc(USART0));
            }
        }

        md_gpio_toggle_pin_output(LED_PORT, LED_PIN);
        md_delay_1ms(250);
    }
}

/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
