/**********************************************************************************
 *
 * @file    main.c
 * @brief   Main file for DEMO
 *
 * @date    25 Oct 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          25 Oct 2021     biyq           the first version
 *          15 Mar  2023    shicc           version:1.0.1

 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */


/* Includes ----------------------------------------------------------------- */

#include <string.h>
#include "main.h"


/* Public Variables ---------------------------------------------------------- */

uint32_t md_system_clock  = 32000000U;
uint32_t  len_hex;

/* Public Function ---------------------------------------------------------- */
void SystemInit()
{}

/**
  * @brief  Get APB clock.
  *         function prototype can be referred in md_usart.c.
  * @retval The value of APB clock.
  */
uint32_t md_cmu_get_pclk_clock(void)
{
    uint32_t sys_div  = READ_BITS(CMU->CFGR, CMU_CFGR_SYSDIV_MSK, CMU_CFGR_SYSDIV_POSS);
    uint32_t apb_div = READ_BITS(CMU->CFGR, CMU_CFGR_PCLKDIV_MSK, CMU_CFGR_PCLKDIV_POSS);

    return (md_system_clock >> sys_div) >> apb_div;
}

/**
  * @brief  get the clock frequency of system
  *         function prototype can be referred in md_cmu.h.
  * @param  None
  * @retval None
  */
uint32_t md_cmu_get_clock(void)
{
    return md_system_clock;
}


/* Private Macros ------------------------------------------------------------ */


/* Private Variables--------------------------------------------------------- */

static md_usart_init_t s_init;

/* Private Constants --------------------------------------------------------- */


/* Private function prototypes ----------------------------------------------- */


/* Private Function ---------------------------------------------------------- */
/**
  * @brief  init pins used fot usart and boot.
  * @param  None
  * @retval None
  */
void usart0_pins_init(void)
{
    md_gpio_init_t gpio_config;

    /* Initialize tx pin */
    gpio_config.mode  = MD_GPIO_MODE_OUTPUT;
    gpio_config.odos  = MD_GPIO_PUSH_PULL;
    gpio_config.pupd  = MD_GPIO_PUSH_UP;
    gpio_config.odrv = MD_GPIO_OUT_DRIVE_NORMAL;
    gpio_config.flt   = MD_GPIO_FILTER_DISABLE;
    gpio_config.type  = MD_GPIO_TYPE_CMOS;
    gpio_config.func  = MD_GPIO_FUNC_3;
    md_gpio_init(GPIOA, MD_GPIO_PIN_13, &gpio_config);

    /* Initialize rx pin */
    gpio_config.mode  = MD_GPIO_MODE_INPUT;
    gpio_config.odos  = MD_GPIO_PUSH_PULL;
    gpio_config.pupd  = MD_GPIO_PUSH_UP;
    gpio_config.odrv = MD_GPIO_OUT_DRIVE_NORMAL;
    gpio_config.flt   = MD_GPIO_FILTER_DISABLE;
    gpio_config.type  = MD_GPIO_TYPE_CMOS;
    gpio_config.func  = MD_GPIO_FUNC_3;
    md_gpio_init(GPIOA, MD_GPIO_PIN_14, &gpio_config);

    return;
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
    /* Configure system clock: use default internal HRC */
    /* Initialize Systick to tick per 1ms */
    NVIC_SetPriority(SysTick_IRQn, 15);
    SysTick_Config(md_cmu_get_clock() / 1000);
    /* Enable ALL peripheral */
    SYSCFG_UNLOCK();
    md_cmu_enable_perh_all();
    SYSCFG_LOCK();

    /* Init communication parameters */
    usart0_pins_init();
    md_usart_init_struct(&s_init);
    s_init.word_length = MD_USART_WORD_LENGTH_9B;
    s_init.parity = MD_USART_PARITY_EVEN;
    md_usart_init(COMMUNICATION_TYPE, &s_init);

    /* Config usart0 interrupt */
    NVIC_SetPriority(USART0_IRQn, 0x01);
    NVIC_EnableIRQ(USART0_IRQn);

    /* Init state machine */
    fsm_comm_func_init();
    fsm_exec_func_init();
    fsm_para_init();
    
#ifdef __USE_CRC32
    /* Determine whether the length is valid */
    len_hex = *((uint32_t *)CRC_CAL_ADDR + 1);
    if (len_hex <= CRC_CAL_ADDR - APP_ADDR)
    {
        /* If the verification is valid, jump to app */
        if (verify_valid() == PASS)
        {
            g_isp_data.p_go(GO_APP);
        }
    }
#else
    g_isp_data.p_go(GO_APP);
#endif

    while (1)
    {
        while (fsm_is_trigger())
        { 
            proc_fsm();
        }
    }
}

