/**********************************************************************************
 *
 * @file    exec_proc.c
 * @brief   define the exection functions for state machine
 *
 * @date    25 Oct 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          25 Oct 2021     biyq            the first version
 *          15 Mar  2023    shicc           version:1.0.1
 
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ----------------------------------------------------------------- */

#include "exec_proc.h"
#include "main.h"

/* Private Macros ------------------------------------------------------------ */
typedef  void (*FunVoidType)(void);

#if (APP_ADDR & 0x03)
    #error " app address not aligned"
#endif/*  */

/* Public Variables ---------------------------------------------------------- */


/* Private Constants --------------------------------------------------------- */


/* Private function prototypes ----------------------------------------------- */


/* Private function prototypes ----------------------------------------------- */


/* Public function prototypes ----------------------------------------------- */


/* Private Variables--------------------------------------------------------- */


/* Public Function ---------------------------------------------------------- */


/* Private Function ---------------------------------------------------------- */

/**
  * @brief  define the peripheral register clear function.
  * @param  None
  * @retval None
  */
static void sfr_reset(void)
{
    SYSCFG_UNLOCK();
    md_rmu_enable_gpio_reset();
    md_rmu_enable_usart0_reset();
    SYSCFG_LOCK();
}

/**
  * @brief  define the page erase function.
  * @param  None
  * @retval None
  */
static fsm_iap_result_t fsm_page_erase(uint32_t PageCode)
{

    volatile fsm_iap_result_t status = IAP_SUCCESS;

    /* size of each flash page is 512byte */
    uint32_t address = (PageCode << 9);

    if (address & 0x3)
        return IAP_FAIL;

    __disable_irq();

    SYSCFG_UNLOCK();
    /* set to app flash*/
    CLEAR_BIT(SYSCFG->MEMRMP, SYSCFG_MEMRMP_BFRMPEN_MSK);
    SYSCFG_LOCK();

    FLASH_REG_UNLOCK();

    if (IAP_SUCCESS != IAPROM_PAGE_ERASE(address, ~address, 0))
        status = IAP_FAIL;

    FLASH_REG_LOCK();

    SYSCFG_UNLOCK();
    /* set to boot flash*/
    SET_BIT(SYSCFG->MEMRMP, SYSCFG_MEMRMP_BFRMPEN_MSK);
    SYSCFG_LOCK();

    __enable_irq();

    return status;
}

/**
  * @brief  define the words program function.
  * @param  None
  * @retval None
  */
static fsm_iap_result_t fsm_words_program(uint32_t address, uint32_t data_address, uint32_t len)
{
    volatile fsm_iap_result_t status = IAP_SUCCESS;

    if (address & 0x3)
        return IAP_FAIL;

    __disable_irq();

    SYSCFG_UNLOCK();
    /* set to app flash*/
    CLEAR_BIT(SYSCFG->MEMRMP, SYSCFG_MEMRMP_BFRMPEN_MSK);
    SYSCFG_LOCK();

    if ((address & 0x1ff) == 0)
    {
        FLASH_REG_UNLOCK();
        IAPROM_PAGE_ERASE(address, ~address, 0);
    }

    FLASH_REG_UNLOCK();

    if (IAP_SUCCESS != IAPROM_WORDS_PROGRAM(address, ~address, data_address, len, AUTO_ERASE_FALSE))
        status = IAP_FAIL;

    FLASH_REG_LOCK();

    SYSCFG_UNLOCK();
    /* set to boot flash*/
    SET_BIT(SYSCFG->MEMRMP, SYSCFG_MEMRMP_BFRMPEN_MSK);
    SYSCFG_LOCK();

    __enable_irq();

    return status;
}

/**
  * @brief  define the checksum function.
  * @param  None
  * @retval None
  */
static uint32_t fsm_get_crc32(uint32_t *data_ptr, uint32_t len)
{
    /* Not Supported */
    return FAIL;
}

/**
  * @brief  define the check empty function.
  * @param  None
  * @retval None
  */
static fsm_result_t fsm_check_empty(uint32_t *data_ptr, uint32_t len)
{
    uint32_t  i;

    if ((((uint32_t)data_ptr) % 4 != 0) || (len % 4 != 0))
    {
        return FAIL;
    }

    __disable_irq();

    SYSCFG_UNLOCK();
    /* set to app flash*/
    CLEAR_BIT(SYSCFG->MEMRMP, SYSCFG_MEMRMP_BFRMPEN_MSK);
    SYSCFG_LOCK();

    for (i = 0; i < len; i += 4)
    {
        if (*(data_ptr++) != 0xFFFFFFFF)
        {
            break;
        }
    }

    SYSCFG_UNLOCK();
    /* set to boot flash*/
    SET_BIT(SYSCFG->MEMRMP, SYSCFG_MEMRMP_BFRMPEN_MSK);
    SYSCFG_LOCK();

    __enable_irq();

    if (i < len)
        return FAIL;
    else
        return PASS;
}
/**
  * @brief  define the memory read function.
  * @param  None
  * @retval None
  */
static void fsm_read_memo(uint8_t *dest, void *src, uint32_t len)
{
    memcpy(dest, (const void *)src, len);

    return;
}

#ifdef  __USE_CRC32
/**
  * @brief  init crc32.
  * @param  None
  * @retval None
  */
static uint8_t init_crc32(void)
{ 
    md_crc_init_t crc_init;
    
    /* Clear crc_handle_t structure */
    memset(&crc_init, 0x0, sizeof(crc_init));
    /* Initialize CRC */
    md_crc_init_struct(&crc_init); 
    crc_init.mode = MD_CRC_MODE_32;
    crc_init.len = MD_CRC_DATASIZE_32;
    crc_init.order = MD_CRC_BYTORD_LOW;
    crc_init.seed = 0xFFFFFFFF;
    crc_init.chs_inv = ENABLE;
    crc_init.chs_rev = ENABLE;
    crc_init.data_inv = DISABLE;
    crc_init.data_rev = ENABLE;
    md_crc_init(&crc_init);
    
    return 0;
}

/**
  * @brief  check specified checksum function.
  * @param  None
  * @retval None
  */
uint8_t verify_valid(void)
{
    uint32_t len_hex, crc_cal_result, crc_hex_result;
    uint32_t i;
    uint32_t *data_ptr = (uint32_t *)APP_ADDR;
    
    if (!md_crc_is_enabled(CRC))
        init_crc32();
    
    if(CRC_CAL_ADDR & 0x03)
        return FAIL;
    
    /* get info from specified address */
    crc_hex_result = *((uint32_t *)CRC_CAL_ADDR); 
    len_hex = *((uint32_t *)CRC_CAL_ADDR + 1);
    
    if (len_hex > 0x7000 || len_hex & 0x03 || !len_hex)
        return FAIL;

    __disable_irq();
    /* set to app flash*/
    MD_BOOT_FROM_FLASH();

    for (i = 0; i < len_hex / 4; i++)
        md_crc_write_data(CRC, *data_ptr++);
        crc_cal_result = md_crc_get_check_result(CRC);

    /* set to boot flash*/
    MD_BOOT_FROM_BOOT_FLASH();
    __enable_irq();

    if (crc_hex_result == crc_cal_result)
    {
        return PASS;
    }
    else
    {
        MD_CRC_RESET();
        MD_CRC_DISABLE();
        return FAIL;
    }
} 
#endif

/**
  * @brief  define the function used to jump to app program.
  * @param  None
  * @retval None
  */
static void fsm_go(uint32_t para)
{
    FunVoidType JumpToApplication = NULL;
    uint32_t m_JumpAddress;
    uint32_t addr;
    
    __disable_irq();
    
    if(para == GO_APP)
    {
        addr = APP_ADDR;
    } 
    else if(para == GO_BOOT) 
    {
        addr = BOOT_ADDR;
    }
    /* reset registers of peripherals */
    sfr_reset();

    /* disable all peripherals' clock */
    SYSCFG_UNLOCK();
    md_cmu_disable_perh_all();
    SYSCFG_LOCK();
    /* disable all peripherals which may cause an interrupt,
    and clear all possible undisposed interrupt flag */
    NVIC->ICER[0] = 0xFFFFFFFF;
    NVIC->ICPR[0] = 0xFFFFFFFF;

    /* disable systick and clear the pending bit */
    SysTick->CTRL = 0;
    SCB->ICSR |= SCB_ICSR_PENDSTCLR_Msk;

    /* set start adress to app/boot flash*/
    if(para == GO_APP)
    {
        MD_BOOT_FROM_FLASH();
    }
    else if(para == GO_BOOT)
    {
        MD_BOOT_FROM_BOOT_FLASH();
    }
    /* interrupt vector remap */
    SYSCFG_UNLOCK();
    SYSCFG->MEMRMP = 0x10000;
    SYSCFG->VTOR = addr;
    SYSCFG_LOCK();

    __enable_irq();

    m_JumpAddress = *(volatile uint32_t *)((addr & 0xFFFFFF00) + 4);
    JumpToApplication = (FunVoidType) m_JumpAddress;
    
    /* init stack top */
    __set_MSP(*(volatile uint32_t *)(addr & 0xFFFFFF00));
    /* jump to app/boot flash */
    JumpToApplication();

}

/**
  * @brief  assign function pointer to all the state machine subfunction.
  * @param  None
  * @retval None
  */
void fsm_exec_func_init(void)
{
    g_isp_data.p_page_erase   = &fsm_page_erase;
    g_isp_data.p_words_progrm = &fsm_words_program;
    g_isp_data.p_get_crc32    = &fsm_get_crc32;
    g_isp_data.p_check_empty  = &fsm_check_empty;
    g_isp_data.p_go           = &fsm_go;
    g_isp_data.p_read_memo    = &fsm_read_memo;
}

