/**********************************************************************************
 *
 * @file    exec_proc.c
 * @brief   define the exection functions for state machine
 *
 * @date    25 Oct 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          25 Oct 2021     biyq            the first version
 *          15 Mar  2023    shicc           version:1.0.1
 
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */


/* Includes ----------------------------------------------------------------- */

#include "comm_proc.h"


/* Private Macros ------------------------------------------------------------ */


/* Public Variables ---------------------------------------------------------- */


/* Private Constants --------------------------------------------------------- */


/* Private function prototypes ----------------------------------------------- */


/* Private Variables--------------------------------------------------------- */

/**
  *@brief enum definition of state machine procedure
  */
typedef struct comm_s
{
    uint8_t      dummydata;
    uint8_t      *comm_tx_buf;
    uint8_t      *comm_rx_buf;
    uint32_t     comm_tx_cnt;
    uint32_t     comm_rx_cnt;
    uint32_t     comm_tx_size;
    uint32_t     comm_rx_size;
} comm_t;

static comm_t comm_handler;


/* Private Function ---------------------------------------------------------- */

/**
  * @brief  this function is designed to judge whether date the state machine wanna send has been read out by upper
            this function will always return 0, due to the fact, this function only be called via tbc interrupt when
            all expected data sent out in interrupt program
  * @param  None
  * @retval 0x00 to indicate all data expected to send has been read out by upper,
  *         0x01 to indicate process of send is in progress
  */
static fsm_result_t fsm_is_send_over(void)
{
    return PASS;
}

/**
  * @brief  this function is designed to judge whether date the state machine wanna receive has been completed
            this function will always return 0, due to the fact, this function only be called when all expected data
            received in interrupt program
  * @param  None
  * @retval 0x00 to indicate all data expected to receive has been completed,
  *         0x01 to indicate process of receive is in progress
  */
static fsm_result_t fsm_is_recv_over(void)
{
    return PASS;
}

/**
  * @brief  define the send function.
  * @param  None
  * @retval None
  */
static void fsm_send_data(uint8_t *tx_buf, uint32_t len)
{
    if (comm_handler.comm_tx_buf  != tx_buf)
        comm_handler.comm_tx_buf  = tx_buf;

    comm_handler.comm_tx_size = len;
    comm_handler.comm_tx_cnt = 0;

    md_usart_clear_flag_txc(COMMUNICATION_TYPE);

    /* enable commnication to send only */
    md_usart_enable_it_txc(COMMUNICATION_TYPE);
    md_usart_send_data8(COMMUNICATION_TYPE, *(comm_handler.comm_tx_buf + comm_handler.comm_tx_cnt));
    comm_handler.comm_tx_cnt++;

    /*   */
    return;
}

/**
  * @brief  define the receive function.
  * @param  None
  * @retval None
  */
static void fsm_recv_data(uint8_t *rx_buf, uint32_t len)
{
    if (comm_handler.comm_rx_buf != rx_buf)
        comm_handler.comm_rx_buf = rx_buf;

    comm_handler.comm_rx_size  = len;
    comm_handler.comm_rx_cnt = 0;

    /* enable communication to receive only */
    md_usart_enable_it_rxne(COMMUNICATION_TYPE);

    /*   */
    return;
}


/* Public Function ---------------------------------------------------------- */

/**
  * @brief  assign function pointer related to communication to all the state machine subfunction.
  * @param  None
  * @retval None
  */
void fsm_comm_func_init(void)
{
    g_isp_data.p_send_func    = &fsm_send_data;
    g_isp_data.p_recv_func    = &fsm_recv_data;
    g_isp_data.p_is_send_over = &fsm_is_send_over;
    g_isp_data.p_is_recv_over = &fsm_is_recv_over;
}

/**
  * @brief  function to handle communication interrupt.
  * @param  None
  * @retval None
  */
void fsm_int_exec(void)
{
    /* byte send complete */
    if (md_usart_is_enabled_it_txc(COMMUNICATION_TYPE) && md_usart_is_active_flag_txc(COMMUNICATION_TYPE))
    {
        md_usart_clear_flag_txc(COMMUNICATION_TYPE);

        /* if all expected bytes has been sent out then close the interrupt and call fsm_send_out to process the rest */
        if (comm_handler.comm_tx_size == comm_handler.comm_tx_cnt)
        {
            md_usart_disable_it_txc(COMMUNICATION_TYPE);

            fsm_send_over();
        }

        /* if all expected bytes has not been sent out then continue to receive */
        if (comm_handler.comm_tx_size != comm_handler.comm_tx_cnt)
        {
            while (!md_usart_is_active_flag_txemp(COMMUNICATION_TYPE));

            md_usart_send_data8(COMMUNICATION_TYPE, *(comm_handler.comm_tx_buf + comm_handler.comm_tx_cnt));
            comm_handler.comm_tx_cnt++;

            /* allign timer to default when MCU receive each byte */
            g_isp_data.u_frame_timer = FRAME_INTERVAL;
        }
        else
        {
            md_usart_disable_it_txc(COMMUNICATION_TYPE);

            fsm_send_over();
        }
    }

    /* byte receive */
    if (md_usart_is_enabled_it_rxne(COMMUNICATION_TYPE) && md_usart_is_active_flag_rxne(COMMUNICATION_TYPE))
    {
        md_usart_clear_flag_rxne(COMMUNICATION_TYPE);

        *(comm_handler.comm_rx_buf + comm_handler.comm_rx_cnt) = (uint8_t)md_usart_recv_data8(COMMUNICATION_TYPE);
        comm_handler.comm_rx_cnt++;

        /* receive over */
        if (comm_handler.comm_rx_size == comm_handler.comm_rx_cnt)
        {
            md_usart_disable_it_rxne(COMMUNICATION_TYPE);

            fsm_recv_over();
        }

        /* allign timer to default when MCU receive each byte */
        g_isp_data.u_frame_timer = FRAME_INTERVAL;
    }
}

/**
  * @brief  function to handle systick interrupt.
  * @param  None
  * @retval None
  */
void fsm_systick_int_exec(void)
{
    /* the time interval of received adjacent bytes in the same frame is more than expected, reset tx/rx FIFO */
    if (g_isp_data.u_frame_timer != 0)
    {
        g_isp_data.u_frame_timer--;

        if (g_isp_data.u_frame_timer == 0)
        {
            /*  */
            fsm_para_init();
            return;
        }
    }
}
