/**********************************************************************************
 *
 * @file    main.c
 * @brief   Main file for DEMO
 *
 * @date    27 Oct 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          27 Oct 2021     Biyq            the first version
 *          22 Feb  2023    shicc           version:1.0.1
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */


/* Includes ----------------------------------------------------------------- */

#include <string.h>
#include "main.h"
#include "iap_rom.h"

/* Public Variables ---------------------------------------------------------- */

uint32_t md_system_clock  = 32000000U;


/* Public Function ---------------------------------------------------------- */
void SystemInit()
{}

/**
  * @brief  Get APB clock.
  *         function prototype can be referred in md_usart.c.
  * @retval The value of APB clock.
  */
uint32_t md_cmu_get_pclk_clock(void)
{
    uint32_t sys_div  = READ_BITS(CMU->CFGR, CMU_CFGR_SYSDIV_MSK, CMU_CFGR_SYSDIV_POSS);
    uint32_t apb_div = READ_BITS(CMU->CFGR, CMU_CFGR_PCLKDIV_MSK, CMU_CFGR_PCLKDIV_POSS);

    return (md_system_clock >> sys_div) >> apb_div;
}

/**
  * @brief  get the clock frequency of system
  *         function prototype can be referred in md_cmu.h.
  * @param  None
  * @retval None
  */
uint32_t md_cmu_get_clock(void)
{
    return md_system_clock;
}


/* Private Macros ------------------------------------------------------------ */


/* Private Variables--------------------------------------------------------- */

static md_spi_init_t s_init;


/* Private Constants --------------------------------------------------------- */


/* Private function prototypes ----------------------------------------------- */
typedef  void (*FunVoidType)(void);

/* Private Function ---------------------------------------------------------- */
/**
  * @brief  define the peripheral register clear function.
  * @param  None
  * @retval None
  */
static void sfr_reset(void)
{
    SYSCFG_UNLOCK();
    md_rmu_enable_gpio_reset();
    md_rmu_enable_usart0_reset();
    SYSCFG_LOCK();
}

/**
  * @brief  define the function used to jump to app program.
  * @param  None
  * @retval None
  */
static void fsm_go(uint32_t para)
{
    FunVoidType JumpToApplication = NULL;
    uint32_t m_JumpAddress;
    uint32_t addr;
    
    __disable_irq();
    
    if(para == GO_APP)
    {
        addr = APP_ADDR;
    } 
    else if(para == GO_BOOT) 
    {
        addr = BOOT_ADDR;
        WRITE_REG(MSC->FLASHKEY, 0x8ACE0246);
        WRITE_REG(MSC->FLASHKEY, 0x9BDF1357);
        IAPROM_PAGE_ERASE(CRC_CAL_PAGE_ADDR, ~CRC_CAL_PAGE_ADDR, 0);
        WRITE_REG(MSC->FLASHKEY, 0);
        WRITE_REG(MSC->FLASHKEY, 0);
    }
    /* reset registers of peripherals */
    sfr_reset();

    /* disable all peripherals' clock */
    SYSCFG_UNLOCK();
    md_cmu_disable_perh_all();
    SYSCFG_LOCK();
    /* disable all peripherals which may cause an interrupt,
    and clear all possible undisposed interrupt flag */
    NVIC->ICER[0] = 0xFFFFFFFF;
    NVIC->ICPR[0] = 0xFFFFFFFF;

    /* disable systick and clear the pending bit */
    SysTick->CTRL = 0;
    SCB->ICSR |= SCB_ICSR_PENDSTCLR_Msk;

    /* set start adress to app/boot flash*/
    if(para == GO_APP)
        MD_BOOT_FROM_FLASH();
    else if(para == GO_BOOT)
        MD_BOOT_FROM_BOOT_FLASH();

    /* interrupt vector remap */
    SYSCFG_UNLOCK();
    SYSCFG->MEMRMP = 0x10000;
    SYSCFG->VTOR = addr;
    SYSCFG_LOCK();

    __enable_irq();

    m_JumpAddress = *(volatile uint32_t *)((addr & 0xFFFFFF00) + 4);
    JumpToApplication = (FunVoidType) m_JumpAddress;
    
    /* init stack top */
    __set_MSP(*(volatile uint32_t *)(addr & 0xFFFFFF00));
    /* jump to app/boot flash */
    JumpToApplication();
}

/**
  * @brief:  Init gpio for led1.
  * @param:  None
  * @retval: None
  */
void gpio_init(void)
{
    md_gpio_init_t gpio_init;

    /* Initialize LED pin */
    md_gpio_init_struct(&gpio_init);      /* initialize the gpio_init */
    gpio_init.mode  = MD_GPIO_MODE_OUTPUT;
    gpio_init.odos  = MD_GPIO_PUSH_PULL;
    gpio_init.pupd  = MD_GPIO_FLOATING;
    gpio_init.odrv  = MD_GPIO_OUT_DRIVE_NORMAL;
    gpio_init.type  = MD_GPIO_TYPE_CMOS;
    gpio_init.func  = MD_GPIO_FUNC_1;
    md_gpio_init(LED_PORT, LED_PIN, &gpio_init);
    
    md_gpio_write_pin(LED_PORT, LED_PIN, 1);

    return;
}
/**
  * @brief  init pins used fot spi communication.
  * @param  None
  * @retval None
  */
void spi0_pins_init(void)
{
    md_gpio_init_t gpio_config;

    /*From machine initialization, SCK pin configuration for the input mode*/
    gpio_config.mode = MD_GPIO_MODE_INPUT;
    gpio_config.odos = MD_GPIO_PUSH_PULL;
    gpio_config.pupd = MD_GPIO_PUSH_UP;
    gpio_config.odrv = MD_GPIO_OUT_DRIVE_NORMAL;
    gpio_config.flt  = MD_GPIO_FILTER_DISABLE;
    gpio_config.type = MD_GPIO_TYPE_CMOS;
    gpio_config.func = MD_GPIO_FUNC_2;
    md_gpio_init(SPI_PORT, SPI_SCK_PIN, &gpio_config);

    /*From machine initialization MISO pins, configured to output mode*/
    gpio_config.mode = MD_GPIO_MODE_OUTPUT;
    gpio_config.odos = MD_GPIO_PUSH_PULL;
    gpio_config.pupd = MD_GPIO_PUSH_UP;
    gpio_config.odrv = MD_GPIO_OUT_DRIVE_NORMAL;
    gpio_config.flt  = MD_GPIO_FILTER_DISABLE;
    gpio_config.type = MD_GPIO_TYPE_CMOS;
    gpio_config.func = MD_GPIO_FUNC_2;
    md_gpio_init(SPI_PORT, SPI_MISO_PIN, &gpio_config);

    /*From machine initialization, MOSI pin configured as input mode*/
    gpio_config.mode = MD_GPIO_MODE_INPUT;
    gpio_config.odos = MD_GPIO_PUSH_PULL;
    gpio_config.pupd = MD_GPIO_PUSH_UP;
    gpio_config.odrv = MD_GPIO_OUT_DRIVE_NORMAL;
    gpio_config.flt  = MD_GPIO_FILTER_DISABLE;
    gpio_config.type = MD_GPIO_TYPE_CMOS;
    gpio_config.func = MD_GPIO_FUNC_2;
    md_gpio_init(SPI_PORT, SPI_MOSI_PIN, &gpio_config);

    /*From machine initialization, NSS pin configured as input mode*/
    gpio_config.mode = MD_GPIO_MODE_INPUT;
    gpio_config.odos = MD_GPIO_PUSH_PULL;
    gpio_config.pupd = MD_GPIO_PUSH_UP;
    gpio_config.odrv = MD_GPIO_OUT_DRIVE_NORMAL;
    gpio_config.flt  = MD_GPIO_FILTER_DISABLE;
    gpio_config.type = MD_GPIO_TYPE_CMOS;
    gpio_config.func = MD_GPIO_FUNC_2;
    md_gpio_init(SPI_PORT, SPI_NSS_PIN, &gpio_config);

    return;
}


/* Public Variables ---------------------------------------------------------- */

/**
  * @brief  Test main function
  * @retval Status.
  */
int main(void)
{
    /* Configure system clock: use default internal HRC */
    /* Initialize Systick to tick per 1ms */
    NVIC_SetPriority(SysTick_IRQn, 15);
    SysTick_Config(md_cmu_get_clock() / 1000);
    /* Enable ALL peripheral */
    SYSCFG_UNLOCK();
    md_cmu_enable_perh_all();
    SYSCFG_LOCK();
    
    /* init gpio for led1 */
    gpio_init();

    /* Init communication parameters */
    spi0_pins_init();
    /* master mode, clock high in free state, sample in first edge and disable sw manage slaver */
    md_spi_struct_init(&s_init);
    /* change to slaver mode, receive only in double lines simplex transfer mode */
    s_init.mode = MD_SPI_MODE_SLAVER;
    md_spi_init(SPI0, &s_init);

    /* Config uart0 interrupt */
    NVIC_SetPriority(SPI0_IRQn, 0);
    NVIC_EnableIRQ(SPI0_IRQn);

    /*Make to receive the air break*/
    md_spi_enable_rxne_interrupt(SPI0);

    while (1)
    {
        if(g_flag)
        {
            g_flag = 0;

            fsm_go(GO_BOOT);
        }

        md_gpio_toggle_pin_output(LED_PORT, LED_PIN);
        md_delay_1ms(250);
    }
}
