/**********************************************************************************
 *
 * @file    main.c
 * @brief   Main file for DEMO
 *
 * @date    27 Oct 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          27 Oct 2021     Biyq            the first version
 *          22 Feb  2023    shicc           version:1.0.1
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */


/* Includes ----------------------------------------------------------------- */

#include <string.h>
#include "main.h"


/* Public Variables ---------------------------------------------------------- */

uint32_t md_system_clock  = 32000000U;
uint32_t  len_hex;

/* Public Function ---------------------------------------------------------- */
void SystemInit()
{}

/**
  * @brief  Get APB clock.
  *         function prototype can be referred in md_usart.c.
  * @retval The value of APB clock.
  */
uint32_t md_cmu_get_pclk_clock(void)
{
    uint32_t sys_div  = READ_BITS(CMU->CFGR, CMU_CFGR_SYSDIV_MSK, CMU_CFGR_SYSDIV_POSS);
    uint32_t apb_div = READ_BITS(CMU->CFGR, CMU_CFGR_PCLKDIV_MSK, CMU_CFGR_PCLKDIV_POSS);

    return (md_system_clock >> sys_div) >> apb_div;
}

/**
  * @brief  get the clock frequency of system
  *         function prototype can be referred in md_cmu.h.
  * @param  None
  * @retval None
  */
uint32_t md_cmu_get_clock(void)
{
    return md_system_clock;
}


/* Private Macros ------------------------------------------------------------ */


/* Private Variables--------------------------------------------------------- */

static md_spi_init_t s_init;


/* Private Constants --------------------------------------------------------- */


/* Private function prototypes ----------------------------------------------- */


/* Private Function ---------------------------------------------------------- */

void spi_miso_pin_init_out_data(void)
{
    SPI_PORT->FUNC1 &= (~(GPIO_FUNC1_FSEL_IO12_MSK));
    SPI_PORT->FUNC1 |= MD_GPIO_FUNC_2 << GPIO_FUNC1_FSEL_IO12_POSS;
}

void spi_miso_pin_init_out_1evel(void)
{
    SPI_PORT->FUNC1 &= (~(GPIO_FUNC1_FSEL_IO12_MSK));
    SPI_PORT->FUNC1 |= MD_GPIO_FUNC_1 << GPIO_FUNC1_FSEL_IO12_POSS;
}

/**
  * @brief  init pins used fot spi communication.
  * @param  None
  * @retval None
  */
void spi0_pins_init(void)
{
    md_gpio_init_t gpio_config;

    /*From machine initialization, SCK pin configuration for the input mode*/
    gpio_config.mode = MD_GPIO_MODE_INPUT;
    gpio_config.odos = MD_GPIO_PUSH_PULL;
    gpio_config.pupd = MD_GPIO_PUSH_UP;
    gpio_config.odrv = MD_GPIO_OUT_DRIVE_NORMAL;
    gpio_config.flt  = MD_GPIO_FILTER_DISABLE;
    gpio_config.type = MD_GPIO_TYPE_CMOS;
    gpio_config.func = MD_GPIO_FUNC_2;
    md_gpio_init(SPI_PORT, SPI_SCK_PIN, &gpio_config);

    /*From machine initialization MISO pins, configured to output mode*/
    gpio_config.mode = MD_GPIO_MODE_OUTPUT;
    gpio_config.odos = MD_GPIO_PUSH_PULL;
    gpio_config.pupd = MD_GPIO_PUSH_UP;
    gpio_config.odrv = MD_GPIO_OUT_DRIVE_NORMAL;
    gpio_config.flt  = MD_GPIO_FILTER_DISABLE;
    gpio_config.type = MD_GPIO_TYPE_CMOS;
    gpio_config.func = MD_GPIO_FUNC_2;
    md_gpio_init(SPI_PORT, SPI_MISO_PIN, &gpio_config);

    /*From machine initialization, MOSI pin configured as input mode*/
    gpio_config.mode = MD_GPIO_MODE_INPUT;
    gpio_config.odos = MD_GPIO_PUSH_PULL;
    gpio_config.pupd = MD_GPIO_PUSH_UP;
    gpio_config.odrv = MD_GPIO_OUT_DRIVE_NORMAL;
    gpio_config.flt  = MD_GPIO_FILTER_DISABLE;
    gpio_config.type = MD_GPIO_TYPE_CMOS;
    gpio_config.func = MD_GPIO_FUNC_2;
    md_gpio_init(SPI_PORT, SPI_MOSI_PIN, &gpio_config);
                                 
    spi_miso_pin_init_out_1evel();
    
    /*From machine initialization, NSS pin configured as input mode*/
    gpio_config.mode = MD_GPIO_MODE_INPUT;
    gpio_config.odos = MD_GPIO_PUSH_PULL;
    gpio_config.pupd = MD_GPIO_PUSH_UP;
    gpio_config.odrv = MD_GPIO_OUT_DRIVE_NORMAL;
    gpio_config.flt  = MD_GPIO_FILTER_DISABLE;
    gpio_config.type = MD_GPIO_TYPE_CMOS;
    gpio_config.func = MD_GPIO_FUNC_2;
    md_gpio_init(SPI_PORT, SPI_NSS_PIN, &gpio_config);

    return;
}


/* Public Variables ---------------------------------------------------------- */

/**
  * @brief  Test main function
  * @retval Status.
  */
int main(void)
{
    /* Configure system clock: use default internal HRC */
    /* Initialize Systick to tick per 1ms */
    NVIC_SetPriority(SysTick_IRQn, 15);
    SysTick_Config(md_cmu_get_clock() / 1000);
    /* Enable ALL peripheral */
    SYSCFG_UNLOCK();
    md_cmu_enable_perh_all();
    SYSCFG_LOCK();

    /* Init communication parameters */
    spi0_pins_init();
    /* master mode, clock high in free state, sample in first edge and disable sw manage slaver */
    md_spi_struct_init(&s_init);
    /* change to slaver mode, receive only in double lines simplex transfer mode */
    s_init.mode = MD_SPI_MODE_SLAVER;
    md_spi_init(SPI0, &s_init);


    /* Config SPI0 interrupt */
    NVIC_SetPriority(SPI0_IRQn, 0);
    NVIC_EnableIRQ(SPI0_IRQn);

    /* Init state machine */
    fsm_comm_func_init();
    fsm_exec_func_init();
    fsm_para_init();

#ifdef  __USE_CRC32
    /* Determine whether the length is valid */
    len_hex = *((uint32_t *)CRC_CAL_ADDR + 1);
    if (len_hex <= CRC_CAL_ADDR - APP_ADDR)
    {
        /* If the verification is valid, jump to app */
        if (verify_valid() == PASS)
        {
            g_isp_data.p_go(GO_APP);
        }
    }
#else
    g_isp_data.p_go(GO_APP);
#endif

    while (1)
    {
        while (fsm_is_trigger())
        {
            proc_fsm();
        }

    }
}

