/**********************************************************************************
 *
 * @file    exec_proc.c
 * @brief   define the exection functions for state machine
 *
 * @date    28 Oct 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          28 Oct 2021     Biyq            the first version
 *          15 Mar  2023    shicc           version:1.0.1

 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */


/* Includes ----------------------------------------------------------------- */

#include "comm_proc.h"


/* Private Macros ------------------------------------------------------------ */


/* Public Variables ---------------------------------------------------------- */


/* Private Constants --------------------------------------------------------- */


/* Private function prototypes ----------------------------------------------- */
__STATIC_INLINE void md_i2c_clear_flag_addr(I2C_TypeDef *i2c)
{
    __IO uint32_t tmpreg;
    tmpreg = i2c->STAT1;
    (void) tmpreg;
    tmpreg = i2c->STAT2;
    (void) tmpreg;
}

__STATIC_INLINE void md_i2c_clear_flag_stop(I2C_TypeDef *i2c)
{
    __IO uint32_t tmpreg;
    tmpreg = i2c->STAT1;
    (void) tmpreg;

    SET_BIT(i2c->CON1, I2C_CON1_PEN_MSK);
}

/* Private Variables--------------------------------------------------------- */

/**
  *@brief enum definition of state machine procedure
  */
typedef struct comm_s
{
    uint8_t      status;
    uint8_t      *comm_tx_buf;
    uint8_t      *comm_rx_buf;
    uint32_t     comm_tx_cnt;
    uint32_t     comm_rx_cnt;
    uint32_t     comm_tx_size;
    uint32_t     comm_rx_size;
} comm_t;

static comm_t comm_handler;


/* Private Function ---------------------------------------------------------- */

/**
  * @brief  this function is designed to judge whether date the state machine wanna send has been read out by upper
  * @param  None
  * @retval 0x00 to indicate all data expected to send has been read out by upper,
  *         0x01 to indicate process of send is in progress
  */
static fsm_result_t fsm_is_send_over(void)
{
    while (md_i2c_is_active_flag_busy(I2C0));

    return PASS;
}

/**
  * @brief  this function is designed to judge whether date the state machine wanna receive has been completed
  * @param  None
  * @retval 0x00 to indicate all data expected to receive has been completed,
  *         0x01 to indicate process of receive is in progress
  */
static fsm_result_t fsm_is_recv_over(void)
{
    return PASS;
}

/**
  * @brief  define the send function.
  * @param  None
  * @retval None
  */
static void fsm_send_data(uint8_t *tx_buf, uint32_t len)
{
    if (tx_buf != comm_handler.comm_tx_buf)
        comm_handler.comm_tx_buf = tx_buf;

    comm_handler.comm_tx_size = len;
    comm_handler.comm_tx_cnt = 0;

    md_i2c_enable_it_err(I2C0);
    md_i2c_enable_it_evt(I2C0);
    md_i2c_enable_it_buf(I2C0);

    /*   */
    return;
}

/**
  * @brief  define the receive function.
  * @param  None
  * @retval None
  */
static void fsm_recv_data(uint8_t *rx_buf, uint32_t len)
{
    if (rx_buf != comm_handler.comm_rx_buf)
        comm_handler.comm_rx_buf = rx_buf;

    comm_handler.comm_rx_size  = len;
    comm_handler.comm_rx_cnt = 0;
    comm_handler.comm_tx_cnt = 0;

    md_i2c_enable_it_evt(I2C0);
    md_i2c_enable_it_buf(I2C0);

    /*   */
    return;
}


/* Public Function ---------------------------------------------------------- */

/**
  * @brief  assign function pointer related to communication to all the state machine subfunction.
  * @param  None
  * @retval None
  */
void fsm_comm_func_init(void)
{
    g_isp_data.p_send_func    = &fsm_send_data;
    g_isp_data.p_recv_func    = &fsm_recv_data;
    g_isp_data.p_is_send_over = &fsm_is_send_over;
    g_isp_data.p_is_recv_over = &fsm_is_recv_over;
}

/**
  * @brief  function to handle communication interrupt.
  * @param  None
  * @retval None
  */
void fsm_int_exec(void)
{
    /* byte send */
    if ((md_i2c_is_active_flag_txbe(I2C0)) && md_i2c_is_enable_it_buf(I2C0))
    {

        md_i2c_transmit_data8(I2C0, comm_handler.comm_tx_buf[comm_handler.comm_tx_cnt++]);

        /* assign timer to default when MCU receive each byte */
        g_isp_data.u_frame_timer = FRAME_INTERVAL;

        /* if all expected bytes has been put in tx_buf then close the interrupt */
        if (comm_handler.comm_tx_size == comm_handler.comm_tx_cnt)
        {
            md_i2c_disable_it_buf(I2C0);

            /* fsm process */
            fsm_send_over();
        }
    }

    /* byte receive */
    if ((md_i2c_is_active_flag_rxne(I2C0)) && md_i2c_is_enable_it_buf(I2C0))
    {
        comm_handler.comm_rx_buf[comm_handler.comm_rx_cnt++] = (uint8_t)md_i2c_receive_data8(I2C0);

        /* assign timer to default when MCU receive each byte */
        g_isp_data.u_frame_timer = FRAME_INTERVAL;

        /* disable rx interrupt and call fsm_recv_over to process the rest */
        if (comm_handler.comm_rx_cnt == comm_handler.comm_rx_size)
        {
            md_i2c_disable_it_buf(I2C0);

            /* fsm process */
            fsm_recv_over();
        }
    }

    /* NACK */
    if (md_i2c_is_active_flag_af(I2C0) && md_i2c_is_enable_it_err(I2C0))
    {
        md_i2c_clear_flag_af(I2C0);

        /* if all expected bytes has been sent out then close the interrupt and call fsm_send_out() to process the rest */
        if (comm_handler.comm_tx_size == comm_handler.comm_tx_cnt)
        {
            md_i2c_disable_it_err(I2C0);
        }
    }

    /*From the machine address matching*/
    if (md_i2c_is_active_flag_addr(I2C0) && md_i2c_is_enable_it_evt(I2C0))
    {
        md_i2c_clear_flag_addr(I2C0);
    }

    /* stop bit */
    if (md_i2c_is_active_flag_stop(I2C0) && md_i2c_is_enable_it_evt(I2C0))
    {
        md_i2c_clear_flag_stop(I2C0);

        if ((comm_handler.comm_rx_cnt == comm_handler.comm_rx_size && comm_handler.comm_rx_cnt > 0) ||
                (comm_handler.comm_tx_cnt == comm_handler.comm_tx_size && comm_handler.comm_tx_cnt > 0))
            md_i2c_disable_it_evt(I2C0);
    }
}

/**
  * @brief  function to handle systick interrupt.
  * @param  None
  * @retval None
  */
void fsm_systick_int_exec(void)
{

    /* the time interval of received adjacent bytes in the same frame is more than expected, reset tx/rx FIFO */
    if (g_isp_data.u_frame_timer != 0)
    {
        g_isp_data.u_frame_timer--;

        if (g_isp_data.u_frame_timer == 0)
        {
            fsm_para_init();
            return;
        }
    }
}
