/**********************************************************************************
 *
 * @file    udp-server.c
 * @brief   UDP server demo
 *
 * @date    30 Apri 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          7  Nov  2022    shiwa           contiki 6lowpan client demo
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

#include "contiki.h"
#include "net/routing/routing.h"
#include "net/netstack.h"
#include "net/ipv6/simple-udp.h"
#include "uiplib.h"
#include <stdlib.h>
#include <stdarg.h>
#include "cmd_utils.h"
#include "mac_statics.h"

#include "sys/log.h"
#define LOG_MODULE "App"
#define LOG_LEVEL LOG_LEVEL_INFO

#define WITH_SERVER_REPLY  1
#define UDP_CLIENT_PORT 8765
#define UDP_SERVER_PORT 5678

#define SERVER_NODE 1
static struct simple_udp_connection udp_conn;
const char g_app_name[] = "server";

/* Server configuration */
static int echo_messages = 1;
static uip_ipaddr_t req_target_addr;
static int req_target_to = 0;

/* Response buffer */
#define RESPONSE_TOTAL 1024
static char response[RESPONSE_TOTAL];
static int response_len;
#define RESPONSE_CUR (response+response_len)
#define RESPONSE_CAPACITY (RESPONSE_TOTAL-response_len)

/*Request buffer */
static char req_buf[128];
static int req_len = 0;
static uip_sr_node_t *req_cur = NULL;
static struct ctimer req_timer;
static uint32_t req_send_time = 0;
static int req_recv_timeout = 500;

/* UART utils functions */
void uart_send_raw(const char *s, uint32_t len);
uint32_t uart_gets(char *data, uint32_t len);
uint32_t uart_check(void);

PROCESS(udp_server_process, "UDP server");
AUTOSTART_PROCESSES(&udp_server_process);
/*---------------------------------------------------------------------------*/
void append_response(const char *fmt, ...)
{
    va_list args;
    va_start(args, fmt);
    response_len += vsnprintf(RESPONSE_CUR, RESPONSE_CAPACITY, fmt, args);
    va_end(args);
}
void remove_response(int count)
{
    if (count < response_len)
        response_len -= count;
    else
        response_len = 0;
}
#define APPEND_RESPONSE_BY(func,...) (response_len+=func(RESPONSE_CUR,RESPONSE_CAPACITY ,##__VA_ARGS__))

void routes_print_links(const char *str)
{
    uip_ipaddr_t child_ipaddr;
    uip_ipaddr_t parent_ipaddr;
    if (uip_sr_num_nodes() > 0)
    {
        uip_sr_node_t *link;
        /* Our routing links */
        link = uip_sr_node_head();
        while (link != NULL)
        {
            NETSTACK_ROUTING.get_sr_node_ipaddr(&child_ipaddr, link);
            APPEND_RESPONSE_BY(uiplib_ipaddr_snprint, &child_ipaddr);
            append_response("->");
            if (link->parent)
            {
                NETSTACK_ROUTING.get_sr_node_ipaddr(&parent_ipaddr, link->parent);
                APPEND_RESPONSE_BY(uiplib_ipaddr_snprint, &parent_ipaddr);
            }
            else
            {
                append_response("*");
            }
            if (link->lifetime == UIP_SR_INFINITE_LIFETIME)
            {
                append_response("(infinite)");
            }
            else
            {
                append_response("(%d s)", link->lifetime);
            }
            append_response(";");
            link = uip_sr_node_next(link);
        }
    }
    else
    {
        append_response("NO_LINKS");
    }
}
void print_neighbors()
{
    uip_ds6_nbr_t *n = uip_ds6_nbr_head();
    if (!n)
    {
        append_response("NO_NEIGHBORS");
        return;
    }
    while (n)
    {
        APPEND_RESPONSE_BY(uiplib_ipaddr_snprint, &n->ipaddr);
        append_response(";");
        n = uip_ds6_nbr_next(n);
    }
}
void prepare_request(const uip_ipaddr_t *inaddr);
void request_timeout(void *a)
{
    uip_ipaddr_t child_ipaddr;
    NETSTACK_ROUTING.get_sr_node_ipaddr(&child_ipaddr, (uip_sr_node_t *)a);
    response_len = 0;
    append_response("!R Request to ");
    APPEND_RESPONSE_BY(uiplib_ipaddr_snprint, &child_ipaddr);
    append_response(" timeout");

    response[response_len++] = '\r';
    response[response_len++] = '\n';
#if USE_CHECK_SUM
    calc_chksum((uint8_t *)response, response_len);
    response_len += CHECKSUM_LEN;
#endif
    uart_send_raw(response, response_len);
    if (!req_target_to)
        prepare_request(&child_ipaddr);

}
void refresh_dag(char *args)
{
    if (!strcmp(args, "repair"))
    {
        NETSTACK_ROUTING.global_repair("Refresh");
        append_response("OK");
    }
    else
    {
        append_response("INVALID_PARAM");
    }
}
void prepare_request(const uip_ipaddr_t *inaddr)
{
    uip_ipaddr_t toaddr;
    if (inaddr && req_cur)
    {
        NETSTACK_ROUTING.get_sr_node_ipaddr(&toaddr, req_cur);
        if (!uip_ipaddr_cmp(&toaddr, inaddr))
        {
            return;
        }

        append_response(" in %lums", clock_time() - req_send_time);
        req_send_time = 0;
        ctimer_stop(&req_timer);
        req_cur = uip_sr_node_next(req_cur);
    }

    if (req_cur)
    {
        NETSTACK_ROUTING.get_sr_node_ipaddr(&toaddr, req_cur);
        simple_udp_sendto(&udp_conn, req_buf, req_len, &toaddr);
        req_send_time = clock_time();
        ctimer_set(&req_timer, req_recv_timeout, request_timeout, req_cur);
    }
    else
    {
        req_len = 0;
    }
}
void request_all(const char *cmd)
{
    extern int rpl_dag_root_is_root(void);
    if (req_len != 0)
    {
        append_response("ERROR_BUSY");
        return;
    }
    //prepare command
    req_len = snprintf(req_buf, 127, "?%s\r\n", cmd);
#if USE_CHECK_SUM
    calc_chksum((uint8_t *)req_buf, req_len);
    req_len += CHECKSUM_LEN;
#endif
    if (rpl_dag_root_is_root())
    {
        if (req_target_to)
        {
            //send to one selected client
            append_response("send to ");
            APPEND_RESPONSE_BY(uiplib_ipaddr_snprint, &req_target_addr);
            req_send_time = clock_time();
            ctimer_set(&req_timer, req_recv_timeout, request_timeout, req_cur);
            simple_udp_sendto(&udp_conn, req_buf, req_len, &req_target_addr);
        }
        else if (uip_sr_num_nodes() > 1)
        {
            //send to all
            uip_ipaddr_t child_ipaddr;
            uip_sr_node_t *link;

            link = uip_sr_node_head();
            req_cur = uip_sr_node_next(link);
            append_response("send to %d clients:", uip_sr_num_nodes() - 1);
            while (link != NULL)
            {
                NETSTACK_ROUTING.get_sr_node_ipaddr(&child_ipaddr, link);
                if (link->parent != NULL)
                {
                    APPEND_RESPONSE_BY(uiplib_ipaddr_snprint, &child_ipaddr);
                    append_response(";");
                    //simple_udp_sendto(&udp_conn, req_buf, req_len, &child_ipaddr);
                }
                link = uip_sr_node_next(link);
            }
            prepare_request(NULL);
        }
        else
        {
            req_len = 0;
            append_response("NO_LINKS");
        }
    }
}
void request_to(const char *args)
{
    if (!strcmp(args, "all"))
    {
        append_response("OK");
        req_target_to = 0;
    }
    else if (!strcmp(args, "show"))
    {
        if (req_target_to)
        {
            APPEND_RESPONSE_BY(uiplib_ipaddr_snprint, &req_target_addr);
        }
        else
        {
            append_response("all");
        }
    }
    else
    {
        uip_sr_node_t *link;
        uip_ipaddr_t child_ipaddr;
        uiplib_ip6addrconv(args, &req_target_addr);

        link = uip_sr_node_head();
        while (link != NULL)
        {
            NETSTACK_ROUTING.get_sr_node_ipaddr(&child_ipaddr, link);
            if (link->parent != NULL && uip_ipaddr_cmp(&child_ipaddr, &req_target_addr))
            {
                break;
            }
            link = uip_sr_node_next(link);
        }
        if (link)
        {
            req_target_to = 1;
            append_response("OK");
        }
        else
        {
            req_target_to = 0;
            append_response("NOTFOUND");
        }
    }
}
void print_info(char *args)
{
    append_response("OK ");
    if (!strcmp(args, "CSMA"))
    {
        append_response("CSMA:");
        append_response("INPUT=%d;", csma_statics[CSMA_STAT_INPUT]);
        append_response("ACK_IGNORED=%d;", csma_statics[CSMA_STAT_ACK_IGNORED]);
        append_response("PARSE_FAIL=%d;", csma_statics[CSMA_STAT_PARSE_FAILED]);
        append_response("NOT_FOR_US=%d;", csma_statics[CSMA_STAT_NOT_FOR_US]);
        append_response("FROM_SELF=%d;", csma_statics[CSMA_STAT_FROM_SELF]);
        append_response("DUPLICATE=%d;", csma_statics[CSMA_STAT_DUPLICATE]);
        append_response("OUTPUT=%d;", csma_statics[CSMA_STAT_OUTPUT]);
        append_response("ERROR=%d;", csma_statics[CSMA_STAT_TXERR]);
        append_response("NOACK=%d;", csma_statics[CSMA_STAT_NOACK]);
        append_response("COLLISION=%d;", csma_statics[CSMA_STAT_COLLISION]);
    }
    else if (!strcmp(args, "IP"))
    {
        append_response("IP:");
        append_response("SENT=%d;", uip_stat.ip.sent);
        append_response("RECV=%d;", uip_stat.ip.recv);
        append_response("DROP=%d;", uip_stat.ip.drop);
        append_response("FORWARDED=%d;", uip_stat.ip.forwarded);
        append_response("VHLERR=%d;", uip_stat.ip.vhlerr);
        append_response("HBLENERR=%d;", uip_stat.ip.hblenerr);
        append_response("LBLENERR=%d;", uip_stat.ip.lblenerr);
        append_response("FRAGERR=%d;", uip_stat.ip.fragerr);
        append_response("CHKERR=%d;", uip_stat.ip.chkerr);
        append_response("PROTOERR=%d;", uip_stat.ip.protoerr);
    }
#if UIP_TCP
    else if (!strcmp(args, "IP"))
    {
        append_response("TCP:");
        append_response("SENT=%d;", uip_stat.tcp.sent);
        append_response("RECV=%d;", uip_stat.tcp.recv);
        append_response("DROP=%d;", uip_stat.tcp.drop);
        append_response("CHKERR=%d;", uip_stat.tcp.chkerr);
        append_response("ACKERR=%d;", uip_stat.tcp.ackerr);
        append_response("RST=%d;", uip_stat.tcp.rst);
        append_response("REXMIT=%d;", uip_stat.tcp.rexmit);
        append_response("SYNDROP=%d;", uip_stat.tcp.syndrop);
        append_response("SYNRST=%d;", uip_stat.tcp.synrst);
    }
#endif
#if UIP_UDP
    else if (!strcmp(args, "UDP"))
    {
        append_response("UDP:");
        append_response("SENT=%d;", uip_stat.udp.sent);
        append_response("RECV=%d;", uip_stat.udp.recv);
        append_response("DROP=%d;", uip_stat.udp.drop);
        append_response("CHKERR=%d;", uip_stat.udp.chkerr);
    }
#endif
    else if (!strcmp(args, "ICMP"))
    {
        append_response("ICMP:");
        append_response("SENT=%d;", uip_stat.icmp.sent);
        append_response("RECV=%d;", uip_stat.icmp.recv);
        append_response("ROP=%d;", uip_stat.icmp.drop);
        append_response("TYPEERR=%d;", uip_stat.icmp.typeerr);
        append_response("CHKERR=%d;", uip_stat.icmp.chkerr);
    }
    else if (!strcmp(args, "ND6"))
    {
        append_response("ND6:");
        append_response("SENT=%d;", uip_stat.nd6.sent);
        append_response("RECV=%d;", uip_stat.nd6.recv);
        append_response("DROP=%d;", uip_stat.nd6.drop);
    }
    else if (!strcmp(args, "RESET"))
    {
        memset(csma_statics, 0, sizeof(int)*CSMA_STAT_NUM);
        memset(&uip_stat, 0, sizeof(uip_stats_t));
    }
#if SERVER_NODE==0
    else if (!strcmp(args, "get"))
    {
        append_response("INVALID_PARAM");
    }
    else if (!strcmp(args, "reset"))
    {
        append_response("INVALID_PARAM");
    }
#endif
    else
    {
        remove_response(3);
        append_response("INVALID_PARAM");
    }
}
void set_config(char *cmd)
{
    char *args;
    if (!strcmp_prefix(cmd, "recv_timeout", &args))
    {
        int val = atoi(args);
        if (val >= 0)
        {
            req_recv_timeout = val;
            append_response("OK");
        }
        else
        {
            append_response("INVALID_PARAM");
        }
    }
    else
    {
        append_response("INVALID_PARAM");
    }
}
/*---------------------------------------------------------------------------*/
static void
udp_rx_callback(struct simple_udp_connection *c,
                const uip_ipaddr_t *sender_addr,
                uint16_t sender_port,
                const uip_ipaddr_t *receiver_addr,
                uint16_t receiver_port,
                const uint8_t *data,
                uint16_t datalen)
{
    if (data[0] == '!')
    {
        int result = process_cmd((char *)data, datalen, '!');
        response_len = sprintf(response, "!R Received from ");
        APPEND_RESPONSE_BY(uiplib_ipaddr_snprint, sender_addr);
        if (datalen >= RESPONSE_CAPACITY - 2-CHECKSUM_LEN)
        {
            result=CMD_LEN_TOO_LONG;
        }
        append_response(" %s", get_cmd_error_str(result));
        prepare_request(sender_addr);
        if (!result)
        {
            append_response(":");
            //remove checksum and "\r\n" in message ('\n' has been replaced to '\0' by process_cmd)
            memcpy(RESPONSE_CUR, data, datalen - CHECKSUM_LEN - 2);
            response_len += datalen - CHECKSUM_LEN - 2;
        }
        //add message end
        response[response_len++] = '\r';
        response[response_len++] = '\n';
#if USE_CHECK_SUM
        calc_chksum((uint8_t *)response, response_len);
        response_len += CHECKSUM_LEN;
#endif
        uart_send_raw(response, response_len);
    }
    else
    {
        if (echo_messages)
        {
            LOG_INFO("Received request '%.*s' from ", datalen, (char *) data);
            LOG_INFO_6ADDR(sender_addr);
            LOG_INFO_("\n");
            /* send back the same string to the client as an echo reply */
            LOG_INFO("Sending response.\n");
        }
        simple_udp_sendto(&udp_conn, data, datalen, sender_addr);
    }
}

PROCESS_THREAD(udp_server_process, ev, data)
{
    static struct etimer periodic_timer;
    static char buf[64];
    static uint32_t len = 0;
    PROCESS_BEGIN();

    /* Initialize DAG root */
    NETSTACK_ROUTING.root_start();

    /* Initialize UDP connection */
    simple_udp_register(&udp_conn, UDP_SERVER_PORT, NULL,
                        UDP_CLIENT_PORT, udp_rx_callback);

    etimer_set(&periodic_timer, 100);

    while (1)
    {
        PROCESS_WAIT_EVENT();

        char *args;
        len = uart_gets(buf, sizeof(buf) - 1);
        if (len)
        {
            int result = process_cmd(buf, len, '?');
            if (result == 0)
            {
                char *cmd = buf + 1;
                response[0] = '!';
                response_len = 1 + strcpy_prefix(response + 1, cmd);
                response[response_len++] = ' ';
                if (!strcmp(cmd, "N"))
                {
                    print_neighbors();
                }
                else if (!strcmp_prefix(cmd, "R", &args))
                {
                    request_all(args);
                }
                else if (!strcmp_prefix(cmd, "T", &args))
                {
                    request_to(args);
                }
                else if (!strcmp(cmd, "L"))
                {
                    routes_print_links("links");
                }
                else if (!strcmp_prefix(cmd, "I", &args))
                {
                    print_info(args);
                }
                else if (!strcmp_prefix(cmd, "e", &args))
                {
                    echo_messages = args[0] == '0' ? 0 : 1;
                    append_response("OK");
                }
                else if (!strcmp_prefix(cmd, "C", &args))
                {
                    refresh_dag(args);
                }
                else if (!strcmp_prefix(cmd, "X", &args))
                {
                    set_config(args);
                }
                else
                {
                    response_len = sprintf(response, "!? %s", get_cmd_error_str(CMD_UNKNOWN));
                }
            }
            else
            {
                response_len = sprintf(response, "!? %s", get_cmd_error_str(result));
            }
            response[response_len++] = '\r';
            response[response_len++] = '\n';
#if USE_CHECK_SUM
            calc_chksum((uint8_t *)response, response_len);
            response_len += CHECKSUM_LEN;
#endif
            uart_send_raw(response, response_len);
        }
        etimer_set(&periodic_timer, 100);
    }
    PROCESS_END();
}
void uart_recv_callback()
{
    process_poll(&udp_server_process);
}
/*---------------------------------------------------------------------------*/
