/**********************************************************************************
 *
 * @file    irq.c
 * @brief   Interrupt handler
 *
 * @date    30 Apri 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          7  Nov  2022    shiwa           contiki 6lowpan client demo
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

#include "contiki.h"
#include "net/routing/routing.h"
#include "lib/random.h"
#include "net/netstack.h"
#include "net/ipv6/simple-udp.h"
#include <stdint.h>
#include <inttypes.h>
#include "uiplib.h"
#include "radio_gpio.h"
#include "radio_shield_config.h"
#include "cmd_utils.h"
#include "mac_statics.h"

#include "sys/log.h"
#define LOG_MODULE "App"
#define LOG_LEVEL LOG_LEVEL_INFO


void uart_send_raw(const char*s,uint32_t len);
uint32_t uart_gets(char *data, uint32_t len);
uint32_t uart_check(void);
void es32_led_set(int led,int val);

#define WITH_SERVER_REPLY  1
#define UDP_CLIENT_PORT 8765
#define UDP_SERVER_PORT 5678

#define SEND_INTERVAL         (2 * CLOCK_SECOND)

static struct simple_udp_connection udp_conn;
uip_ipaddr_t dest_ipaddr;

static int send_interval = (2 * CLOCK_SECOND);
static int check_interval=(CLOCK_SECOND/2);
static int send_rand = (CLOCK_SECOND/5);
static int send_min_len = 0;
static int send_enabled = 1;
static struct simple_udp_connection udp_conn;
static uint32_t rx_count = 0;
static uint32_t tx_count = 0;
static uint32_t missed_tx_count = 0;
static volatile int waiting_response = 0;

static int sent_len=0;
static uint32_t sent_time=0;
static char send_buf[260];
/* Response buffer */
#define RESPONSE_TOTAL 1024
static char response[RESPONSE_TOTAL];
static int response_len;
#define response_cur (response+response_len)
#define response_capacity (RESPONSE_TOTAL-response_len)

void print_neighbors()
{
    uip_ds6_nbr_t *n = uip_ds6_nbr_head();
    if (!n) {
        response_len += sprintf(response_cur, " NO_NEIGHBORS\r\n");
        return;
    }
    response_len += sprintf(response_cur, "\r\n");
    while (n) {
        response_len += uiplib_ipaddr_snprint(response_cur, RESPONSE_TOTAL - response_len, &n->ipaddr);
        response_len += sprintf(response_cur, "\r\n");
        n = uip_ds6_nbr_next(n);
    }
}
void print_info(char*args)
{
    response_len += sprintf(response_cur, " OK\r\n");
    if (!strcmp(args,"CSMA"))
    {
        response_len += sprintf(response_cur, "CSMA:TOTAL_INPUT=%8d\r\n",csma_statics[CSMA_STAT_INPUT]);
        response_len += sprintf(response_cur, "     ACK_IGNORED=%8d\r\n",csma_statics[CSMA_STAT_ACK_IGNORED]);
        response_len += sprintf(response_cur, "     PARSE_FAIL =%8d\r\n",csma_statics[CSMA_STAT_PARSE_FAILED]);
        response_len += sprintf(response_cur, "     NOT_FOR_US =%8d\r\n",csma_statics[CSMA_STAT_NOT_FOR_US]);
        response_len += sprintf(response_cur, "     FROM_SELF  =%8d\r\n",csma_statics[CSMA_STAT_FROM_SELF]); 
        response_len += sprintf(response_cur, "     DUPLICATE  =%8d\r\n",csma_statics[CSMA_STAT_DUPLICATE]);
        
        response_len += sprintf(response_cur, "     OUTPUT     =%8d\r\n",csma_statics[CSMA_STAT_OUTPUT]); 
        response_len += sprintf(response_cur, "     ERROR      =%8d\r\n",csma_statics[CSMA_STAT_TXERR]); 
        response_len += sprintf(response_cur, "     NOACK      =%8d\r\n",csma_statics[CSMA_STAT_NOACK]); 
        response_len += sprintf(response_cur, "     COLLISION  =%8d\r\n",csma_statics[CSMA_STAT_COLLISION]); 
    }
    else if (!strcmp(args,"IP"))
    {
        response_len += sprintf(response_cur, "IP  :SENT       =%8d\r\n",uip_stat.ip.sent);
        response_len += sprintf(response_cur, "     RECV       =%8d\r\n",uip_stat.ip.recv);
        response_len += sprintf(response_cur, "     DROP       =%8d\r\n",uip_stat.ip.drop);
        response_len += sprintf(response_cur, "     FORWARDED  =%8d\r\n",uip_stat.ip.forwarded);
        response_len += sprintf(response_cur, "     VHLERR     =%8d\r\n",uip_stat.ip.vhlerr);
        response_len += sprintf(response_cur, "     HBLENERR   =%8d\r\n",uip_stat.ip.hblenerr);
        response_len += sprintf(response_cur, "     LBLENERR   =%8d\r\n",uip_stat.ip.lblenerr);
        response_len += sprintf(response_cur, "     FRAGERR    =%8d\r\n",uip_stat.ip.fragerr);
        response_len += sprintf(response_cur, "     CHKERR     =%8d\r\n",uip_stat.ip.chkerr);
        response_len += sprintf(response_cur, "     PROTOERR   =%8d\r\n",uip_stat.ip.protoerr);
    }
    #if UIP_TCP
    else if (!strcmp(args,"IP"))
    {
    }
    #endif
    #if UIP_UDP
    else if (!strcmp(args,"UDP"))
    {
        response_len += sprintf(response_cur, "UDP :SENT       =%8d\r\n",uip_stat.udp.sent);
        response_len += sprintf(response_cur, "     RECV       =%8d\r\n",uip_stat.udp.recv);
        response_len += sprintf(response_cur, "     DROP       =%8d\r\n",uip_stat.udp.drop);
        response_len += sprintf(response_cur, "     CHKERR     =%8d\r\n",uip_stat.udp.chkerr);
    }
    #endif
    else if (!strcmp(args,"ICMP"))
    {
        response_len += sprintf(response_cur, "ICMP:SENT       =%8d\r\n",uip_stat.icmp.sent);
        response_len += sprintf(response_cur, "     RECV       =%8d\r\n",uip_stat.icmp.recv);
        response_len += sprintf(response_cur, "     DROP       =%8d\r\n",uip_stat.icmp.drop);
        response_len += sprintf(response_cur, "     TYPEERR    =%8d\r\n",uip_stat.icmp.typeerr);
        response_len += sprintf(response_cur, "     CHKERR     =%8d\r\n",uip_stat.icmp.chkerr);
    }
    else if (!strcmp(args,"ND6"))
    {
        response_len += sprintf(response_cur, "ND6 :SENT       =%8d\r\n",uip_stat.nd6.sent);
        response_len += sprintf(response_cur, "     RECV       =%8d\r\n",uip_stat.nd6.recv);
        response_len += sprintf(response_cur, "     DROP       =%8d\r\n",uip_stat.nd6.drop);
    }
    else if (!strcmp(args,"RESET"))
    {
        memset(csma_statics,0,sizeof(int)*CSMA_STAT_NUM);
        memset(&uip_stat,0,sizeof(uip_stats_t));
    }
    else
    {
        response_len += sprintf(response_cur-4, "INVALID_PARAM\r\n")-4;
    }
}
void stats_info(char *args)
{
    if (!strcmp(args,"get"))
    {
        response_len += sprintf(response_cur, " TX=%d,RX=%d,Missed=%d\r\n", tx_count, rx_count, missed_tx_count);
    }
    else if (!strcmp(args,"reset"))
    {
        tx_count = 0;
        rx_count = 0;
        missed_tx_count = 0;
        response_len += sprintf(response_cur, " OK\r\n");
    }
    else
    {
        print_info(args);
        //response_len += sprintf(response_cur, " INVALID_PARAM\r\n");
    }

}

void send_interval_control(char *args)
{
    char*param;
    if (!strcmp_prefix(args,"int",&param))
    {
        int new_int = atoi(param);
        if (new_int > 0 && new_int <= 1000 * 1000) {
            send_interval = new_int;
            response_len += sprintf(response_cur, " OK\r\n");
        } else {
            response_len += sprintf(response_cur, " INVALID_PARAM\r\n");
        }
    }
    else if (!strcmp_prefix(args,"en",&param))
    {
        int new_int = atoi(param);
        if (new_int==0||new_int==1)
        {
            response_len += sprintf(response_cur, " OK\r\n");
            send_enabled = new_int;
        }
        else
        {            
            response_len += sprintf(response_cur, " INVALID_PARAM\r\n");
        }
    } else if (!strcmp_prefix(args,"rand",&param))
    {
        int new_int = atoi(param);
        if (new_int>=0&&new_int<send_interval)
        {
            response_len += sprintf(response_cur, " OK\r\n");
            send_rand = new_int;
        }
        else
        {            
            response_len += sprintf(response_cur, " INVALID_PARAM\r\n");
        }
    } else if (!strcmp_prefix(args,"len",&param))
    {
        int new_int = atoi(param);
        if (new_int>=0&&new_int<500)
        {
            response_len += sprintf(response_cur, " OK\r\n");
            send_min_len = new_int;
        }
        else
        {            
            response_len += sprintf(response_cur, " INVALID_PARAM\r\n");
        }
    }
    else
    {
        response_len += sprintf(response_cur, " INVALID_PARAM\r\n");
    }
}
/*---------------------------------------------------------------------------*/
PROCESS(udp_client_process, "UDP client");
AUTOSTART_PROCESSES(&udp_client_process);
/*---------------------------------------------------------------------------*/
static void
udp_rx_callback(struct simple_udp_connection *c,
                const uip_ipaddr_t *sender_addr,
                uint16_t sender_port,
                const uip_ipaddr_t *receiver_addr,
                uint16_t receiver_port,
                const uint8_t *data,
                uint16_t datalen)
{
    if (!uip_ipaddr_cmp(&dest_ipaddr,sender_addr))
    {
        uiplib_ipaddr_snprint(response, 128, sender_addr);
        printf("Error received from other: %s\r\n", response);
        return;
    } 
    char *text = (char *)data;
    if (data[0] == '?') {
        uiplib_ipaddr_snprint(response, 128, sender_addr);
        printf("Received command from %s:\r\n", response);
        printf("%.*s\r\n", datalen, text);

        int result = process_cmd(text, datalen, '?');
        if (result == 0) {
            char *cmd = text + 1;
            char *args;
            response[0] = '!';
            response_len = 1 + strcpy_prefix(response + 1, cmd);

            //deal with commands
            if (!strcmp(cmd, "N")) {
                print_neighbors();
            } else if (!strcmp_prefix(cmd, "E", &args)) {
                response_len+=sprintf(response_cur,"\r\n");
                if (args[0]=='\r'&&args[1]=='\n')
                {
                    memcpy(response_cur,args+2,datalen-2-3-1);
                }
            } else if (!strcmp_prefix(text + 1, "I", &args)) {
                stats_info(args);
            } else if (!strcmp_prefix(text + 1, "S", &args)) {
                send_interval_control(args);
            } else {
                response_len = sprintf(response, "!? %s\r\n", get_cmd_error_str(CMD_UNKNOWN));
            }
        } else {
            response_len = sprintf(response, "!? %s\r\n", get_cmd_error_str(result));
        }
        
        uart_send_raw(response,response_len);
#if USE_CHECK_SUM
        calc_chksum((uint8_t *)response, response_len);
        response_len += CHECKSUM_LEN;
#endif
        simple_udp_sendto(&udp_conn, response, response_len, sender_addr);

    } else if (waiting_response) {
        //if we have sent a message and waiting for response
        LOG_INFO("Received response '%.*s' from ", datalen, (char *) data);
        LOG_INFO_6ADDR(sender_addr);
        LOG_INFO_(" in %lu ms\n",clock_time()-sent_time);
        if (sent_len != datalen) {
            LOG_INFO("Response len error,received %d,but expected %d\n", datalen, sent_len);
        } else if (memcmp(send_buf, (char *)data, datalen)) {
            LOG_INFO("Response data error, expected is %s\n", send_buf);
        } else {
            rx_count++;
            waiting_response = 0;
        }
        LOG_INFO_("\n");
    } else {
        LOG_INFO("Received '%.*s' from ", datalen, (char *) data);
        LOG_INFO_6ADDR(sender_addr);
        LOG_INFO_("while not waiting for response\n");
    }
}
/*---------------------------------------------------------------------------*/
PROCESS_THREAD(udp_client_process, ev, data)
{
    static struct etimer periodic_timer;
    static int unreachable_count=0;
    PROCESS_BEGIN();
    RadioShieldLedInit(RADIO_SHIELD_LED);
    /* Initialize UDP connection */
    simple_udp_register(&udp_conn, UDP_CLIENT_PORT, NULL,
                        UDP_SERVER_PORT, udp_rx_callback);

    etimer_set(&periodic_timer, check_interval+(random_rand()% send_rand));
    RadioShieldLedOn(RADIO_SHIELD_LED);
    while (1) {
        PROCESS_WAIT_EVENT_UNTIL(etimer_expired(&periodic_timer));

        if (NETSTACK_ROUTING.node_is_reachable() &&
            NETSTACK_ROUTING.get_root_ipaddr(&dest_ipaddr)) {
            es32_led_set(0,0);
            es32_led_set(1,1);
            if (waiting_response) {
                missed_tx_count++;
                waiting_response = 0;
            }
            if (send_enabled) {
                /* Print statistics every 10th TX */
                if (tx_count % 10 == 0) {
                    LOG_INFO("Tx/Rx/MissedTx: %" PRIu32 "/%" PRIu32 "/%" PRIu32 "\n",
                             tx_count, rx_count, missed_tx_count);
                }

                /* Send to DAG root */
                LOG_INFO("Sending request %"PRIu32" to ", tx_count);
                LOG_INFO_6ADDR(&dest_ipaddr);
                LOG_INFO_("\n");
                sent_len=snprintf(send_buf, 260, "hello %" PRIu32 "", tx_count);
                if (send_min_len>0&&sent_len<send_min_len)
                {
                    while (sent_len<send_min_len)
                    {
                        send_buf[sent_len]='A'+(sent_len%26);
                        sent_len++;
                    }
                }
                send_buf[sent_len]=0;
                waiting_response = 1;
                sent_time=clock_time();
                simple_udp_sendto(&udp_conn, send_buf, sent_len, &dest_ipaddr);
                tx_count++;
            }
            unreachable_count=0; 
            
            etimer_set(&periodic_timer, send_interval-send_rand/2+(random_rand()%send_rand));
        } else {
            es32_led_set(0,1);
            es32_led_set(1,0);
            LOG_INFO("Not reachable yet,%d\n",unreachable_count);
            unreachable_count++;
            
            etimer_set(&periodic_timer, check_interval);
        }
        RadioShieldLedToggle(RADIO_SHIELD_LED);
        
    }

    PROCESS_END();
}

void uart_recv_callback()
{
    //process_poll(&udp_client_process);
}
/*---------------------------------------------------------------------------*/
