/**********************************************************************************
 *
 * @file    main.c
 * @brief   Main file for DEMO
 *
 * @date    30 Apri 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          30 Apri 2021    yanght          the first version
 *          7  Nov  2022    shiwa           ES32 for contiki
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */
#include <contiki.h>
#include <string.h>
#include "main.h"
#include <stdio.h>
/* Private Macros ------------------------------------------------------------ */

/* Private Variables --------------------------------------------------------- */

/* Public Variables ---------------------------------------------------------- */

uart_handle_t g_uart_init;

uint8_t g_rx_buf[256];
uint32_t g_rx_len = 0;

static GPIO_TypeDef*LedPorts[4]={
    LED_0_PORT,            
    LED_1_PORT,
    LED_2_PORT,
    LED_3_PORT,
};
static uint32_t LedPins[4]={
    LED_0_PIN,            
    LED_1_PIN,
    LED_2_PIN,
    LED_3_PIN,
};
/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Function ---------------------------------------------------------- */

/**
  * @brief  Initializate pin of uart module.
  * @retval None
  */
void init_uart_pin(void)
{
    gpio_init_t gpio_init;
    memset(&gpio_init, 0x00, sizeof(gpio_init_t));

    /* Initialize tx pin: PC0 */
    gpio_init.mode = GPIO_MODE_OUTPUT;
    gpio_init.odos = GPIO_PUSH_PULL;
    gpio_init.pupd = GPIO_PUSH_UP;
    gpio_init.podrv = GPIO_OUT_DRIVE_1;
    gpio_init.nodrv = GPIO_OUT_DRIVE_0_1;
    gpio_init.flt  = GPIO_FILTER_DISABLE;
    gpio_init.type = GPIO_TYPE_CMOS;
    gpio_init.func = GPIO_FUNC_3;
    ald_gpio_init(UART_TX_PORT, UART_TX_PIN, &gpio_init);

    /* Initialize rx pin: PC1 */
    gpio_init.mode = GPIO_MODE_INPUT;
    gpio_init.odos = GPIO_PUSH_PULL;
    gpio_init.pupd = GPIO_PUSH_UP;
    gpio_init.podrv = GPIO_OUT_DRIVE_1;
    gpio_init.nodrv = GPIO_OUT_DRIVE_0_1;
    gpio_init.flt  = GPIO_FILTER_DISABLE;
    gpio_init.type = GPIO_TYPE_CMOS;
    gpio_init.func = GPIO_FUNC_3;
    ald_gpio_init(UART_RX_PORT, UART_RX_PIN, &gpio_init);

    return;
}

/**
  * @brief  Send message complete.
  * @param  arg: Pointer to uart_handle_t structure.
  * @retval None.
  */
void uart_send_complete(uart_handle_t *arg)
{
    return;
}

/**
  * @brief  Receive a message complete.
  * @param  arg: Pointer to uart_handle_t structure.
  * @retval None.
  */
void uart_recv_complete(uart_handle_t *huart)
{
    extern void uart_recv_callback();
    g_rx_len=huart->rx_count;
    uart_recv_callback();
    return;
}

/**
  * @brief  Occurs error.
  * @param  arg: Pointer to uart_handle_t structure.
  * @retval None.
  */
void uart_error(uart_handle_t *arg)
{
    return;
}

/**
  * @brief  Initialize ES32
  * @retval Status.
  */
void es32_uart_init(void)
{
    /* Initialize ALD */
    ald_cmu_init();
    ald_cmu_pll1_config(CMU_PLL1_INPUT_HRC_6, CMU_PLL1_OUTPUT_72M);
    ald_cmu_clock_config(CMU_CLOCK_PLL1, 72000000);
    ald_cmu_perh_clock_config(CMU_PERH_ALL, ENABLE);
    ald_mcu_irq_config(UART2_IRQn, 3, 3, ENABLE);

    /* Initialize pin */
    init_uart_pin();

    memset(&g_uart_init, 0x00, sizeof(uart_handle_t));
    /* Initialize uart */
    g_uart_init.perh             = UART_X;
    g_uart_init.init.baud        = 115200;
    g_uart_init.init.word_length = UART_WORD_LENGTH_8B;
    g_uart_init.init.stop_bits   = UART_STOP_BITS_1;
    g_uart_init.init.parity      = UART_PARITY_NONE;
    g_uart_init.init.mode        = UART_MODE_UART;
    g_uart_init.init.fctl        = UART_HW_FLOW_CTL_DISABLE;
    g_uart_init.rx_cplt_cbk      = uart_recv_complete;
    g_uart_init.tx_cplt_cbk      = uart_send_complete;
    ald_uart_init(&g_uart_init);

    ald_uart_rx_fifo_config(&g_uart_init, UART_RXFIFO_1BYTE);
    ald_uart_tx_fifo_config(&g_uart_init, UART_TXFIFO_EMPTY);
    
    ald_uart_recv_frame_by_it(&g_uart_init, g_rx_buf, sizeof(g_rx_buf),100);
}
void uart_send_raw(const char*s,uint32_t len)
{
    ald_uart_send(&g_uart_init, (uint8_t*)s, len, 1000);
}
int fputc(int ch, FILE *fp)
{
    uint8_t c = ch;
    ald_uart_send(&g_uart_init, &c, 1, 1000);
    return 0;
}
void uart_send_ch(char c)
{
    if (c == '\n') {
        fputc('\r', stdout);
    }
    fputc(c, stdout);
}
void uart_send_str(const char *s)
{
    while (*s) {
        uart_send_ch(*s++);
    }

}
uint32_t uart_check()
{
    return g_rx_len;
}
uint32_t uart_gets(char*data,uint32_t len)
{
    if (g_rx_len==0)
    {
        return 0;
    }
    len=MIN(len,g_rx_len);
    memcpy(data,g_rx_buf,len);
    g_rx_len=0;
    ald_uart_recv_frame_by_it(&g_uart_init, g_rx_buf, sizeof(g_rx_buf),100);
    return len;
}

void es32_led_init()
{
    gpio_init_t gpio_init;
    gpio_init.mode = GPIO_MODE_OUTPUT;
    gpio_init.odos = GPIO_PUSH_PULL;
    gpio_init.pupd = GPIO_PUSH_UP;
    gpio_init.podrv = GPIO_OUT_DRIVE_1;
    gpio_init.nodrv = GPIO_OUT_DRIVE_6;
    gpio_init.flt  = GPIO_FILTER_DISABLE;
    gpio_init.type = GPIO_TYPE_CMOS;
    gpio_init.func = GPIO_FUNC_1;
    for (int i=0;i<4;i++)
    {
        ald_gpio_init(LedPorts[i],LedPins[i], &gpio_init);
        ald_gpio_write_pin( LedPorts[i],LedPins[i],SET);
    }
}
void es32_led_set(int led,int val)
{
    ald_gpio_write_pin( LedPorts[led],LedPins[led],val);
} 
void es32_led_toggle(int led)
{
    ald_gpio_toggle_pin( LedPorts[led],LedPins[led]);
}
void es32_init()
{
    es32_uart_init();
    es32_led_init();
}
/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
