/**********************************************************************************
 *
 * @file    udp-server.c
 * @brief   UDP server demo
 *
 * @date    30 Apri 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          7  Nov  2022    shiwa           contiki 6lowpan client demo
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

#include "contiki.h"
#include "net/routing/routing.h"
#include "net/netstack.h"
#include "net/ipv6/simple-udp.h"
#include "uiplib.h"
#include <stdlib.h>
#include "cmd_utils.h"
#include "mac_statics.h"

#include "sys/log.h"
#define LOG_MODULE "App"
#define LOG_LEVEL LOG_LEVEL_INFO

#define WITH_SERVER_REPLY  1
#define UDP_CLIENT_PORT 8765
#define UDP_SERVER_PORT 5678

static struct simple_udp_connection udp_conn;
const char g_app_name[] = "server";

/* Server configuration */
static int echo_messages = 1;
static uip_ipaddr_t req_target_addr;
static int req_target_to = 0;

/* Response buffer */
#define RESPONSE_TOTAL 1024
static char response[RESPONSE_TOTAL];
static int response_len;
#define response_cur (response+response_len)
#define response_capacity (RESPONSE_TOTAL-response_len)

/*Request buffer */
static char req_buf[128];
static int req_len = 0;
static uip_sr_node_t *req_cur = NULL;
static struct ctimer req_timer;
static uint32_t req_send_time = 0;
static int req_recv_timeout = 500;

/* UART utils functions */
void uart_send_raw(const char *s, uint32_t len);
uint32_t uart_gets(char *data, uint32_t len);
uint32_t uart_check(void);

PROCESS(udp_server_process, "UDP server");
AUTOSTART_PROCESSES(&udp_server_process);
/*---------------------------------------------------------------------------*/

void routes_print_links(const char *str)
{
    extern int rpl_dag_root_is_root(void);
    if (rpl_dag_root_is_root())
    {
        if (uip_sr_num_nodes() > 0)
        {
            uip_sr_node_t *link;
            /* Our routing links */
            response_len += sprintf(response_cur, "\r\n");
            link = uip_sr_node_head();
            while (link != NULL)
            {
                response_len += uip_sr_link_snprint(response_cur, response_capacity, link);
                response_len += sprintf(response_cur, "\r\n");
                link = uip_sr_node_next(link);
            }
        }
        else
        {
            response_len += sprintf(response_cur, " NO_LINKS\r\n");
        }
    }
}
void print_neighbors()
{
    uip_ds6_nbr_t *n = uip_ds6_nbr_head();
    if (!n)
    {
        response_len += sprintf(response_cur, " NO_NEIGHBORS\r\n");
        return;
    }
    response_len += sprintf(response_cur, "\r\n");
    while (n)
    {
        response_len += uiplib_ipaddr_snprint(response_cur, RESPONSE_TOTAL - response_len, &n->ipaddr);
        response_len += sprintf(response_cur, "\r\n");
        n = uip_ds6_nbr_next(n);
    }
}
void prepare_request(const uip_ipaddr_t *inaddr);
void request_timeout(void *a)
{
    uip_ipaddr_t child_ipaddr;
    NETSTACK_ROUTING.get_sr_node_ipaddr(&child_ipaddr, (uip_sr_node_t *)a);
    response_len = 0;
    response_len += sprintf(response_cur, "!R Request to ");
    response_len += uiplib_ipaddr_snprint(response_cur, RESPONSE_TOTAL - response_len, &child_ipaddr);
    response_len += sprintf(response_cur, " timeout\r\n");
#if USE_CHECK_SUM
    calc_chksum((uint8_t *)response, response_len);
    response_len += CHECKSUM_LEN;
#endif
    uart_send_raw(response, response_len);
    prepare_request(&child_ipaddr);

}
void refresh_dag(char *args)
{
    if (!strcmp(args, "repair"))
    {
        NETSTACK_ROUTING.global_repair("Refresh");
        response_len += sprintf(response_cur, " OK\r\n");
    }
    else
    {
        response_len += sprintf(response_cur, " INVALID_PARAM\r\n");
    }
}
void prepare_request(const uip_ipaddr_t *inaddr)
{
    uip_ipaddr_t toaddr;
    if (inaddr && req_cur)
    {
        NETSTACK_ROUTING.get_sr_node_ipaddr(&toaddr, req_cur);
        if (!uip_ipaddr_cmp(&toaddr, inaddr))
        {
            return;
        }

        response_len += sprintf(response_cur, " in %lums", clock_time() - req_send_time);
        req_send_time = 0;
        ctimer_stop(&req_timer);
        req_cur = uip_sr_node_next(req_cur);
    }

    if (req_cur)
    {
        NETSTACK_ROUTING.get_sr_node_ipaddr(&toaddr, req_cur);
        simple_udp_sendto(&udp_conn, req_buf, req_len, &toaddr);
        req_send_time = clock_time();
        ctimer_set(&req_timer, req_recv_timeout, request_timeout, req_cur);
    }
    else
    {
        req_len = 0;
    }
}
void request_all(const char *cmd)
{
    extern int rpl_dag_root_is_root(void);
    if (req_len != 0)
    {
        response_len += sprintf(response_cur, " ERROR_BUSY\r\n");
        return;
    }
    //prepare command
    req_len = snprintf(req_buf, 127, "?%s\r\n", cmd);
#if USE_CHECK_SUM
    calc_chksum((uint8_t *)req_buf, req_len);
    req_len += CHECKSUM_LEN;
#endif
    if (rpl_dag_root_is_root())
    {
        if (req_target_to)
        {
            //send to one selected client
            response_len += sprintf(response_cur, " send to ");
            response_len += uiplib_ipaddr_snprint(response_cur, RESPONSE_TOTAL - response_len, &req_target_addr);
            response_len += sprintf(response_cur, "\r\n");
            req_send_time = clock_time();
            simple_udp_sendto(&udp_conn, req_buf, req_len, &req_target_addr);
        }
        else if (uip_sr_num_nodes() > 1)
        {
            //send to all
            uip_ipaddr_t child_ipaddr;
            uip_sr_node_t *link;

            link = uip_sr_node_head();
            req_cur = uip_sr_node_next(link);
            response_len += sprintf(response_cur, " send to %d clients\r\n", uip_sr_num_nodes() - 1);
            while (link != NULL)
            {
                NETSTACK_ROUTING.get_sr_node_ipaddr(&child_ipaddr, link);
                if (link->parent != NULL)
                {
                    response_len += uiplib_ipaddr_snprint(response_cur, RESPONSE_TOTAL - response_len, &child_ipaddr);
                    response_len += sprintf(response_cur, "\r\n");
                    //simple_udp_sendto(&udp_conn, req_buf, req_len, &child_ipaddr);
                }
                link = uip_sr_node_next(link);
            }
            prepare_request(NULL);
        }
        else
        {
            req_len = 0;
            response_len += sprintf(response_cur, " NO_LINKS\r\n");
        }
    }
}
void request_to(const char *args)
{
    if (!strcmp(args, "all"))
    {
        response_len += sprintf(response_cur, " OK\r\n");
        req_target_to = 0;
    }
    else if (!strcmp(args, "show"))
    {
        if (req_target_to)
        {
            response_len += sprintf(response_cur, " ");
            response_len += uiplib_ipaddr_snprint(response_cur, 100, &req_target_addr);
            response_len += sprintf(response_cur, "\r\n");
        }
        else
        {
            response_len += sprintf(response_cur, " all\r\n");
        }
    }
    else
    {
        uip_sr_node_t *link;
        uip_ipaddr_t child_ipaddr;
        uiplib_ip6addrconv(args, &req_target_addr);

        link = uip_sr_node_head();
        while (link != NULL)
        {
            NETSTACK_ROUTING.get_sr_node_ipaddr(&child_ipaddr, link);
            if (link->parent != NULL && uip_ipaddr_cmp(&child_ipaddr, &req_target_addr))
            {
                break;
            }
            link = uip_sr_node_next(link);
        }
        if (link)
        {
            req_target_to = 1;
            response_len += sprintf(response_cur, " OK\r\n");
        }
        else
        {
            req_target_to = 0;
            response_len += sprintf(response_cur, " NOTFOUND\r\n");
        }
    }
}
void print_info(char *args)
{
    response_len += sprintf(response_cur, " OK\r\n");
    if (!strcmp(args, "CSMA"))
    {
        response_len += sprintf(response_cur, "CSMA:INPUT      =%8d\r\n", csma_statics[CSMA_STAT_INPUT]);
        response_len += sprintf(response_cur, "     ACK_IGNORED=%8d\r\n", csma_statics[CSMA_STAT_ACK_IGNORED]);
        response_len += sprintf(response_cur, "     PARSE_FAIL =%8d\r\n", csma_statics[CSMA_STAT_PARSE_FAILED]);
        response_len += sprintf(response_cur, "     NOT_FOR_US =%8d\r\n", csma_statics[CSMA_STAT_NOT_FOR_US]);
        response_len += sprintf(response_cur, "     FROM_SELF  =%8d\r\n", csma_statics[CSMA_STAT_FROM_SELF]);
        response_len += sprintf(response_cur, "     DUPLICATE  =%8d\r\n", csma_statics[CSMA_STAT_DUPLICATE]);

        response_len += sprintf(response_cur, "     OUTPUT     =%8d\r\n", csma_statics[CSMA_STAT_OUTPUT]);
        response_len += sprintf(response_cur, "     ERROR      =%8d\r\n", csma_statics[CSMA_STAT_TXERR]);
        response_len += sprintf(response_cur, "     NOACK      =%8d\r\n", csma_statics[CSMA_STAT_NOACK]);
        response_len += sprintf(response_cur, "     COLLISION  =%8d\r\n", csma_statics[CSMA_STAT_COLLISION]);
    }
    else if (!strcmp(args, "IP"))
    {
        response_len += sprintf(response_cur, "IP  :SENT       =%8d\r\n", uip_stat.ip.sent);
        response_len += sprintf(response_cur, "     RECV       =%8d\r\n", uip_stat.ip.recv);
        response_len += sprintf(response_cur, "     DROP       =%8d\r\n", uip_stat.ip.drop);
        response_len += sprintf(response_cur, "     FORWARDED  =%8d\r\n", uip_stat.ip.forwarded);
        response_len += sprintf(response_cur, "     VHLERR     =%8d\r\n", uip_stat.ip.vhlerr);
        response_len += sprintf(response_cur, "     HBLENERR   =%8d\r\n", uip_stat.ip.hblenerr);
        response_len += sprintf(response_cur, "     LBLENERR   =%8d\r\n", uip_stat.ip.lblenerr);
        response_len += sprintf(response_cur, "     FRAGERR    =%8d\r\n", uip_stat.ip.fragerr);
        response_len += sprintf(response_cur, "     CHKERR     =%8d\r\n", uip_stat.ip.chkerr);
        response_len += sprintf(response_cur, "     PROTOERR   =%8d\r\n", uip_stat.ip.protoerr);
    }
#if UIP_TCP
    else if (!strcmp(args, "IP"))
    {
    }
#endif
#if UIP_UDP
    else if (!strcmp(args, "UDP"))
    {
        response_len += sprintf(response_cur, "UDP :SENT       =%8d\r\n", uip_stat.udp.sent);
        response_len += sprintf(response_cur, "     RECV       =%8d\r\n", uip_stat.udp.recv);
        response_len += sprintf(response_cur, "     DROP       =%8d\r\n", uip_stat.udp.drop);
        response_len += sprintf(response_cur, "     CHKERR     =%8d\r\n", uip_stat.udp.chkerr);
    }
#endif
    else if (!strcmp(args, "ICMP"))
    {
        response_len += sprintf(response_cur, "ICMP:SENT       =%8d\r\n", uip_stat.icmp.sent);
        response_len += sprintf(response_cur, "     RECV       =%8d\r\n", uip_stat.icmp.recv);
        response_len += sprintf(response_cur, "     DROP       =%8d\r\n", uip_stat.icmp.drop);
        response_len += sprintf(response_cur, "     TYPEERR    =%8d\r\n", uip_stat.icmp.typeerr);
        response_len += sprintf(response_cur, "     CHKERR     =%8d\r\n", uip_stat.icmp.chkerr);
    }
    else if (!strcmp(args, "ND6"))
    {
        response_len += sprintf(response_cur, "ND6 :SENT       =%8d\r\n", uip_stat.nd6.sent);
        response_len += sprintf(response_cur, "     RECV       =%8d\r\n", uip_stat.nd6.recv);
        response_len += sprintf(response_cur, "     DROP       =%8d\r\n", uip_stat.nd6.drop);
    }
    else if (!strcmp(args, "RESET"))
    {
        memset(csma_statics, 0, sizeof(int)*CSMA_STAT_NUM);
        memset(&uip_stat, 0, sizeof(uip_stats_t));
    }
    else
    {
        response_len += sprintf(response_cur - 4, "INVALID_PARAM\r\n") - 4;
    }
}
void set_config(char *cmd)
{
    char *args;
    if (!strcmp_prefix(cmd, "recv_timeout", &args))
    {
        int val = atoi(args);
        if (val >= 0)
        {
            req_recv_timeout = val;
            response_len += sprintf(response_cur, " OK\r\n");
        }
        else
        {
            response_len += sprintf(response_cur, " INVALID_PARAM\r\n");
        }
    }
    else
    {
        response_len += sprintf(response_cur, " INVALID_PARAM\r\n");
    }
}
/*---------------------------------------------------------------------------*/
static void
udp_rx_callback(struct simple_udp_connection *c,
                const uip_ipaddr_t *sender_addr,
                uint16_t sender_port,
                const uip_ipaddr_t *receiver_addr,
                uint16_t receiver_port,
                const uint8_t *data,
                uint16_t datalen)
{
    if (data[0] == '!')
    {
        int result = process_cmd((char *)data, datalen, '!');
        response_len = sprintf(response, "!R Received from ");
        response_len += uiplib_ipaddr_snprint(response_cur, 128, sender_addr);
        response_len += sprintf(response_cur, " %s", get_cmd_error_str(result));
        prepare_request(sender_addr);
        response_len += sprintf(response_cur, "\r\n");
        if (!result)
        {
            memcpy(response_cur, data, datalen - CHECKSUM_LEN);
            response_len += datalen - CHECKSUM_LEN;
            response_len += sprintf(response_cur, "\r\n");
        }
#if USE_CHECK_SUM
        calc_chksum((uint8_t *)response, response_len);
        response_len += CHECKSUM_LEN;
#endif
        uart_send_raw(response, response_len);
    }
    else
    {
        if (echo_messages)
        {
            LOG_INFO("Received request '%.*s' from ", datalen, (char *) data);
            LOG_INFO_6ADDR(sender_addr);
            LOG_INFO_("\n");
            /* send back the same string to the client as an echo reply */
            LOG_INFO("Sending response.\n");
        }
        simple_udp_sendto(&udp_conn, data, datalen, sender_addr);
    }
}

PROCESS_THREAD(udp_server_process, ev, data)
{
    static struct etimer periodic_timer;
    static char buf[64];
    static uint32_t len = 0;
    PROCESS_BEGIN();

    /* Initialize DAG root */
    NETSTACK_ROUTING.root_start();

    /* Initialize UDP connection */
    simple_udp_register(&udp_conn, UDP_SERVER_PORT, NULL,
                        UDP_CLIENT_PORT, udp_rx_callback);

    etimer_set(&periodic_timer, 100);

    while (1)
    {
        PROCESS_WAIT_EVENT();

        char *args;
        len = uart_gets(buf, sizeof(buf) - 1);
        if (len)
        {
            int result = process_cmd(buf, len, '?');
            if (result == 0)
            {
                char *cmd = buf + 1;
                response[0] = '!';
                response_len = 1 + strcpy_prefix(response + 1, cmd);
                if (!strcmp(cmd, "N"))
                {
                    print_neighbors();
                }
                else if (!strcmp_prefix(cmd, "R", &args))
                {
                    request_all(args);
                }
                else if (!strcmp_prefix(cmd, "T", &args))
                {
                    request_to(args);
                }
                else if (!strcmp(cmd, "L"))
                {
                    routes_print_links("links");
                }
                else if (!strcmp_prefix(cmd, "I", &args))
                {
                    print_info(args);
                }
                else if (!strcmp_prefix(cmd, "e", &args))
                {
                    echo_messages = args[0] == '0' ? 0 : 1;
                    response_len += sprintf(response_cur, " OK\r\n");
                }
                else if (!strcmp_prefix(cmd, "C", &args))
                {
                    refresh_dag(args);
                }
                else if (!strcmp_prefix(cmd, "X", &args))
                {
                    set_config(args);
                }
                else
                {
                    response_len = sprintf(response, "!? %s\r\n", get_cmd_error_str(CMD_UNKNOWN));
                }
            }
            else
            {
                response_len = sprintf(response, "!? %s\r\n", get_cmd_error_str(result));
            }
#if USE_CHECK_SUM
            calc_chksum((uint8_t *)response, response_len);
            response_len += CHECKSUM_LEN;
#endif
            uart_send_raw(response, response_len);
        }
        etimer_set(&periodic_timer, 100);
    }
    PROCESS_END();
}
void uart_recv_callback()
{
    process_poll(&udp_server_process);
}
/*---------------------------------------------------------------------------*/
