/**********************************************************************************
 *
 * @file    irq.c
 * @brief   Interrupt handler
 *
 * @date    30 Apri 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          7  Nov  2022    shiwa           contiki 6lowpan client demo
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

#include "contiki.h"
#include "net/routing/routing.h"
#include "lib/random.h"
#include "net/netstack.h"
#include "net/ipv6/simple-udp.h"
#include <stdint.h>
#include <inttypes.h>
#include "uiplib.h"
#include "radio_gpio.h"
#include "radio_shield_config.h"
#include "cmd_utils.h"
#include "mac_statics.h"
#include <stdarg.h>
#include "sys/log.h"
#define LOG_MODULE "App"
#define LOG_LEVEL LOG_LEVEL_INFO
#define SERVER_NODE 0


void uart_send_raw(const char *s, uint32_t len);
uint32_t uart_gets(char *data, uint32_t len);
uint32_t uart_check(void);
void es32_led_set(int led, int val);

#define WITH_SERVER_REPLY  1
#define UDP_CLIENT_PORT 8765
#define UDP_SERVER_PORT 5678

#define SEND_INTERVAL         (2 * CLOCK_SECOND)

static struct simple_udp_connection udp_conn;
uip_ipaddr_t dest_ipaddr;

static int send_interval = (2 * CLOCK_SECOND);
static int check_interval = (CLOCK_SECOND / 2);
static int send_rand = (CLOCK_SECOND / 5);
static int send_min_len = 0;
static int send_enabled = 1;
static struct simple_udp_connection udp_conn;
static uint32_t rx_count = 0;
static uint32_t tx_count = 0;
static uint32_t missed_tx_count = 0;
static volatile int waiting_response = 0;

static int sent_len = 0;
static uint32_t sent_time = 0;
static char send_buf[128];
/* Response buffer */
#define RESPONSE_TOTAL 384
static char response[RESPONSE_TOTAL];
static int response_len;
#define RESPONSE_CUR (response+response_len)
#define RESPONSE_CAPACITY (RESPONSE_TOTAL-response_len)

void append_response(const char *fmt, ...)
{
    va_list args;
    va_start(args, fmt);
    response_len += vsnprintf(RESPONSE_CUR, RESPONSE_CAPACITY, fmt, args);
    va_end(args);
}
void remove_response(int count)
{
    if (count < response_len)
        response_len -= count;
    else
        response_len = 0;
}
#define APPEND_RESPONSE_BY(func,...) (response_len+=func(RESPONSE_CUR,RESPONSE_CAPACITY ,##__VA_ARGS__))

void print_neighbors()
{
    uip_ds6_nbr_t *n = uip_ds6_nbr_head();
    if (!n)
    {
        append_response("NO_NEIGHBORS");
        return;
    }
    while (n)
    {
        APPEND_RESPONSE_BY(uiplib_ipaddr_snprint, &n->ipaddr);
        append_response(";");
        n = uip_ds6_nbr_next(n);
    }
}
void print_info(char *args)
{
    append_response("OK ");
    if (!strcmp(args, "CSMA"))
    {
        append_response("CSMA:");
        append_response("INPUT=%d;", csma_statics[CSMA_STAT_INPUT]);
        append_response("ACK_IGNORED=%d;", csma_statics[CSMA_STAT_ACK_IGNORED]);
        append_response("PARSE_FAIL=%d;", csma_statics[CSMA_STAT_PARSE_FAILED]);
        append_response("NOT_FOR_US=%d;", csma_statics[CSMA_STAT_NOT_FOR_US]);
        append_response("FROM_SELF=%d;", csma_statics[CSMA_STAT_FROM_SELF]);
        append_response("DUPLICATE=%d;", csma_statics[CSMA_STAT_DUPLICATE]);
        append_response("OUTPUT=%d;", csma_statics[CSMA_STAT_OUTPUT]);
        append_response("ERROR=%d;", csma_statics[CSMA_STAT_TXERR]);
        append_response("NOACK=%d;", csma_statics[CSMA_STAT_NOACK]);
        append_response("COLLISION=%d;", csma_statics[CSMA_STAT_COLLISION]);
    }
    else if (!strcmp(args, "IP"))
    {
        append_response("IP:");
        append_response("SENT=%d;", uip_stat.ip.sent);
        append_response("RECV=%d;", uip_stat.ip.recv);
        append_response("DROP=%d;", uip_stat.ip.drop);
        append_response("FORWARDED=%d;", uip_stat.ip.forwarded);
        append_response("VHLERR=%d;", uip_stat.ip.vhlerr);
        append_response("HBLENERR=%d;", uip_stat.ip.hblenerr);
        append_response("LBLENERR=%d;", uip_stat.ip.lblenerr);
        append_response("FRAGERR=%d;", uip_stat.ip.fragerr);
        append_response("CHKERR=%d;", uip_stat.ip.chkerr);
        append_response("PROTOERR=%d;", uip_stat.ip.protoerr);
    }
#if UIP_TCP
    else if (!strcmp(args, "TCP"))
    {
        append_response("TCP:");
        append_response("SENT=%d;", uip_stat.tcp.sent);
        append_response("RECV=%d;", uip_stat.tcp.recv);
        append_response("DROP=%d;", uip_stat.tcp.drop);
        append_response("CHKERR=%d;", uip_stat.tcp.chkerr);
        append_response("ACKERR=%d;", uip_stat.tcp.ackerr);
        append_response("RST=%d;", uip_stat.tcp.rst);
        append_response("REXMIT=%d;", uip_stat.tcp.rexmit);
        append_response("SYNDROP=%d;", uip_stat.tcp.syndrop);
        append_response("SYNRST=%d;", uip_stat.tcp.synrst);
    }
#endif
#if UIP_UDP
    else if (!strcmp(args, "UDP"))
    {
        append_response("UDP:");
        append_response("SENT=%d;", uip_stat.udp.sent);
        append_response("RECV=%d;", uip_stat.udp.recv);
        append_response("DROP=%d;", uip_stat.udp.drop);
        append_response("CHKERR=%d;", uip_stat.udp.chkerr);
    }
#endif
    else if (!strcmp(args, "ICMP"))
    {
        append_response("ICMP:");
        append_response("SENT=%d;", uip_stat.icmp.sent);
        append_response("RECV=%d;", uip_stat.icmp.recv);
        append_response("ROP=%d;", uip_stat.icmp.drop);
        append_response("TYPEERR=%d;", uip_stat.icmp.typeerr);
        append_response("CHKERR=%d;", uip_stat.icmp.chkerr);
    }
    else if (!strcmp(args, "ND6"))
    {
        append_response("ND6:");
        append_response("SENT=%d;", uip_stat.nd6.sent);
        append_response("RECV=%d;", uip_stat.nd6.recv);
        append_response("DROP=%d;", uip_stat.nd6.drop);
    }
    else if (!strcmp(args, "RESET"))
    {
        memset(csma_statics, 0, sizeof(int)*CSMA_STAT_NUM);
        memset(&uip_stat, 0, sizeof(uip_stats_t));
    }
    else
    {
        remove_response(3);
        append_response("INVALID_PARAM") ;
    }
}
void stats_info(char *args)
{
    if (!strcmp(args, "get"))
    {
        append_response("TX=%d,RX=%d,Missed=%d", tx_count, rx_count, missed_tx_count);
    }
    else if (!strcmp(args, "reset"))
    {
        tx_count = 0;
        rx_count = 0;
        missed_tx_count = 0;
        append_response("OK");
    }
    else
    {
        print_info(args);
    }

}

void send_interval_control(char *args)
{
    char *param;
    if (!strcmp_prefix(args, "int", &param))
    {
        int new_int = atoi(param);
        if (new_int > 0 && new_int <= 1000 * 1000)
        {
            send_interval = new_int;
            append_response("OK");
        }
        else
        {
            append_response("INVALID_PARAM");
        }
    }
    else if (!strcmp_prefix(args, "en", &param))
    {
        int new_int = atoi(param);
        if (new_int == 0 || new_int == 1)
        {
            append_response("OK");
            send_enabled = new_int;
        }
        else
        {
            append_response("INVALID_PARAM");
        }
    }
    else if (!strcmp_prefix(args, "rand", &param))
    {
        int new_int = atoi(param);
        if (new_int >= 0 && new_int < send_interval)
        {
            append_response("OK");
            send_rand = new_int;
        }
        else
        {
            append_response("INVALID_PARAM");
        }
    }
    else if (!strcmp_prefix(args, "len", &param))
    {
        int new_int = atoi(param);
        if (new_int >= 0 && new_int < 500)
        {
            append_response("OK");
            send_min_len = new_int;
        }
        else
        {
            append_response("INVALID_PARAM");
        }
    }
    else
    {
        append_response("INVALID_PARAM");
    }
}
/*---------------------------------------------------------------------------*/
PROCESS(udp_client_process, "UDP client");
AUTOSTART_PROCESSES(&udp_client_process);
/*---------------------------------------------------------------------------*/
static void
udp_rx_callback(struct simple_udp_connection *c,
                const uip_ipaddr_t *sender_addr,
                uint16_t sender_port,
                const uip_ipaddr_t *receiver_addr,
                uint16_t receiver_port,
                const uint8_t *data,
                uint16_t datalen)
{
    if (!uip_ipaddr_cmp(&dest_ipaddr, sender_addr))
    {
        APPEND_RESPONSE_BY(uiplib_ipaddr_snprint, sender_addr);
        printf("Error received from other: %s\r\n", response);
        return;
    }
    char *text = (char *)data;
    if (data[0] == '?')
    {
        APPEND_RESPONSE_BY(uiplib_ipaddr_snprint, sender_addr);
        printf("Received command from %s:\r\n", response);
        printf("%.*s\r\n", datalen, text);

        int result = process_cmd(text, datalen, '?');
        if (datalen >= RESPONSE_CAPACITY - 2 - CHECKSUM_LEN)
        {
            result = CMD_LEN_TOO_LONG;
        }
        if (result == 0)
        {
            char *cmd = text + 1;
            char *args;
            response[0] = '!';
            response_len = 1 + strcpy_prefix(response + 1, cmd);
            response[response_len++] = ' ';

            //deal with commands
            if (!strcmp(cmd, "N"))
            {
                print_neighbors();
            }
            else if (!strcmp_prefix(text + 1, "I", &args))
            {
                stats_info(args);
            }
            else if (!strcmp_prefix(text + 1, "S", &args))
            {
                send_interval_control(args);
            }
            else
            {
                response_len = sprintf(response, "!? %s", get_cmd_error_str(CMD_UNKNOWN));
            }
        }
        else
        {
            response_len = sprintf(response, "!? %s", get_cmd_error_str(result));
        }
        response[response_len++] = '\r';
        response[response_len++] = '\n';

        uart_send_raw(response, response_len);
#if USE_CHECK_SUM
        calc_chksum((uint8_t *)response, response_len);
        response_len += CHECKSUM_LEN;
#endif
        simple_udp_sendto(&udp_conn, response, response_len, sender_addr);

    }
    else if (waiting_response)
    {
        //if we have sent a message and waiting for response
        LOG_INFO("Received response '%.*s' from ", datalen, (char *) data);
        LOG_INFO_6ADDR(sender_addr);
        LOG_INFO_(" in %lu ms\n", clock_time() - sent_time);
        if (sent_len != datalen)
        {
            LOG_INFO("Response len error,received %d,but expected %d\n", datalen, sent_len);
        }
        else if (memcmp(send_buf, (char *)data, datalen))
        {
            LOG_INFO("Response data error, expected is %s\n", send_buf);
        }
        else
        {
            rx_count++;
            waiting_response = 0;
        }
        LOG_INFO_("\n");
    }
    else
    {
        LOG_INFO("Received '%.*s' from ", datalen, (char *) data);
        LOG_INFO_6ADDR(sender_addr);
        LOG_INFO_("while not waiting for response\n");
    }
}
/*---------------------------------------------------------------------------*/
PROCESS_THREAD(udp_client_process, ev, data)
{
    static struct etimer periodic_timer;
    static int unreachable_count = 0;
    PROCESS_BEGIN();
    RadioShieldLedInit(RADIO_SHIELD_LED);
    /* Initialize UDP connection */
    simple_udp_register(&udp_conn, UDP_CLIENT_PORT, NULL,
                        UDP_SERVER_PORT, udp_rx_callback);

    etimer_set(&periodic_timer, check_interval + (random_rand() % send_rand));
    RadioShieldLedOn(RADIO_SHIELD_LED);
    while (1)
    {
        PROCESS_WAIT_EVENT_UNTIL(etimer_expired(&periodic_timer));

        if (NETSTACK_ROUTING.node_is_reachable() &&
                NETSTACK_ROUTING.get_root_ipaddr(&dest_ipaddr))
        {
            es32_led_set(0, 0);
            es32_led_set(1, 1);
            if (waiting_response)
            {
                missed_tx_count++;
                waiting_response = 0;
            }
            if (send_enabled)
            {
                /* Print statistics every 10th TX */
                if (tx_count % 10 == 0)
                {
                    LOG_INFO("Tx/Rx/MissedTx: %" PRIu32 "/%" PRIu32 "/%" PRIu32 "\n",
                             tx_count, rx_count, missed_tx_count);
                }

                /* Send to DAG root */
                LOG_INFO("Sending request %"PRIu32" to ", tx_count);
                LOG_INFO_6ADDR(&dest_ipaddr);
                LOG_INFO_("\n");
                sent_len = snprintf(send_buf, 260, "hello %" PRIu32 "", tx_count);
                if (send_min_len > 0 && sent_len < send_min_len)
                {
                    while (sent_len < send_min_len)
                    {
                        send_buf[sent_len] = 'A' + (sent_len % 26);
                        sent_len++;
                    }
                }
                send_buf[sent_len] = 0;
                waiting_response = 1;
                sent_time = clock_time();
                simple_udp_sendto(&udp_conn, send_buf, sent_len, &dest_ipaddr);
                tx_count++;
            }
            unreachable_count = 0;

            etimer_set(&periodic_timer, send_interval - send_rand / 2 + (random_rand() % send_rand));
        }
        else
        {
            es32_led_set(0, 1);
            es32_led_set(1, 0);
            LOG_INFO("Not reachable yet,%d\n", unreachable_count);
            unreachable_count++;

            etimer_set(&periodic_timer, check_interval);
        }
        RadioShieldLedToggle(RADIO_SHIELD_LED);

    }

    PROCESS_END();
}

void uart_recv_callback()
{
    //process_poll(&udp_client_process);
}
/*---------------------------------------------------------------------------*/
