/**********************************************************************************
 *
 * @file    cmd_utils.c
 * @brief   Command utils
 *
 * @date    30 Apri 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          7  Nov  2022    shiwa           Command and checksum utils
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */
#include "cmd_utils.h"

static const char *CMD_ERR_STRS[] =
{
    "OK",
    "LEN_TOO_SHORT",
    "ERROR_FORMAT",
    "ERROR_CHECKSUM",
    "UNKNOWN_CMD",
    "CMD_LEN_TOO_LONG",
    "INVALID_PARAM"
};

int strcmp_prefix(char *s, const char *p, char **last)
{
    while (*p && *s)
    {
        if (*p != *s)
            break;
        ++p;
        ++s;
    }
    if (*p == 0)
    {
        if (*s == ' ' || *s == 0 || *s == '\t' || *s == '\r' || *s == '\n')
        {
            if (last)
                *last = s + 1;
            return 0;
        }
    }
    return *p - *s;
}

int strcpy_prefix(char *s, const char *p)
{
    int count = 0;
    while (*p && *p != ' ')
    {
        *s = *p;
        ++p;
        ++s;
        ++count;
    }
    return count;
}

bool check_sum(uint8_t *s, uint32_t len)
{
    uint8_t chk = 0;
    for (int i = 0; i < len - CHECKSUM_LEN; i++)
    {
        chk += s[i];
    }
    return chk == s[len - 1];
}
uint8_t calc_chksum(uint8_t *data, uint32_t len)
{
    uint8_t chk = 0;
    for (int i = 0; i < len; i++)
    {
        chk += data[i];
    }
    data[len] = (char)chk;
    return chk;
}

int parse_args(char *s, uint32_t len, char **argv, int *argc)
{
    int arg_num = 0;
    arg_num = 1;
    argv[0] = s + 1;
    for (int i = 0; i < len; i++)
    {
        if (s[i] == ' ')
        {
            s[i] = 0;
            argv[arg_num] = s + i + 1;
            arg_num++;
        }
    }
    if (argc)
        *argc = arg_num;
    return arg_num;
}

int process_cmd(char *s, uint32_t len, char type)
{
#if USE_CHECK_SUM
    // "?A\r\n"+CHK
    if (len < 4 + CHECKSUM_LEN)
        return CMD_LEN_TOO_SHORT;
#else
    if (len < 4)
        return CMD_LEN_TOO_SHORT;
#endif
    if (s[0] != type)
        return CMD_ERROR_FORMAT;

    if (s[len - CHECKSUM_LEN - 2] != '\r' || s[len - CHECKSUM_LEN - 1] != '\n')
        return CMD_ERROR_FORMAT;
#if USE_CHECK_SUM
    if (!check_sum((uint8_t *)s, len))
        return CMD_ERROR_CHECKSUM;
#endif
    s[len - CHECKSUM_LEN - 2] = 0;
    return 0;
}


const char *get_cmd_error_str(int errcode)
{
    if (errcode < 0 || errcode >= CMD_RESULTS_NUM)
    {
        return "UNKNOWN_ERROR";
    }
    return CMD_ERR_STRS[errcode];
}
