/**
******************************************************************************
* @file    radio_spi.c
* @author  Central Labs
* @version V1.0.0
* @date    18-Apr-2018
* @brief   This file provides code for the configuration of the SPI instances.
******************************************************************************
* @attention
*
* <h2><center>&copy; COPYRIGHT(c) 2018 STMicroelectronics</center></h2>
*
* Redistribution and use in source and binary forms, with or without modification,
* are permitted provided that the following conditions are met:
*   1. Redistributions of source code must retain the above copyright notice,
*      this list of conditions and the following disclaimer.
*   2. Redistributions in binary form must reproduce the above copyright notice,
*      this list of conditions and the following disclaimer in the documentation
*      and/or other materials provided with the distribution.
*   3. Neither the name of STMicroelectronics nor the names of its contributors
*      may be used to endorse or promote products derived from this software
*     without specific prior written permission.
*
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
* AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
* IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
* DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
* FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
* SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
* OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*
******************************************************************************
*/


/* Includes ------------------------------------------------------------------*/
#include "radio_spi.h"
/**
* @addtogroup BSP
* @{
*/


/**
* @addtogroup X-NUCLEO-IDS01Ax
* @{
*/


/**
* @defgroup RADIO_SPI_Private_TypesDefinitions       RADIO_SPI Private Types Definitions
* @{
*/

/**
* @}
*/


/**
* @defgroup RADIO_SPI_Private_Defines                RADIO_SPI Private Defines
* @{
*/

/**
* @}
*/


/**
* @defgroup RADIO_SPI_Private_Macros                 RADIO_SPI Private Macros
* @{
*/
/**
* @}
*/


/**
* @defgroup RADIO_SPI_Private_Variables              RADIO_SPI Private Variables
* @{
*/
spi_handle_t pSpiHandle;
uint32_t SpiTimeout = 1000;                         /*<! Value of Timeout when SPI communication fails */
/**
* @}
*/


/**
* @defgroup RADIO_SPI_Private_FunctionPrototypes     RADIO_SPI Private Function Prototypes
* @{
*/
void RadioSpiInit(void);
void HAL_SPI_MspInit(spi_handle_t *pSpiHandle);
/* void HAL_SPI_MspDeInit(SPI_HandleTypeDef* pSpiHandle); */
static void SPI_Write(uint8_t Value);
static void SPI_Error(void);
StatusBytes RadioSpiWriteRegisters(uint8_t cRegAddress, uint8_t cNbBytes, uint8_t *pcBuffer);
StatusBytes RadioSpiReadRegisters(uint8_t cRegAddress, uint8_t cNbBytes, uint8_t *pcBuffer);
StatusBytes RadioSpiCommandStrobes(uint8_t cCommandCode);
StatusBytes RadioSpiWriteFifo(uint8_t cNbBytes, uint8_t *pcBuffer);
StatusBytes RadioSpiReadFifo(uint8_t cNbBytes, uint8_t *pcBuffer);

/**
* @}
*/


/**
* @defgroup RADIO_SPI_Private_Functions              RADIO_SPI Private Functions
* @{
*/


/**
* @brief  Initializes SPI HAL.
* @param  None
* @retval None
*/
void RadioSpiInit(void)
{
    if (!pSpiHandle.perh||ald_spi_get_state(&pSpiHandle) == RESET) {
        /* SPI Config */
        pSpiHandle.perh = RADIO_SPI;
        pSpiHandle.init.mode      = SPI_MODE_MASTER;
        pSpiHandle.init.dir      = SPI_DIRECTION_2LINES;
        pSpiHandle.init.data_size = SPI_DATA_SIZE_8;
        pSpiHandle.init.baud      = SPI_BAUD_16;
        pSpiHandle.init.phase     = SPI_CPHA_FIRST;
        pSpiHandle.init.polarity  = SPI_CPOL_LOW;
        pSpiHandle.init.first_bit = SPI_FIRSTBIT_MSB;
        pSpiHandle.init.ss_en     = DISABLE;
        pSpiHandle.init.crc_calc  = DISABLE;
        pSpiHandle.tx_cplt_cbk    = NULL;
        pSpiHandle.rx_cplt_cbk    = NULL;
        pSpiHandle.err_cbk        = NULL;

        HAL_SPI_MspInit(&pSpiHandle);
        ald_spi_init(&pSpiHandle);
    }
}


/**
* @brief  Initializes SPI MSP.
* @param  SPI_HandleTypeDef* pSpiHandle
* @retval None
*/
void HAL_SPI_MspInit(spi_handle_t *pSpiHandle)
{

    gpio_init_t x;

    /* Initialize nss pin */
    x.mode  = GPIO_MODE_OUTPUT;
    x.odos  = GPIO_PUSH_PULL;
    x.pupd  = GPIO_PUSH_UP;
    x.podrv = GPIO_OUT_DRIVE_1;
    x.nodrv = GPIO_OUT_DRIVE_1;
    x.flt   = GPIO_FILTER_DISABLE;
    x.type  = GPIO_TYPE_TTL;
    x.func  = GPIO_FUNC_1;
    ald_gpio_init(RADIO_SPI_CS_PORT, RADIO_SPI_CS_PIN, &x);

    /* Initialize SCK pin */
    x.mode  = GPIO_MODE_OUTPUT;
    x.odos  = GPIO_PUSH_PULL;
    x.pupd  = GPIO_PUSH_DOWN;
    x.podrv = GPIO_OUT_DRIVE_6;
    x.nodrv = GPIO_OUT_DRIVE_6;
    x.flt   = GPIO_FILTER_DISABLE;
    x.type  = GPIO_TYPE_TTL;
    x.func  = RADIO_SPI_SCK_AF;
    ald_gpio_init(RADIO_SPI_SCK_PORT, RADIO_SPI_SCK_PIN, &x);

    /* Initialize MISO pin */
    x.mode  = GPIO_MODE_INPUT;
    x.odos  = GPIO_PUSH_PULL;
    x.pupd  = GPIO_PUSH_DOWN;
    x.podrv = GPIO_OUT_DRIVE_6;
    x.nodrv = GPIO_OUT_DRIVE_6;
    x.flt   = GPIO_FILTER_DISABLE;
    x.type  = GPIO_TYPE_TTL;
    x.func  = RADIO_SPI_MISO_AF;
    ald_gpio_init(RADIO_SPI_MISO_PORT, RADIO_SPI_MISO_PIN, &x);

    /* Initialize MOSI pin */
    x.mode  = GPIO_MODE_OUTPUT;
    x.odos  = GPIO_PUSH_PULL;
    x.pupd  = GPIO_PUSH_DOWN;
    x.podrv = GPIO_OUT_DRIVE_6;
    x.nodrv = GPIO_OUT_DRIVE_6;
    x.flt   = GPIO_FILTER_DISABLE;
    x.type  = GPIO_TYPE_TTL;
    x.func  = RADIO_SPI_MOSI_AF;
    ald_gpio_init(RADIO_SPI_MOSI_PORT, RADIO_SPI_MOSI_PIN, &x);
    
    ald_gpio_write_pin(RADIO_SPI_CS_PORT, RADIO_SPI_CS_PIN,SET);
}

/**
* @}
*/

/**
* @brief  SPI Write a byte to device
* @param  Value: value to be written
* @retval None
*/
static void SPI_Write(uint8_t Value)
{
    ald_status_t status = OK;

    status = ald_spi_send(&pSpiHandle, (uint8_t *) &Value, 1, SpiTimeout);

    /* Check the communication status */
    if (status != OK) {
        /* Execute user timeout callback */
        SPI_Error();
    }
}


/**
* @brief  SPI error treatment function
* @param  None
* @retval None
*/
static void SPI_Error(void)
{
    /* De-initialize the SPI communication BUS */
    ald_spi_reset(&pSpiHandle);

    /* Re-Initiaize the SPI communication BUS */
    RadioSpiInit();
}


/**
* @brief  Write single or multiple RF Transceivers register
* @param  cRegAddress: base register's address to be write
* @param  cNbBytes: number of registers and bytes to be write
* @param  pcBuffer: pointer to the buffer of values have to be written into registers
* @retval StatusBytes
*/
StatusBytes RadioSpiWriteRegisters(uint8_t cRegAddress, uint8_t cNbBytes, uint8_t *pcBuffer)
{
    uint8_t aHeader[2] = {0};
    uint16_t tmpstatus = 0x0000;

    StatusBytes *pStatus = (StatusBytes *)&tmpstatus;

    /* Built the aHeader bytes */
    aHeader[0] = WRITE_HEADER;
    aHeader[1] = cRegAddress;

    SPI_ENTER_CRITICAL();

    /* Puts the SPI chip select low to start the transaction */
    RadioSpiCSLow();

    for (volatile uint16_t Index = 0; Index < CS_TO_SCLK_DELAY; Index++);

    /* Write the aHeader bytes and read the SPIRIT1 status bytes */
    ald_spi_send_recv(&pSpiHandle, (uint8_t *)&aHeader[0], (uint8_t *) & (tmpstatus), 1, SpiTimeout);
    tmpstatus = tmpstatus << 8;

    /* Write the aHeader bytes and read the SPIRIT1 status bytes */
    ald_spi_send_recv(&pSpiHandle, (uint8_t *)&aHeader[1], (uint8_t *)&tmpstatus, 1, SpiTimeout);
    
    #if 1
    //ald_spi_send_bytes_fast(&pSpiHandle,pcBuffer,cNbBytes,SpiTimeout);
    ald_spi_send(&pSpiHandle,pcBuffer,cNbBytes,SpiTimeout);
    #else
    /* Writes the registers according to the number of bytes */
    for (int index = 0; index < cNbBytes; index++) {
        SPI_Write(pcBuffer[index]);
    }
    /* To be sure to don't rise the Chip Select before the end of last sending */
    while (ald_spi_get_flag_status(&pSpiHandle, SPI_IF_TXE) == RESET);

    #endif
    /* Puts the SPI chip select high to end the transaction */
    RadioSpiCSHigh();

    SPI_EXIT_CRITICAL();

    return *pStatus;

}


/**
* @brief  Read single or multiple SPIRIT1 register
* @param  cRegAddress: base register's address to be read
* @param  cNbBytes: number of registers and bytes to be read
* @param  pcBuffer: pointer to the buffer of registers' values read
* @retval StatusBytes
*/
StatusBytes RadioSpiReadRegisters(uint8_t cRegAddress, uint8_t cNbBytes, uint8_t *pcBuffer)
{
    uint16_t tmpstatus = 0x00;
    StatusBytes *pStatus = (StatusBytes *)&tmpstatus;

    uint8_t aHeader[2] = {0};
    uint8_t dummy = 0xFF;

    /* Built the aHeader bytes */
    aHeader[0] = READ_HEADER;
    aHeader[1] = cRegAddress;

    SPI_ENTER_CRITICAL();

    /* Put the SPI chip select low to start the transaction */
    RadioSpiCSLow();

    for (volatile uint16_t Index = 0; Index < CS_TO_SCLK_DELAY; Index++);

    /* Write the aHeader bytes and read the SPIRIT1 status bytes */
    ald_spi_send_recv(&pSpiHandle, (uint8_t *)&aHeader[0], (uint8_t *) & (tmpstatus), 1, SpiTimeout);
    tmpstatus = tmpstatus << 8;

    /* Write the aHeader bytes and read the SPIRIT1 status bytes */
    ald_spi_send_recv(&pSpiHandle, (uint8_t *)&aHeader[1], (uint8_t *)&tmpstatus, 1, SpiTimeout);
    
    #if 1
    //ald_spi_master_recv_bytes_fast(&pSpiHandle,pcBuffer,cNbBytes);
    ald_spi_recv(&pSpiHandle,pcBuffer,cNbBytes,SpiTimeout);
    #else
    for (int index = 0; index < cNbBytes; index++) {
        ald_spi_send_recv(&pSpiHandle, (uint8_t *)&dummy, (uint8_t *) & (pcBuffer)[index], 1, SpiTimeout);
    }

    /* To be sure to don't rise the Chip Select before the end of last sending */
    while (ald_spi_get_flag_status(&pSpiHandle, SPI_IF_TXE) == RESET);
    #endif
    /* Put the SPI chip select high to end the transaction */
    RadioSpiCSHigh();

    SPI_EXIT_CRITICAL();

    return *pStatus;

}


/**
* @brief  Send a command
* @param  cCommandCode: command code to be sent
* @retval StatusBytes
*/
StatusBytes RadioSpiCommandStrobes(uint8_t cCommandCode)
{
    uint8_t aHeader[2] = {0};
    uint16_t tmpstatus = 0x0000;

    StatusBytes *pStatus = (StatusBytes *)&tmpstatus;

    /* Built the aHeader bytes */
    aHeader[0] = COMMAND_HEADER;
    aHeader[1] = cCommandCode;

    SPI_ENTER_CRITICAL();

    /* Puts the SPI chip select low to start the transaction */
    RadioSpiCSLow();

    for (volatile uint16_t Index = 0; Index < CS_TO_SCLK_DELAY; Index++);
    /* Write the aHeader bytes and read the SPIRIT1 status bytes */
    ald_spi_send_recv(&pSpiHandle, (uint8_t *)&aHeader[0], (uint8_t *)&tmpstatus, 1, SpiTimeout);
    tmpstatus = tmpstatus << 8;

    /* Write the aHeader bytes and read the SPIRIT1 status bytes */
    ald_spi_send_recv(&pSpiHandle, (uint8_t *)&aHeader[1], (uint8_t *)&tmpstatus, 1, SpiTimeout);

    /* To be sure to don't rise the Chip Select before the end of last sending */
    while (ald_spi_get_flag_status(&pSpiHandle, SPI_IF_TXE) == RESET);

    /* Puts the SPI chip select high to end the transaction */
    RadioSpiCSHigh();

    SPI_EXIT_CRITICAL();

    return *pStatus;

}


/**
* @brief  Write data into TX FIFO
* @param  cNbBytes: number of bytes to be written into TX FIFO
* @param  pcBuffer: pointer to data to write
* @retval StatusBytes
*/
StatusBytes RadioSpiWriteFifo(uint8_t cNbBytes, uint8_t *pcBuffer)
{
    uint16_t tmpstatus = 0x0000;
    StatusBytes *pStatus = (StatusBytes *)&tmpstatus;

    uint8_t aHeader[2] = {0};

    /* Built the aHeader bytes */
    aHeader[0] = WRITE_HEADER;
    aHeader[1] = LINEAR_FIFO_ADDRESS;

    SPI_ENTER_CRITICAL();

    /* Put the SPI chip select low to start the transaction */
    RadioSpiCSLow();

    for (volatile uint16_t Index = 0; Index < CS_TO_SCLK_DELAY; Index++);

    /* Write the aHeader bytes and read the SPIRIT1 status bytes */
    ald_spi_send_recv(&pSpiHandle, (uint8_t *)&aHeader[0], (uint8_t *)&tmpstatus, 1, SpiTimeout);
    tmpstatus = tmpstatus << 8;

    /* Write the aHeader bytes and read the SPIRIT1 status bytes */
    ald_spi_send_recv(&pSpiHandle, (uint8_t *)&aHeader[1], (uint8_t *)&tmpstatus, 1, SpiTimeout);
    
    #if 1
    //ald_spi_send_bytes_fast(&pSpiHandle,pcBuffer,cNbBytes,SpiTimeout);
    ald_spi_send(&pSpiHandle,pcBuffer,cNbBytes,SpiTimeout);
    #else
    /* Writes the registers according to the number of bytes */
    for (int index = 0; index < cNbBytes; index++) {
        SPI_Write(pcBuffer[index]);
    }
    /* To be sure to don't rise the Chip Select before the end of last sending */
    while (ald_spi_get_flag_status(&pSpiHandle, SPI_IF_TXE) == RESET);

    #endif
    /* Put the SPI chip select high to end the transaction */
    RadioSpiCSHigh();

    SPI_EXIT_CRITICAL();

    return *pStatus;
}

/**
* @brief  Read data from RX FIFO
* @param  cNbBytes: number of bytes to read from RX FIFO
* @param  pcBuffer: pointer to data read from RX FIFO
* @retval StatusBytes
*/
StatusBytes RadioSpiReadFifo(uint8_t cNbBytes, uint8_t *pcBuffer)
{
    uint16_t tmpstatus = 0x0000;
    StatusBytes *pStatus = (StatusBytes *)&tmpstatus;

    uint8_t aHeader[2];
    uint8_t dummy = 0xFF;

    /* Built the aHeader bytes */
    aHeader[0] = READ_HEADER;
    aHeader[1] = LINEAR_FIFO_ADDRESS;

    SPI_ENTER_CRITICAL();

    /* Put the SPI chip select low to start the transaction */
    RadioSpiCSLow();

    for (volatile uint16_t Index = 0; Index < CS_TO_SCLK_DELAY; Index++);

    /* Write the aHeader bytes and read the SPIRIT1 status bytes */
    ald_spi_send_recv(&pSpiHandle, (uint8_t *)&aHeader[0], (uint8_t *)&tmpstatus, 1, SpiTimeout);
    tmpstatus = tmpstatus << 8;

    /* Write the aHeader bytes and read the SPIRIT1 status bytes */
    ald_spi_send_recv(&pSpiHandle, (uint8_t *)&aHeader[1], (uint8_t *)&tmpstatus, 1, SpiTimeout);
    
    
    #if 1
    //ald_spi_master_recv_bytes_fast(&pSpiHandle,pcBuffer,cNbBytes);
    ald_spi_recv(&pSpiHandle,pcBuffer,cNbBytes,SpiTimeout);
    #else
    for (int index = 0; index < cNbBytes; index++) {
        ald_spi_send_recv(&pSpiHandle, (uint8_t *)&dummy, (uint8_t *)&pcBuffer[index], 1, SpiTimeout);
    }

    /* To be sure to don't rise the Chip Select before the end of last sending */
    while (ald_spi_get_flag_status(&pSpiHandle, SPI_IF_TXE) == RESET);
    #endif
    /* Put the SPI chip select high to end the transaction */
    RadioSpiCSHigh();

    SPI_EXIT_CRITICAL();

    return *pStatus;
}


/**
* @}
*/

/************************ (C) COPYRIGHT STMicroelectronics *****END OF FILE****/
