/**
  *********************************************************************************
  *
  * @file    drv_uart.c
  * @brief   UART driver for rt-thread-nano console
  *
  * @version V1.0
  * @date    26 Apr 2021
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          02 Sep 2022     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  *********************************************************************************
  */
#include "drv_uart.h"
#include <string.h>
#include <rtthread.h>

#ifdef CONSOLE_USE_UART
uart_handle_t h_uart;
#endif
#ifdef CONSOLE_USE_LPUART
lpuart_handle_t h_lpuart;
#endif
#define UART_SEND_TIMEOUT 100
/**
  * @brief  Initialize UART pin
  * @param  None.
  * @retval None.
  */
void init_uart_pin()
{
    gpio_init_t x;

    /* Initialize tx pin */
    x.mode = GPIO_MODE_OUTPUT;
    x.odos = GPIO_PUSH_PULL;
    x.pupd = GPIO_PUSH_UP;
    x.podrv = GPIO_OUT_DRIVE_1;
    x.nodrv = GPIO_OUT_DRIVE_0_1;
    x.flt  = GPIO_FILTER_DISABLE;
    x.type = GPIO_TYPE_CMOS;
    x.func = CONSOLE_UART_TX_FUNC;
    ald_gpio_init(CONSOLE_UART_TX_PORT, CONSOLE_UART_TX_PIN, &x);

    /* Initialize rx pin */
    x.mode = GPIO_MODE_INPUT;
    x.odos = GPIO_PUSH_PULL;
    x.pupd = GPIO_PUSH_UP;
    x.podrv = GPIO_OUT_DRIVE_1;
    x.nodrv = GPIO_OUT_DRIVE_0_1;
    x.flt  = GPIO_FILTER_DISABLE;
    x.type = GPIO_TYPE_CMOS;
    x.func = CONSOLE_UART_RX_FUNC;
    ald_gpio_init(CONSOLE_UART_RX_PORT, CONSOLE_UART_RX_PIN, &x);
}

/**
  * @brief  Initialize UART
  * @param  None.
  * @retval None.
  */
void init_uart()
{
    #ifdef CONSOLE_USE_UART
    /* clear uart_handle_t structure */
    memset(&h_uart, 0x0, sizeof(h_uart));
    /* Initialize uart */
    h_uart.perh             = CONSOLE_UART;
    h_uart.init.baud        = 115200;
    h_uart.init.word_length = UART_WORD_LENGTH_8B;
    h_uart.init.stop_bits   = UART_STOP_BITS_1;
    h_uart.init.parity      = UART_PARITY_NONE;
    h_uart.init.mode        = UART_MODE_UART;
    h_uart.init.fctl        = UART_HW_FLOW_CTL_DISABLE;
    h_uart.tx_cplt_cbk      = NULL;
    h_uart.rx_cplt_cbk      = NULL;
    h_uart.error_cbk        = NULL;
    ald_uart_init(&h_uart);
    #endif
    #ifdef CONSOLE_USE_LPUART
    /* clear uart_handle_t structure */
    memset(&h_lpuart, 0x0, sizeof(h_lpuart));
    /* Initialize uart */
    h_lpuart.perh             = CONSOLE_UART;
    h_lpuart.init.baud        = 9600;
    h_lpuart.init.word_length = LPUART_WORD_LENGTH_8B;
    h_lpuart.init.stop_bits   = LPUART_STOP_BITS_1;
    h_lpuart.init.parity      = LPUART_PARITY_NONE;
    h_lpuart.init.mode        = LPUART_MODE_UART;
    h_lpuart.init.fctl        = LPUART_HW_FLOW_CTL_NONE;
    h_lpuart.tx_cplt_cbk      = NULL;
    h_lpuart.rx_cplt_cbk      = NULL;
    h_lpuart.error_cbk        = NULL;
    ald_lpuart_init(&h_lpuart);
    #endif
}

#ifdef CONSOLE_USE_UART
/**
  * @brief  send string by uart
  * @param  s : string to output
  * @retval None.
  */
void uart_send_str(const char*s)
{ 
    uint16_t len = 0;

    len = rt_strlen(s);
    
    ald_uart_send(&h_uart,(uint8_t*)s,len,1000);
}

/**
  * @brief  UART send one character
  * @param  c: character want to send
  * @retval None.
  */
void uart_send_ch(char c)
{
    uint32_t t=ald_get_tick();
    while (!ald_uart_get_status(&h_uart,UART_STATUS_TFTH)&&ald_get_tick()-t<UART_SEND_TIMEOUT);
    h_uart.perh->TXBUF=(uint8_t)c;
}


/**
  * @brief  UART get one character
  * @param  c: character received (-1 if nothing received)
  * @retval None.
  */
int uart_get_ch()
{    
    int ch = -1;

    if (ald_uart_get_status(&h_uart, UART_STATUS_RFTH))
    {
        ch = (int8_t)(h_uart.perh->RXBUF & 0xFF);
    }
    return ch;
}
#endif
#ifdef CONSOLE_USE_LPUART
/**
  * @brief  send string by uart
  * @param  s : string to output
  * @retval None.
  */
void uart_send_str(const char*s)
{ 
    uint16_t len = 0;

    len = rt_strlen(s);
    
    ald_lpuart_send(&h_lpuart,(uint8_t*)s,len,1000);
}

/**
  * @brief  UART send one character
  * @param  c: character want to send
  * @retval None.
  */
void uart_send_ch(char c)
{
    uint32_t t=ald_get_tick();
    while (!ald_lpuart_get_status(&h_lpuart,LPUART_STAT_TXEMP)&&ald_get_tick()-t<UART_SEND_TIMEOUT);
    h_lpuart.perh->TXDR=(uint8_t)c;
    while (ald_lpuart_get_status(&h_lpuart,LPUART_STAT_TXEMP)&&ald_get_tick()-t<UART_SEND_TIMEOUT);
}


/**
  * @brief  UART get one character
  * @param  c: character received (-1 if nothing received)
  * @retval None.
  */
int uart_get_ch()
{    
    int ch = -1;

    if (!ald_lpuart_get_status(&h_lpuart, LPUART_STAT_RXEMP))
    {
        ch = (int8_t)(h_lpuart.perh->RXDR & 0xFF);
    }
    return ch;
}
#endif
/**
  * @brief  rt-thread read one char
  * @param  None.
  * @retval char read or -1 if no char
  */
char rt_hw_console_getchar(void)
{
    int ch = uart_get_ch();
    if (ch==-1)
        rt_thread_mdelay(1);
    return ch;
}

/**
  * @brief  rt-thread console output to uart
  * @param  str : string to output
  * @retval None.
  */
void rt_hw_console_output(const char *str)
{
    rt_size_t len = rt_strlen(str);
    rt_size_t i;
    
    for (i=0;i<len;i++)
    {
        char c=str[i];
        if (c=='\n')
            uart_send_ch('\r');
        uart_send_ch(c);
    }
}

#ifdef RT_USING_CONSOLE
/**
  * @brief  Initialize UART
  * @param  None.
  * @retval 0
  */
int init_console_uart()
{
    init_uart_pin();
    init_uart();
    return 0;
}
INIT_BOARD_EXPORT(init_console_uart); 
#endif
