/**********************************************************************************
 *
 * @file    main.c
 * @brief   ADC_SingleConversion C file
 *
 * @date    30 Apri 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          2021/09/03      liuhy           the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */
#include <string.h>
#include "main.h"

/* Private Macros ------------------------------------------------------------ */
/* Private Constants --------------------------------------------------------- */
/* Private function prototypes ----------------------------------------------- */

static UINT32 AppTaskCreate(void);
static UINT32 Creat_Test_Task(void);
static void Test_Task(void);
static void BSP_Init(void);

/* Private Variables --------------------------------------------------------- */
/* Public Variables ---------------------------------------------------------- */

UINT32 g_Test_Task_Handle;

/* Private Function ---------------------------------------------------------- */

/***************************************************************
  * @brief  主函数
  * @param  无
  * @retval 无
  * @note   第一步：开发板硬件初始化
            第二步：创建APP应用任务
            第三步：启动LiteOS，开始多任务调度，启动失败则输出错误信息
  **************************************************************/
int main(void)
{
    UINT32 uwRet = LOS_OK;  /* 定义一个任务创建的返回值，默认为创建成功 */

    /* 板载相关初始化 */
    BSP_Init();

    printf("example start\r\n");

    /* LiteOS 内核初始化 */
    uwRet = LOS_KernelInit();

    if (uwRet != LOS_OK)
    {
        printf("[err]LiteOS core init err! code = 0x%X\r\n", uwRet);
        return LOS_NOK;
    }

    uwRet = AppTaskCreate();

    if (uwRet != LOS_OK)
    {
        printf("[err]AppTaskCreate err! cde = 0x%X\r\n", uwRet);
        return LOS_NOK;
    }

    /* 开启LiteOS任务调度 */
    LOS_Start();

    /* 正常情况下不会执行到这里 */
    while (1);

}

/*********************************************************************
  * @ 函数名  ： AppTaskCreate
  * @ 功能说明： 任务创建，为了方便管理，所有的任务创建函数都可以放在这个函数里面
  * @ 参数    ： 无
  * @ 返回值  ： 无
  *******************************************************************/
static UINT32 AppTaskCreate(void)
{
    /* 定义一个返回类型变量，初始化为LOS_OK */
    UINT32 uwRet = LOS_OK;
    uwRet = Creat_Test_Task();

    if (uwRet != LOS_OK)
    {
        printf("Test_Task task creation failed! Failure code = 0x%X\r\n", uwRet);
        return uwRet;
    }

    return LOS_OK;
}

/******************************************************************
  * @ 函数名  ： Creat_LED_Task
  * @ 功能说明： 创建LED_Task任务
  * @ 参数    ：
  * @ 返回值  ： 无
  ******************************************************************/
static UINT32 Creat_Test_Task()
{
    UINT32 uwRet = LOS_OK;              /* 定义一个创建任务的返回类型，初始化为创建成功的返回值 */
    TSK_INIT_PARAM_S task_init_param;

    task_init_param.usTaskPrio = 4;             /* 优先级，数值越小，优先级越高 */
    task_init_param.pcName = "Test_Task";                       /* 任务名 */
    task_init_param.pfnTaskEntry = (TSK_ENTRY_FUNC)Test_Task;   /* 任务函数名 */
    task_init_param.uwStackSize = 1024;                     /* 堆栈大小 */

    uwRet = LOS_TaskCreate(&g_Test_Task_Handle, &task_init_param);/* 创建任务 */
    return uwRet;
}

/*********************************************************************************
  * @ 函数名  ： Clear_Task
  * @ 功能说明： 写入已经初始化成功的内存池地址数据
  * @ 参数    ： void
  * @ 返回值  ： 无
  ********************************************************************************/
static void Test_Task(void)
{
    UINT32 uwRet = LOS_OK;          /* 定义一个初始化的返回类型，初始化为成功的返回值 */
    printf("\r\nlist init ......\r\n");

    LOS_DL_LIST *head;              /* 定义一个双向链表的头节点 */
    head = (LOS_DL_LIST *)LOS_MemAlloc(m_aucSysMem0, sizeof(LOS_DL_LIST));  /* 动态申请头结点的内存 */

    LOS_ListInit(head);             /* 初始化双向链表 */

    if (!LOS_ListEmpty(head))       /* 判断是否初始化成功 */
    {
        printf("list init fail\r\n");
    }
    else
    {
        printf("list init succes\r\n");
    }

    printf("\r\nAdd node and tail node......\r\n");         /* 插入节点：顺序插入与从末尾插入 */

    LOS_DL_LIST *node1 = (LOS_DL_LIST *)LOS_MemAlloc(m_aucSysMem0, sizeof(LOS_DL_LIST));/* 动态申请第一个结点的内存 */
    LOS_DL_LIST *node2 = (LOS_DL_LIST *)LOS_MemAlloc(m_aucSysMem0, sizeof(LOS_DL_LIST));/* 动态申请第二个结点的内存 */
    LOS_DL_LIST *tail = (LOS_DL_LIST *)LOS_MemAlloc(m_aucSysMem0, sizeof(LOS_DL_LIST)); /* 动态申请尾结点的内存 */

    printf("Add the first node and the second node.....\r\n");
    LOS_ListAdd(head, node1);                   /* 添加第一个节点，连接在头结点上 */
    LOS_ListAdd(node1, node2);                  /* 添加第二个节点，连接在一个节点上 */

    if ((node1->pstPrev == head) || (node2->pstPrev == node1))  /* 判断是否插入成功 */
    {
        printf("add node succes\r\n");
    }
    else
    {
        printf("Failed to add node!\r\n");
    }

    printf("Insert the tail node into the end of the bidirectional linked list.....\r\n");
    LOS_ListTailInsert(head, tail);             /* 将尾节点插入双向链表的末尾 */

    if (tail->pstPrev == node2)                 /* 判断是否插入成功 */
    {
        printf("The end node of the linked list was added successfully!\r\n");
    }
    else
    {
        printf("Failed to add end of list node!\r\n");
    }

    printf("\r\nDelete node......\r\n");                /* 删除已有节点 */
    LOS_ListDelete(node1);                  /* 删除第一个节点 */
    LOS_MemFree(m_aucSysMem0, node1);       /* 释放第一个节点的内存，保证后面的程序回收内存空间 */

    if (head->pstNext == node2)             /* 判断是否删除成功 */
    {
        printf("Delete node succes\r\n");
    }
    else
    {
        printf("Delete node fail\r\n");

    }

    while (1)
    {
        DEBUG_LED1_TOGGLE
        DEBUG_LED2_TOGGLE
        printf("Task running!\r\n");
        LOS_TaskDelay(2000);
    }
}


/*******************************************************************
  * @ 函数名  ： BSP_Init
  * @ 功能说明： 板级外设初始化，所有板子上的初始化均可放在这个函数里面
  * @ 参数    ：
  * @ 返回值  ： 无
  ******************************************************************/
static void BSP_Init(void)
{
    md_cmu_pll1_config(MD_CMU_PLL1_INPUT_HOSC_3, MD_CMU_PLL1_OUTPUT_72M);
    md_cmu_clock_config(MD_CMU_CLOCK_PLL1, 72000000);

    /* Enable ALL peripheral */
    SYSCFG_UNLOCK();
    md_cmu_enable_perh_all();
    SYSCFG_LOCK();

    debug_gpio_init();
    debug_uart_init();

}

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
