/**********************************************************************************
 *
 * @file    main.c
 * @brief   ADC_SingleConversion C file
 *
 * @date    30 Apri 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          2021/09/03      liuhy           the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */
#include <string.h>
#include "main.h"
/* 板级外设头文件 */
#include "debug_uart.h"
#include "debug_gpio.h"
#include "debug_it.h"

/* Private Macros ------------------------------------------------------------ */
/* Private Constants --------------------------------------------------------- */

/* 定义消息队列长度 */
#define  TEST_QUEUE_LEN     10
#define  TEST_QUEUE_SIZE  10

/* Private function prototypes ----------------------------------------------- */

static UINT32 AppTaskCreate(void);
static UINT32 Creat_key_task(void);
static UINT32 Creat_uart_task(void);

static void key_task(void);
static void uart_task(void);
static void BSP_Init(void);

/* Private Variables --------------------------------------------------------- */
/* Public Variables ---------------------------------------------------------- */

/* 定义任务句柄 */
UINT32 g_key_task_Handle;
UINT32 g_uart_task_Handle;

/* 定义消息队列的句柄 */
UINT32 g_Test_Queue_Handle;
/* 定义二值信号量的句柄 */
UINT32 g_BinarySem_Handle;

uint8_t g_Uart_Rx_Buf[UART_RBUF_SIZE];

/* Private Function ---------------------------------------------------------- */

/***************************************************************
  * @brief  主函数
  * @param  无
  * @retval 无
  * @note   第一步：开发板硬件初始化
            第二步：创建APP应用任务
            第三步：启动LiteOS，开始多任务调度，启动失败则输出错误信息
  **************************************************************/
int main(void)
{
    UINT32 uwRet = LOS_OK;  /* 定义一个任务创建的返回值，默认为创建成功 */

    /* 板载相关初始化 */
    BSP_Init();

    printf("example start\r\n");

    /* LiteOS 内核初始化 */
    uwRet = LOS_KernelInit();

    if (uwRet != LOS_OK)
    {
        printf("[err]LiteOS core init err! code = 0x%X\r\n", uwRet);
        return LOS_NOK;
    }

    uwRet = AppTaskCreate();

    if (uwRet != LOS_OK)
    {
        printf("[err]AppTaskCreate err! cde = 0x%X\r\n", uwRet);
        return LOS_NOK;
    }

    /* 开启LiteOS任务调度 */
    LOS_Start();

    /* 正常情况下不会执行到这里 */
    while (1);

}


/*********************************************************************
  * @ 函数名  ： AppTaskCreate
  * @ 功能说明： 任务创建，为了方便管理，所有的任务创建函数都可以放在这个函数里面
  * @ 参数    ： 无
  * @ 返回值  ： 无
  *******************************************************************/
static UINT32 AppTaskCreate(void)
{
    /* 定义一个返回类型变量，初始化为LOS_OK */
    UINT32 uwRet = LOS_OK;

    /* 创建一个测试队列*/
    uwRet = LOS_QueueCreate("Test_Queue",               /* 队列的名称 */
                            TEST_QUEUE_LEN,         /* 队列的长度 */
                            &g_Test_Queue_Handle,   /* 队列的ID(句柄) */
                            0,                      /* 队列模式，官方暂时未使用 */
                            TEST_QUEUE_SIZE);   /* 节点大小，单位为字节 */

    if (uwRet != LOS_OK)
    {
        printf("Test_Queue creation failed! Failure code = 0x%X\r\n", uwRet);
        return uwRet;
    }

    /* 创建一个二值信号量*/
    uwRet = LOS_BinarySemCreate(0, &g_BinarySem_Handle);

    if (uwRet != LOS_OK)
    {
        printf("BinarySem creation failed! Failure code = 0x%X\r\n", uwRet);
    }

    uwRet = Creat_key_task();

    if (uwRet != LOS_OK)
    {
        printf("key_task task creation failed! Failure code = 0x%X\r\n", uwRet);
        return uwRet;
    }

    uwRet = Creat_uart_task();

    if (uwRet != LOS_OK)
    {
        printf("uart_task task creation failed! Failure code = 0x%X\r\n", uwRet);
        return uwRet;
    }

    return LOS_OK;
}


/******************************************************************
  * @ 函数名  ： Creat_key_task
  * @ 功能说明： 创建key_task任务
  * @ 参数    ：
  * @ 返回值  ： 无
  ******************************************************************/
static UINT32 Creat_key_task()
{
    /* 定义一个返回类型变量，初始化为LOS_OK */
    UINT32 uwRet = LOS_OK;

    /* 定义一个用于创建任务的参数结构体 */
    TSK_INIT_PARAM_S task_init_param;

    task_init_param.usTaskPrio = 5; /* 任务优先级，数值越小，优先级越高 */
    task_init_param.pcName = "key_task";/* 任务名 */
    task_init_param.pfnTaskEntry = (TSK_ENTRY_FUNC)key_task;/* 任务函数入口 */
    task_init_param.uwStackSize = 1024;     /* 堆栈大小 */

    uwRet = LOS_TaskCreate(&g_key_task_Handle, &task_init_param);/* 创建任务 */
    return uwRet;
}
/*******************************************************************
  * @ 函数名  ： Creat_uart_task
  * @ 功能说明： 创建uart_task任务
  * @ 参数    ：
  * @ 返回值  ： 无
  ******************************************************************/
static UINT32 Creat_uart_task()
{
    /* 定义一个返回类型变量，初始化为LOS_OK */
    UINT32 uwRet = LOS_OK;
    TSK_INIT_PARAM_S task_init_param;

    task_init_param.usTaskPrio = 4; /* 任务优先级，数值越小，优先级越高 */
    task_init_param.pcName = "uart_task";    /* 任务名*/
    task_init_param.pfnTaskEntry = (TSK_ENTRY_FUNC)uart_task;/* 任务函数入口 */
    task_init_param.uwStackSize = 1024; /* 堆栈大小 */

    uwRet = LOS_TaskCreate(&g_uart_task_Handle, &task_init_param);/* 创建任务 */

    return uwRet;
}

/******************************************************************
  * @ 函数名  ： key_task
  * @ 功能说明： key_task任务实现
  * @ 参数    ： NULL
  * @ 返回值  ： NULL
  *****************************************************************/
static void key_task(void)
{
    /* 定义一个返回类型变量，初始化为LOS_OK */
    UINT32 uwRet = LOS_OK;
    UINT32 *r_queue;

    /* 任务都是一个无限循环，不能返回 */
    while (1)
    {
        /* 队列读取（接收），等待时间为一直等待 */
        uwRet = LOS_QueueRead(g_Test_Queue_Handle,  /* 读取（接收）队列的ID(句柄) */
                              &r_queue,           /* 读取（接收）的数据保存位置 */
                              TEST_QUEUE_SIZE,        /* 读取（接收）的数据的长度 */
                              LOS_WAIT_FOREVER);  /* 等待时间：一直等 */

        if (LOS_OK == uwRet)
        {
            printf("the interrupt triggered = KEY%d \r\n", *r_queue);
            DEBUG_LED1_TOGGLE
        }
        else
        {
            printf("Data receiving error, error code = 0x%X \r\n", uwRet);
        }
    }
}
/******************************************************************
  * @ 函数名  ： uart_task
  * @ 功能说明： uart_task任务实现
  * @ 参数    ： NULL
  * @ 返回值  ： NULL
  *****************************************************************/
static void uart_task(void)
{
    /* 定义一个返回类型变量，初始化为LOS_OK */
    UINT32 uwRet = LOS_OK;

    /* 任务都是一个无限循环，不能返回 */
    while (1)
    {
        /* 获取二值信号量 xSemaphore,没获取到则一直等待 */
        uwRet = LOS_SemPend(g_BinarySem_Handle, LOS_WAIT_FOREVER);

        if (LOS_OK == uwRet)
        {
            printf("Data received = %s\r\n", g_Uart_Rx_Buf);
            memset(g_Uart_Rx_Buf, 0, UART_RBUF_SIZE); /* 清零 */
            DEBUG_LED2_TOGGLE
        }

    }
}

/*******************************************************************
  * @ 函数名  ： BSP_Init
  * @ 功能说明： 板级外设初始化，所有板子上的初始化均可放在这个函数里面
  * @ 参数    ：
  * @ 返回值  ： 无
  ******************************************************************/
static void BSP_Init(void)
{
    md_cmu_pll1_config(MD_CMU_PLL1_INPUT_HOSC_3, MD_CMU_PLL1_OUTPUT_72M);
    md_cmu_clock_config(MD_CMU_CLOCK_PLL1, 72000000);

    /* Enable ALL peripheral */
    SYSCFG_UNLOCK();
    md_cmu_enable_perh_all();
    SYSCFG_LOCK();

    debug_gpio_init();
    debug_uart_init();
    debug_it_init();

}

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
